import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { useToast } from "@/hooks/use-toast";
import { Plus, Trash2, Key, Package, Loader2, CheckCircle2, X, User, Phone, Calendar, Hash, Clock } from "lucide-react";

interface OrderInfo {
  id: string;
  customer_name: string;
  customer_phone: string | null;
  amount: string;
  created_at: string;
  payment_status: string | null;
}

interface ProductKey {
  id: string;
  product_id: string;
  key_value: string;
  is_sold: boolean;
  order_id: string | null;
  buyer_name?: string | null;
  buyer_phone?: string | null;
  sold_at: string | null;
  buyer_email?: string | null;
  created_at: string;
  duration_option: string | null;
  orderInfo?: OrderInfo | null;
}

const mapKeyData = (k: any, emailMap: Record<string, string>): ProductKey => ({
  ...k,
  buyer_name: k.orders?.customer_name || null,
  buyer_phone: k.orders?.customer_phone || null,
  buyer_email: k.orders?.user_id ? (emailMap[k.orders.user_id] || null) : null,
  orderInfo: k.orders ? {
    id: k.orders.id,
    customer_name: k.orders.customer_name,
    customer_phone: k.orders.customer_phone,
    amount: k.orders.amount,
    created_at: k.orders.created_at,
    payment_status: k.orders.payment_status,
  } : null,
  orders: undefined,
});

interface Product {
  id: string;
  name: string;
  duration_options: string[] | null;
}

const AdminProductKeys = () => {
  const [products, setProducts] = useState<Product[]>([]);
  const [selectedProduct, setSelectedProduct] = useState<string>("");
  const [selectedDuration, setSelectedDuration] = useState<string>("");
  const [keys, setKeys] = useState<ProductKey[]>([]);
  const [loading, setLoading] = useState(false);
  const [newKeys, setNewKeys] = useState("");
  const [newKeyDuration, setNewKeyDuration] = useState<string>("");
  const [adding, setAdding] = useState(false);
  const [expandedKey, setExpandedKey] = useState<string | null>(null);
  const { toast } = useToast();

  const currentProduct = products.find(p => p.id === selectedProduct);
  const hasDurations = currentProduct?.duration_options && currentProduct.duration_options.length > 0;

  useEffect(() => {
    const fetchProducts = async () => {
      const { data } = await supabase.from("products").select("id, name, duration_options").order("name");
      if (data) setProducts(data as Product[]);
    };
    fetchProducts();
  }, []);

  useEffect(() => {
    if (!selectedProduct) { setKeys([]); return; }
    setSelectedDuration("");
    const fetchKeys = async () => {
      setLoading(true);
      const { data } = await supabase
        .from("product_keys")
        .select("*, orders:order_id(id, customer_name, customer_phone, amount, created_at, payment_status, user_id)")
        .eq("product_id", selectedProduct)
        .order("created_at", { ascending: false });
      if (data) {
        // Fetch emails for user_ids
        const userIds = [...new Set(data.filter((k: any) => k.orders?.user_id).map((k: any) => k.orders.user_id))];
        let emailMap: Record<string, string> = {};
        if (userIds.length > 0) {
          const { data: profiles } = await supabase.from("profiles").select("user_id, email").in("user_id", userIds);
          if (profiles) profiles.forEach((p: any) => { if (p.email) emailMap[p.user_id] = p.email; });
        }
        setKeys(data.map((k: any) => mapKeyData(k, emailMap)) as ProductKey[]);
      }
      setLoading(false);
    };
    fetchKeys();
  }, [selectedProduct]);

  const filteredKeys = selectedDuration
    ? keys.filter(k => k.duration_option === selectedDuration)
    : keys;

  const handleAddKeys = async () => {
    if (!selectedProduct || !newKeys.trim()) return;
    if (hasDurations && !newKeyDuration) {
      toast({ title: "ডিউরেশন সিলেক্ট করুন", variant: "destructive" });
      return;
    }
    setAdding(true);
    const { error } = await supabase.from("product_keys").insert({
      product_id: selectedProduct,
      key_value: newKeys.trim(),
      duration_option: hasDurations ? newKeyDuration : null,
    });
    if (error) {
      toast({ title: "এরর", description: error.message, variant: "destructive" });
    } else {
      toast({ title: "✅ কী যোগ হয়েছে!" });
      setNewKeys("");
      const { data } = await supabase.from("product_keys").select("*, orders:order_id(id, customer_name, customer_phone, amount, created_at, payment_status, user_id)").eq("product_id", selectedProduct).order("created_at", { ascending: false });
      if (data) setKeys(data.map((k: any) => mapKeyData(k, {})) as ProductKey[]);
    }
    setAdding(false);
  };

  const handleDeleteKey = async (id: string) => {
    const { error } = await supabase.from("product_keys").delete().eq("id", id);
    if (error) {
      toast({ title: "এরর", description: error.message, variant: "destructive" });
    } else {
      setKeys(prev => prev.filter(k => k.id !== id));
      toast({ title: "🗑️ কী ডিলিট হয়েছে!" });
    }
  };

  const availableCount = filteredKeys.filter(k => !k.is_sold).length;
  const soldCount = filteredKeys.filter(k => k.is_sold).length;

  // Stock counts per product+duration
  const [stockMap, setStockMap] = useState<Record<string, number>>({});
  useEffect(() => {
    const fetchStocks = async () => {
      const { data } = await supabase.from("product_keys").select("product_id, is_sold");
      if (data) {
        const map: Record<string, number> = {};
        data.forEach((k: any) => {
          if (!k.is_sold) map[k.product_id] = (map[k.product_id] || 0) + 1;
        });
        setStockMap(map);
      }
    };
    fetchStocks();
  }, [keys]);

  return (
    <div className="space-y-6 max-w-4xl">
      {/* Product Selector */}
      <div className="bg-card rounded-2xl border border-border p-6">
        <div className="flex items-center gap-3 mb-4">
          <div className="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center">
            <Key className="h-4 w-4 text-primary" />
          </div>
          <div>
            <h3 className="font-heading font-semibold text-sm">প্রোডাক্ট কী ম্যানেজমেন্ট</h3>
            <p className="text-xs text-muted-foreground">প্রোডাক্ট সিলেক্ট করে কী/ক্রেডেনশিয়াল যোগ করুন</p>
          </div>
        </div>

        <select
          value={selectedProduct}
          onChange={e => setSelectedProduct(e.target.value)}
          className="flex h-10 w-full rounded-xl border border-input bg-background px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-ring"
        >
          <option value="">প্রোডাক্ট সিলেক্ট করুন</option>
          {products.map(p => (
            <option key={p.id} value={p.id}>
              {p.name} {stockMap[p.id] ? `(${stockMap[p.id]} স্টক)` : "(0 স্টক)"}
            </option>
          ))}
        </select>

        {/* Duration filter */}
        {selectedProduct && hasDurations && (
          <div className="mt-3">
            <label className="text-xs text-muted-foreground mb-1 block">ডিউরেশন ফিল্টার</label>
            <div className="flex flex-wrap gap-2">
              <button
                onClick={() => setSelectedDuration("")}
                className={`px-3 py-1.5 text-xs font-medium border rounded-lg transition-all ${
                  !selectedDuration ? "border-primary bg-primary text-primary-foreground" : "border-border bg-card text-foreground hover:border-primary"
                }`}
              >
                সব ({keys.length})
              </button>
              {currentProduct!.duration_options!.map(d => {
                const count = keys.filter(k => k.duration_option === d).length;
                const available = keys.filter(k => k.duration_option === d && !k.is_sold).length;
                return (
                  <button
                    key={d}
                    onClick={() => setSelectedDuration(d)}
                    className={`px-3 py-1.5 text-xs font-medium border rounded-lg transition-all ${
                      selectedDuration === d ? "border-primary bg-primary text-primary-foreground" : "border-border bg-card text-foreground hover:border-primary"
                    }`}
                  >
                    {d} ({available}/{count})
                  </button>
                );
              })}
            </div>
          </div>
        )}
      </div>

      {selectedProduct && (
        <>
          {/* Stats */}
          <div className="grid grid-cols-2 gap-4">
            <div className="bg-card rounded-2xl border border-border p-4 flex items-center gap-3">
              <div className="h-10 w-10 rounded-xl bg-emerald-500/10 flex items-center justify-center">
                <CheckCircle2 className="h-5 w-5 text-emerald-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">স্টক আছে{selectedDuration ? ` (${selectedDuration})` : ""}</p>
                <p className="text-2xl font-bold font-heading">{availableCount}</p>
              </div>
            </div>
            <div className="bg-card rounded-2xl border border-border p-4 flex items-center gap-3">
              <div className="h-10 w-10 rounded-xl bg-amber-500/10 flex items-center justify-center">
                <Package className="h-5 w-5 text-amber-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">বিক্রি হয়েছে{selectedDuration ? ` (${selectedDuration})` : ""}</p>
                <p className="text-2xl font-bold font-heading">{soldCount}</p>
              </div>
            </div>
          </div>

          {/* Add Key */}
          <div className="bg-card rounded-2xl border border-border p-6">
            <div className="flex items-center gap-3 mb-4">
              <div className="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center">
                <Plus className="h-4 w-4 text-primary" />
              </div>
              <div>
                <h3 className="font-heading font-semibold text-sm">নতুন কী যোগ করুন</h3>
                <p className="text-xs text-muted-foreground">একটি কী/ক্রেডেনশিয়াল লিখুন</p>
              </div>
            </div>

            {/* Duration selector for new key */}
            {hasDurations && (
              <div className="mb-3">
                <label className="text-xs font-medium text-foreground mb-1.5 block">ডিউরেশন *</label>
                <div className="flex flex-wrap gap-2">
                  {currentProduct!.duration_options!.map(d => (
                    <button
                      key={d}
                      onClick={() => setNewKeyDuration(d)}
                      className={`px-4 py-2 text-sm font-medium border rounded-lg transition-all ${
                        newKeyDuration === d ? "border-primary bg-primary text-primary-foreground" : "border-border bg-card text-foreground hover:border-primary"
                      }`}
                    >
                      {d}
                    </button>
                  ))}
                </div>
              </div>
            )}

            <textarea
              value={newKeys}
              onChange={e => setNewKeys(e.target.value)}
              placeholder={"email@example.com:password123\nবা মাল্টি-লাইন কী/ক্রেডেনশিয়াল পেস্ট করুন..."}
              className="flex w-full rounded-xl border border-input bg-background px-3 py-2 text-sm font-mono ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring min-h-[100px] resize-y mb-3"
            />
            <Button onClick={handleAddKeys} disabled={adding || !newKeys.trim() || (hasDurations && !newKeyDuration)} className="gap-1.5 rounded-xl">
              {adding ? <Loader2 className="h-4 w-4 animate-spin" /> : <Plus className="h-4 w-4" />}
              {adding ? "যোগ হচ্ছে..." : "কী যোগ করুন"}
            </Button>
          </div>

          {/* Keys List */}
          <div className="bg-card rounded-2xl border border-border overflow-hidden">
            <div className="px-6 py-4 border-b border-border flex items-center gap-3">
              <div className="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center">
                <Key className="h-4 w-4 text-primary" />
              </div>
              <h3 className="font-heading font-semibold text-sm">
                সকল কী ({filteredKeys.length}){selectedDuration ? ` — ${selectedDuration}` : ""}
              </h3>
            </div>
            <div className="divide-y divide-border max-h-[500px] overflow-y-auto">
              {loading ? (
                <div className="flex items-center justify-center py-12">
                  <Loader2 className="h-6 w-6 text-primary animate-spin" />
                </div>
              ) : filteredKeys.length === 0 ? (
                <div className="text-center py-12 text-muted-foreground">
                  <Key className="h-10 w-10 mx-auto mb-3 opacity-30" />
                  <p className="text-sm">কোনো কী নেই। উপরে যোগ করুন।</p>
                </div>
              ) : (
                filteredKeys.map(k => (
                  <div key={k.id}>
                    <div
                      className={`px-6 py-3 flex items-center gap-3 hover:bg-muted/30 transition-colors ${k.is_sold ? "cursor-pointer" : ""}`}
                      onClick={() => k.is_sold && setExpandedKey(expandedKey === k.id ? null : k.id)}
                    >
                      <div className={`h-2 w-2 rounded-full flex-shrink-0 ${k.is_sold ? "bg-amber-500" : "bg-emerald-500"}`} />
                      <code className="text-xs font-mono flex-1 truncate">{k.key_value}</code>
                      {k.duration_option && (
                        <span className="text-[10px] bg-primary/10 text-primary px-2 py-0.5 rounded-lg font-medium flex-shrink-0">
                          {k.duration_option}
                        </span>
                      )}
                      {k.is_sold ? (
                        <div className="flex items-center gap-1.5 flex-shrink-0">
                          <span className="text-[10px] bg-amber-500/10 text-amber-600 px-2 py-0.5 rounded-lg font-medium">বিক্রি হয়েছে</span>
                          {k.buyer_name && (
                            <span className="text-[10px] bg-muted text-muted-foreground px-2 py-0.5 rounded-lg font-medium max-w-[120px] truncate">
                              👤 {k.buyer_name}
                            </span>
                          )}
                        </div>
                      ) : (
                        <span className="text-[10px] bg-emerald-500/10 text-emerald-600 px-2 py-0.5 rounded-lg font-medium flex-shrink-0">স্টক</span>
                      )}
                      {!k.is_sold && (
                        <Button variant="ghost" size="icon" className="h-7 w-7 text-destructive hover:text-destructive flex-shrink-0"
                          onClick={(e) => { e.stopPropagation(); handleDeleteKey(k.id); }}>
                          <Trash2 className="h-3.5 w-3.5" />
                        </Button>
                      )}
                    </div>
                    {/* Expanded detail panel */}
                    {expandedKey === k.id && k.is_sold && (
                      <div className="px-6 py-4 bg-muted/40 border-t border-border">
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                          <div className="flex items-center gap-2">
                            <User className="h-3.5 w-3.5 text-muted-foreground" />
                            <span className="text-xs text-muted-foreground">ক্রেতা:</span>
                            <span className="text-xs font-medium">{k.buyer_name || "N/A"}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Phone className="h-3.5 w-3.5 text-muted-foreground" />
                            <span className="text-xs text-muted-foreground">ফোন:</span>
                            <span className="text-xs font-medium">{k.buyer_phone || "N/A"}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <span className="text-xs text-muted-foreground">📧</span>
                            <span className="text-xs text-muted-foreground">ইমেইল:</span>
                            <span className="text-xs font-medium">{k.buyer_email || "N/A"}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Calendar className="h-3.5 w-3.5 text-muted-foreground" />
                            <span className="text-xs text-muted-foreground">বিক্রির তারিখ:</span>
                            <span className="text-xs font-medium">
                              {k.sold_at ? new Date(k.sold_at).toLocaleDateString("bn-BD", { year: "numeric", month: "long", day: "numeric", hour: "2-digit", minute: "2-digit" }) : "N/A"}
                            </span>
                          </div>
                          {k.orderInfo && (
                            <>
                              <div className="flex items-center gap-2">
                                <Hash className="h-3.5 w-3.5 text-muted-foreground" />
                                <span className="text-xs text-muted-foreground">অর্ডার ID:</span>
                                <span className="text-xs font-mono font-medium">{k.orderInfo.id.slice(0, 8)}...</span>
                              </div>
                              <div className="flex items-center gap-2">
                                <Package className="h-3.5 w-3.5 text-muted-foreground" />
                                <span className="text-xs text-muted-foreground">মূল্য:</span>
                                <span className="text-xs font-medium">{k.orderInfo.amount}</span>
                              </div>
                              <div className="flex items-center gap-2">
                                <CheckCircle2 className="h-3.5 w-3.5 text-muted-foreground" />
                                <span className="text-xs text-muted-foreground">পেমেন্ট:</span>
                                <span className={`text-xs font-medium ${k.orderInfo.payment_status === "paid" ? "text-emerald-600" : "text-amber-600"}`}>
                                  {k.orderInfo.payment_status === "paid" ? "✅ পেইড" : k.orderInfo.payment_status || "N/A"}
                                </span>
                              </div>
                            </>
                          )}
                        </div>
                      </div>
                    )}
                  </div>
                ))
              )}
            </div>
          </div>
        </>
      )}
    </div>
  );
};

export default AdminProductKeys;
