import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { Plus, Trash2, Edit2, Save, XCircle, Loader2, Tag, Percent, DollarSign, History, BarChart3, Users } from "lucide-react";

interface PromoCode {
  id: string;
  code: string;
  discount_type: string;
  discount_value: number;
  min_order_amount: number;
  max_uses: number | null;
  used_count: number;
  is_active: boolean;
  expires_at: string | null;
  created_at: string;
}

interface PromoUsageRecord {
  id: string;
  promo_code_id: string;
  user_id: string;
  order_id: string | null;
  discount_amount: number;
  created_at: string;
  promo_code?: string;
}

const AdminPromoPanel = () => {
  const [promos, setPromos] = useState<PromoCode[]>([]);
  const [loading, setLoading] = useState(true);
  const [showAdd, setShowAdd] = useState(false);
  const [saving, setSaving] = useState(false);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<"codes" | "history" | "report">("codes");
  const [usageHistory, setUsageHistory] = useState<PromoUsageRecord[]>([]);
  const [historyLoading, setHistoryLoading] = useState(false);
  const { toast } = useToast();

  const emptyForm = {
    code: "",
    discount_type: "fixed" as "fixed" | "percentage",
    discount_value: "",
    min_order_amount: "",
    max_uses: "",
    is_active: true,
    expires_at: "",
  };
  const [form, setForm] = useState(emptyForm);

  const fetchPromos = async () => {
    setLoading(true);
    const { data } = await supabase
      .from("promo_codes" as any)
      .select("*")
      .order("created_at", { ascending: false });
    if (data) setPromos(data as any);
    setLoading(false);
  };

  const fetchUsageHistory = async () => {
    setHistoryLoading(true);
    const { data } = await supabase
      .from("promo_usage" as any)
      .select("*")
      .order("created_at", { ascending: false })
      .limit(100);
    if (data) {
      // Enrich with promo code names
      const enriched = (data as any[]).map(u => ({
        ...u,
        promo_code: promos.find(p => p.id === u.promo_code_id)?.code || "—",
      }));
      setUsageHistory(enriched);
    }
    setHistoryLoading(false);
  };

  useEffect(() => { fetchPromos(); }, []);

  useEffect(() => {
    if (activeTab === "history" || activeTab === "report") {
      fetchUsageHistory();
    }
  }, [activeTab, promos]);

  const handleSave = async () => {
    if (!form.code.trim()) {
      toast({ title: "প্রোমো কোড লিখুন", variant: "destructive" });
      return;
    }
    if (!form.discount_value || Number(form.discount_value) <= 0) {
      toast({ title: "ডিসকাউন্ট ভ্যালু দিন", variant: "destructive" });
      return;
    }

    setSaving(true);
    const payload = {
      code: form.code.trim().toUpperCase(),
      discount_type: form.discount_type,
      discount_value: Number(form.discount_value),
      min_order_amount: form.min_order_amount ? Number(form.min_order_amount) : 0,
      max_uses: form.max_uses ? Number(form.max_uses) : null,
      is_active: form.is_active,
      expires_at: form.expires_at || null,
    };

    if (editingId) {
      const { error } = await supabase
        .from("promo_codes" as any)
        .update(payload as any)
        .eq("id", editingId);
      if (error) toast({ title: "আপডেট ব্যর্থ", description: error.message, variant: "destructive" });
      else toast({ title: "✅ প্রোমো কোড আপডেট হয়েছে" });
    } else {
      const { error } = await supabase
        .from("promo_codes" as any)
        .insert(payload as any);
      if (error) toast({ title: "যোগ করা ব্যর্থ", description: error.message, variant: "destructive" });
      else toast({ title: "✅ প্রোমো কোড যোগ হয়েছে" });
    }

    setSaving(false);
    setShowAdd(false);
    setEditingId(null);
    setForm(emptyForm);
    fetchPromos();
  };

  const handleEdit = (promo: PromoCode) => {
    setEditingId(promo.id);
    setForm({
      code: promo.code,
      discount_type: promo.discount_type as "fixed" | "percentage",
      discount_value: String(promo.discount_value),
      min_order_amount: promo.min_order_amount ? String(promo.min_order_amount) : "",
      max_uses: promo.max_uses ? String(promo.max_uses) : "",
      is_active: promo.is_active,
      expires_at: promo.expires_at ? promo.expires_at.slice(0, 16) : "",
    });
    setShowAdd(true);
    setActiveTab("codes");
  };

  const handleDelete = async (id: string) => {
    const { error } = await supabase.from("promo_codes" as any).delete().eq("id", id);
    if (error) toast({ title: "ডিলিট ব্যর্থ", variant: "destructive" });
    else {
      toast({ title: "🗑️ প্রোমো কোড ডিলিট হয়েছে" });
      fetchPromos();
    }
  };

  const toggleActive = async (promo: PromoCode) => {
    await supabase
      .from("promo_codes" as any)
      .update({ is_active: !promo.is_active } as any)
      .eq("id", promo.id);
    fetchPromos();
  };

  // Report stats
  const totalDiscountGiven = usageHistory.reduce((sum, u) => sum + Number(u.discount_amount || 0), 0);
  const uniqueUsers = new Set(usageHistory.map(u => u.user_id)).size;
  const perCodeStats = promos.map(p => {
    const uses = usageHistory.filter(u => u.promo_code_id === p.id);
    const totalDiscount = uses.reduce((s, u) => s + Number(u.discount_amount || 0), 0);
    return { code: p.code, uses: uses.length, totalDiscount, isActive: p.is_active };
  }).filter(s => s.uses > 0).sort((a, b) => b.uses - a.uses);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between flex-wrap gap-2">
        <h2 className="font-heading text-lg font-bold flex items-center gap-2">
          <Tag className="h-5 w-5 text-primary" /> প্রোমো কোড
        </h2>
        <Button
          size="sm"
          className="rounded-xl gap-1.5"
          onClick={() => { setShowAdd(!showAdd); setEditingId(null); setForm(emptyForm); setActiveTab("codes"); }}
        >
          <Plus className="h-4 w-4" /> নতুন কোড
        </Button>
      </div>

      {/* Tabs */}
      <div className="flex gap-2 flex-wrap">
        {[
          { key: "codes" as const, label: "কোড সমূহ", icon: Tag },
          { key: "history" as const, label: "ব্যবহারের হিস্ট্রি", icon: History },
          { key: "report" as const, label: "রিপোর্ট", icon: BarChart3 },
        ].map(tab => (
          <button
            key={tab.key}
            onClick={() => setActiveTab(tab.key)}
            className={`px-4 py-2 rounded-xl text-sm font-medium transition-all flex items-center gap-1.5 ${activeTab === tab.key ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground hover:text-foreground"}`}
          >
            <tab.icon className="h-3.5 w-3.5" />
            {tab.label}
          </button>
        ))}
      </div>

      {/* ===== CODES TAB ===== */}
      {activeTab === "codes" && (
        <>
          {/* Add/Edit Form */}
          {showAdd && (
            <div className="bg-card rounded-2xl border border-border p-5 space-y-4">
              <h3 className="font-heading font-semibold text-sm">
                {editingId ? "প্রোমো কোড এডিট করুন" : "নতুন প্রোমো কোড যোগ করুন"}
              </h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div className="space-y-1.5">
                  <Label className="text-xs">কোড *</Label>
                  <Input value={form.code} onChange={e => setForm(f => ({ ...f, code: e.target.value.toUpperCase() }))} placeholder="SAVE20" className="rounded-xl uppercase" maxLength={30} />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">ডিসকাউন্ট টাইপ</Label>
                  <div className="flex gap-2">
                    <Button type="button" size="sm" variant={form.discount_type === "fixed" ? "default" : "outline"} className="rounded-xl gap-1 flex-1" onClick={() => setForm(f => ({ ...f, discount_type: "fixed" }))}>
                      <DollarSign className="h-3 w-3" /> ৳ Fixed
                    </Button>
                    <Button type="button" size="sm" variant={form.discount_type === "percentage" ? "default" : "outline"} className="rounded-xl gap-1 flex-1" onClick={() => setForm(f => ({ ...f, discount_type: "percentage" }))}>
                      <Percent className="h-3 w-3" /> শতাংশ
                    </Button>
                  </div>
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">ডিসকাউন্ট ভ্যালু * {form.discount_type === "percentage" ? "(%" : "(৳"}{")"}</Label>
                  <Input type="number" value={form.discount_value} onChange={e => setForm(f => ({ ...f, discount_value: e.target.value }))} placeholder={form.discount_type === "percentage" ? "10" : "50"} className="rounded-xl" min={0} />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">মিনিমাম অর্ডার (৳)</Label>
                  <Input type="number" value={form.min_order_amount} onChange={e => setForm(f => ({ ...f, min_order_amount: e.target.value }))} placeholder="0" className="rounded-xl" min={0} />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">সর্বোচ্চ ব্যবহার (খালি = আনলিমিটেড)</Label>
                  <Input type="number" value={form.max_uses} onChange={e => setForm(f => ({ ...f, max_uses: e.target.value }))} placeholder="আনলিমিটেড" className="rounded-xl" min={0} />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">মেয়াদ শেষ (খালি = কোনো মেয়াদ নেই)</Label>
                  <Input type="datetime-local" value={form.expires_at} onChange={e => setForm(f => ({ ...f, expires_at: e.target.value }))} className="rounded-xl" />
                </div>
              </div>
              <div className="flex items-center gap-2">
                <Switch checked={form.is_active} onCheckedChange={v => setForm(f => ({ ...f, is_active: v }))} />
                <Label className="text-xs">সক্রিয়</Label>
              </div>
              <div className="flex gap-2">
                <Button size="sm" className="rounded-xl gap-1.5" onClick={handleSave} disabled={saving}>
                  {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
                  {editingId ? "আপডেট করুন" : "যোগ করুন"}
                </Button>
                <Button size="sm" variant="outline" className="rounded-xl gap-1.5" onClick={() => { setShowAdd(false); setEditingId(null); setForm(emptyForm); }}>
                  <XCircle className="h-4 w-4" /> বাতিল
                </Button>
              </div>
            </div>
          )}

          {/* Promo List */}
          {loading ? (
            <div className="flex justify-center py-12"><Loader2 className="h-6 w-6 animate-spin text-primary" /></div>
          ) : promos.length === 0 ? (
            <div className="text-center py-12 text-muted-foreground">কোনো প্রোমো কোড নেই</div>
          ) : (
            <div className="space-y-3">
              {promos.map(promo => (
                <div key={promo.id} className="bg-card rounded-xl border border-border p-4 flex flex-col sm:flex-row sm:items-center gap-3">
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2 flex-wrap">
                      <span className="font-mono font-bold text-primary text-sm bg-primary/10 px-2 py-0.5 rounded">{promo.code}</span>
                      <span className={`text-xs px-2 py-0.5 rounded ${promo.is_active ? "bg-green-500/10 text-green-600" : "bg-muted text-muted-foreground"}`}>
                        {promo.is_active ? "সক্রিয়" : "নিষ্ক্রিয়"}
                      </span>
                    </div>
                    <p className="text-xs text-muted-foreground mt-1">
                      {promo.discount_type === "percentage" ? `${promo.discount_value}% ছাড়` : `৳${promo.discount_value} ছাড়`}
                      {promo.min_order_amount > 0 && ` • মিনিমাম ৳${promo.min_order_amount}`}
                      {promo.max_uses && ` • ${promo.used_count}/${promo.max_uses} ব্যবহৃত`}
                      {!promo.max_uses && ` • ${promo.used_count} বার ব্যবহৃত`}
                      {promo.expires_at && ` • মেয়াদ: ${new Date(promo.expires_at).toLocaleDateString("bn-BD")}`}
                    </p>
                  </div>
                  <div className="flex items-center gap-2 flex-shrink-0">
                    <Switch checked={promo.is_active} onCheckedChange={() => toggleActive(promo)} />
                    <Button size="sm" variant="outline" className="rounded-lg h-8 w-8 p-0" onClick={() => handleEdit(promo)}>
                      <Edit2 className="h-3.5 w-3.5" />
                    </Button>
                    <Button size="sm" variant="outline" className="rounded-lg h-8 w-8 p-0 text-destructive hover:text-destructive" onClick={() => handleDelete(promo.id)}>
                      <Trash2 className="h-3.5 w-3.5" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          )}
        </>
      )}

      {/* ===== HISTORY TAB ===== */}
      {activeTab === "history" && (
        <div className="space-y-4">
          {historyLoading ? (
            <div className="flex justify-center py-12"><Loader2 className="h-6 w-6 animate-spin text-primary" /></div>
          ) : usageHistory.length === 0 ? (
            <div className="text-center py-12 text-muted-foreground">এখনো কোনো প্রোমো কোড ব্যবহৃত হয়নি</div>
          ) : (
            <div className="bg-card rounded-xl border border-border overflow-hidden">
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead>
                    <tr className="border-b border-border bg-muted/50">
                      <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground">কোড</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground">ইউজার ID</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground">ডিসকাউন্ট</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground">তারিখ</th>
                    </tr>
                  </thead>
                  <tbody>
                    {usageHistory.map(u => (
                      <tr key={u.id} className="border-b border-border last:border-0 hover:bg-muted/30 transition-colors">
                        <td className="px-4 py-3">
                          <span className="font-mono font-bold text-primary text-xs bg-primary/10 px-1.5 py-0.5 rounded">{u.promo_code}</span>
                        </td>
                        <td className="px-4 py-3 text-xs text-muted-foreground font-mono truncate max-w-[120px]">{u.user_id?.slice(0, 8)}...</td>
                        <td className="px-4 py-3 text-xs font-semibold">৳{u.discount_amount}</td>
                        <td className="px-4 py-3 text-xs text-muted-foreground">{new Date(u.created_at).toLocaleString("bn-BD")}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}
        </div>
      )}

      {/* ===== REPORT TAB ===== */}
      {activeTab === "report" && (
        <div className="space-y-4">
          {/* Summary Stats */}
          <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
            <div className="bg-card rounded-xl border border-border p-4 text-center">
              <div className="h-10 w-10 rounded-lg bg-primary/10 flex items-center justify-center mx-auto mb-2">
                <Tag className="h-5 w-5 text-primary" />
              </div>
              <p className="text-2xl font-bold font-heading">{usageHistory.length}</p>
              <p className="text-xs text-muted-foreground mt-1">মোট ব্যবহার</p>
            </div>
            <div className="bg-card rounded-xl border border-border p-4 text-center">
              <div className="h-10 w-10 rounded-lg bg-primary/10 flex items-center justify-center mx-auto mb-2">
                <DollarSign className="h-5 w-5 text-primary" />
              </div>
              <p className="text-2xl font-bold font-heading">৳{totalDiscountGiven}</p>
              <p className="text-xs text-muted-foreground mt-1">মোট ডিসকাউন্ট দেওয়া হয়েছে</p>
            </div>
            <div className="bg-card rounded-xl border border-border p-4 text-center">
              <div className="h-10 w-10 rounded-lg bg-primary/10 flex items-center justify-center mx-auto mb-2">
                <Users className="h-5 w-5 text-primary" />
              </div>
              <p className="text-2xl font-bold font-heading">{uniqueUsers}</p>
              <p className="text-xs text-muted-foreground mt-1">ইউনিক ইউজার</p>
            </div>
          </div>

          {/* Per-code breakdown */}
          {perCodeStats.length > 0 && (
            <div className="bg-card rounded-xl border border-border overflow-hidden">
              <div className="px-4 py-3 border-b border-border bg-muted/50">
                <h3 className="text-sm font-semibold">কোড অনুযায়ী ব্রেকডাউন</h3>
              </div>
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead>
                    <tr className="border-b border-border">
                      <th className="px-4 py-2.5 text-left text-xs font-medium text-muted-foreground">কোড</th>
                      <th className="px-4 py-2.5 text-left text-xs font-medium text-muted-foreground">ব্যবহার</th>
                      <th className="px-4 py-2.5 text-left text-xs font-medium text-muted-foreground">মোট ডিসকাউন্ট</th>
                      <th className="px-4 py-2.5 text-left text-xs font-medium text-muted-foreground">স্ট্যাটাস</th>
                    </tr>
                  </thead>
                  <tbody>
                    {perCodeStats.map(s => (
                      <tr key={s.code} className="border-b border-border last:border-0">
                        <td className="px-4 py-2.5">
                          <span className="font-mono font-bold text-primary text-xs bg-primary/10 px-1.5 py-0.5 rounded">{s.code}</span>
                        </td>
                        <td className="px-4 py-2.5 text-xs">{s.uses} বার</td>
                        <td className="px-4 py-2.5 text-xs font-semibold">৳{s.totalDiscount}</td>
                        <td className="px-4 py-2.5">
                          <span className={`text-xs px-2 py-0.5 rounded ${s.isActive ? "bg-green-500/10 text-green-600" : "bg-muted text-muted-foreground"}`}>
                            {s.isActive ? "সক্রিয়" : "নিষ্ক্রিয়"}
                          </span>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}

          {perCodeStats.length === 0 && !historyLoading && (
            <div className="text-center py-8 text-muted-foreground text-sm">এখনো কোনো রিপোর্ট ডেটা নেই</div>
          )}
        </div>
      )}
    </div>
  );
};

export default AdminPromoPanel;
