import { X, Trash2, ShoppingBag, Minus, Plus } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useCart } from "@/contexts/CartContext";
import { Link } from "react-router-dom";
import { parseDurationOption } from "@/lib/priceUtils";

const CartDrawer = () => {
  const { items, isCartOpen, setIsCartOpen, removeFromCart, updateQuantity, totalItems } = useCart();

  if (!isCartOpen) return null;

  return (
    <>
      {/* Overlay */}
      <div
        className="fixed inset-0 bg-foreground/40 z-[60] backdrop-blur-sm"
        onClick={() => setIsCartOpen(false)}
      />

      {/* Drawer */}
      <div className="fixed top-0 right-0 h-full w-full max-w-md bg-background border-l border-border shadow-2xl z-[70] flex flex-col animate-fade-in">
        {/* Header */}
        <div className="flex items-center justify-between p-4 border-b border-border">
          <h2 className="font-heading text-lg font-bold text-foreground flex items-center gap-2">
            <ShoppingBag className="h-5 w-5 text-primary" />
            My Cart ({totalItems})
          </h2>
          <button
            onClick={() => setIsCartOpen(false)}
            className="p-1.5 rounded-md hover:bg-secondary transition-colors"
          >
            <X className="h-5 w-5 text-foreground" />
          </button>
        </div>

        {/* Items */}
        <div className="flex-1 overflow-y-auto p-4">
          {items.length === 0 ? (
            <div className="flex flex-col items-center justify-center h-full text-center">
              <ShoppingBag className="h-16 w-16 text-muted-foreground/30 mb-4" />
              <p className="font-heading font-semibold text-foreground mb-1">Your cart is empty</p>
              <p className="text-sm text-muted-foreground mb-4">No products in the cart.</p>
              <Button
                variant="outline"
                onClick={() => setIsCartOpen(false)}
                asChild
              >
                <Link to="/">RETURN TO SHOP</Link>
              </Button>
            </div>
          ) : (
            <div className="space-y-4">
              {items.map((item, idx) => (
                <div
                  key={`${item.product.id}-${item.selectedOption}-${idx}`}
                  className="flex gap-3 p-3 bg-secondary/50 rounded-lg border border-border"
                >
                  <Link
                    to={`/product/${item.product.id}`}
                    onClick={() => setIsCartOpen(false)}
                    className="w-16 h-16 flex-shrink-0 bg-secondary rounded-md overflow-hidden"
                  >
                    <img
                      src={item.product.image}
                      alt={item.product.name}
                      className="w-full h-full object-contain p-1"
                    />
                  </Link>
                  <div className="flex-1 min-w-0">
                    <Link
                      to={`/product/${item.product.id}`}
                      onClick={() => setIsCartOpen(false)}
                      className="font-heading text-sm font-semibold text-foreground hover:text-primary transition-colors line-clamp-1"
                    >
                      {item.product.name}
                    </Link>
                    {item.selectedOption && (
                      <p className="text-xs text-muted-foreground mt-0.5">{item.selectedOption}</p>
                    )}
                    <p className="text-sm font-bold text-primary mt-1">
                      {item.selectedOption
                        ? parseDurationOption(item.selectedOption).price || item.product.price
                        : item.product.price}
                    </p>
                    <div className="flex items-center justify-between mt-2">
                      <div className="flex items-center border border-border rounded overflow-hidden">
                        <button
                          onClick={() => updateQuantity(item.product.id, item.selectedOption, item.quantity - 1)}
                          className="px-2 py-1 hover:bg-secondary transition-colors"
                        >
                          <Minus className="h-3 w-3" />
                        </button>
                        <span className="px-3 py-1 text-xs font-semibold border-x border-border">
                          {item.quantity}
                        </span>
                        <button
                          onClick={() => updateQuantity(item.product.id, item.selectedOption, item.quantity + 1)}
                          className="px-2 py-1 hover:bg-secondary transition-colors"
                        >
                          <Plus className="h-3 w-3" />
                        </button>
                      </div>
                      <button
                        onClick={() => removeFromCart(item.product.id, item.selectedOption)}
                        className="p-1.5 text-muted-foreground hover:text-destructive transition-colors"
                      >
                        <Trash2 className="h-4 w-4" />
                      </button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>

        {/* Footer */}
        {items.length > 0 && (
          <div className="p-4 border-t border-border space-y-3">
            <Button
              className="w-full bg-primary text-primary-foreground hover:opacity-90 h-11 font-semibold"
              onClick={() => setIsCartOpen(false)}
              asChild
            >
              <Link to="/checkout">Checkout</Link>
            </Button>
            <Button
              variant="outline"
              className="w-full h-11"
              onClick={() => setIsCartOpen(false)}
              asChild
            >
              <Link to="/">Continue Shopping</Link>
            </Button>
          </div>
        )}
      </div>
    </>
  );
};

export default CartDrawer;
