import { useState, useEffect } from "react";
import { useLocation } from "react-router-dom";
import { motion, AnimatePresence } from "framer-motion";
import { MessageCircle, X } from "lucide-react";
import { supabase } from "@/integrations/supabase/client";

interface SupportLinks {
  whatsapp: string;
  messenger: string;
  telegram: string;
}

const FloatingSupportButton = () => {
  const [open, setOpen] = useState(false);
  const [links, setLinks] = useState<SupportLinks>({ whatsapp: "", messenger: "", telegram: "" });
  const [bannerVisible, setBannerVisible] = useState(!sessionStorage.getItem("app_install_dismissed"));
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const location = useLocation();

  // Hide on admin pages
  const isAdminPage = location.pathname.startsWith("/admin");

  useEffect(() => {
    const onDismiss = () => setBannerVisible(false);
    const onSidebarOpen = () => setSidebarOpen(true);
    const onSidebarClose = () => setSidebarOpen(false);
    window.addEventListener("app-install-dismissed", onDismiss);
    window.addEventListener("mobile-sidebar-open", onSidebarOpen);
    window.addEventListener("mobile-sidebar-close", onSidebarClose);
    return () => {
      window.removeEventListener("app-install-dismissed", onDismiss);
      window.removeEventListener("mobile-sidebar-open", onSidebarOpen);
      window.removeEventListener("mobile-sidebar-close", onSidebarClose);
    };
  }, []);

  useEffect(() => {
    const fetch = async () => {
      const { data } = await supabase
        .from("site_settings")
        .select("key, value")
        .in("key", ["support_whatsapp", "support_messenger", "support_telegram"]);
      if (data) {
        const map: Record<string, string> = {};
        data.forEach((r) => { if (r.value) map[r.key] = r.value; });
        setLinks({
          whatsapp: map.support_whatsapp || "",
          messenger: map.support_messenger || "",
          telegram: map.support_telegram || "",
        });
      }
    };
    fetch();
  }, []);

  const hasAnyLink = links.whatsapp || links.messenger || links.telegram;
  if (!hasAnyLink || isAdminPage || sidebarOpen) return null;

  const channels = [
    {
      key: "whatsapp",
      label: "WhatsApp",
      url: links.whatsapp,
      color: "#25D366",
      icon: (
        <svg viewBox="0 0 24 24" className="h-5 w-5 fill-white">
          <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413z" />
        </svg>
      ),
    },
    {
      key: "messenger",
      label: "Messenger",
      url: links.messenger,
      color: "#0084FF",
      icon: (
        <svg viewBox="0 0 24 24" className="h-5 w-5 fill-white">
          <path d="M12 0C5.373 0 0 4.974 0 11.111c0 3.498 1.744 6.614 4.469 8.654V24l4.088-2.242c1.092.3 2.246.464 3.443.464 6.627 0 12-4.975 12-11.111S18.627 0 12 0zm1.191 14.963l-3.055-3.26-5.963 3.26L10.732 8.2l3.131 3.259L19.752 8.2l-6.561 6.763z" />
        </svg>
      ),
    },
    {
      key: "telegram",
      label: "Telegram",
      url: links.telegram,
      color: "#0088CC",
      icon: (
        <svg viewBox="0 0 24 24" className="h-5 w-5 fill-white">
          <path d="M11.944 0A12 12 0 000 12a12 12 0 0012 12 12 12 0 0012-12A12 12 0 0012 0 12 12 0 0011.944 0zm4.962 7.224c.1-.002.321.023.465.14a.506.506 0 01.171.325c.016.093.036.306.02.472-.18 1.898-.962 6.502-1.36 8.627-.168.9-.499 1.201-.82 1.23-.696.065-1.225-.46-1.9-.902-1.056-.693-1.653-1.124-2.678-1.8-1.185-.78-.417-1.21.258-1.91.177-.184 3.247-2.977 3.307-3.23.007-.032.014-.15-.056-.212s-.174-.041-.249-.024c-.106.024-1.793 1.14-5.061 3.345-.479.33-.913.49-1.302.48-.428-.008-1.252-.241-1.865-.44-.752-.245-1.349-.374-1.297-.789.027-.216.325-.437.893-.663 3.498-1.524 5.83-2.529 6.998-3.014 3.332-1.386 4.025-1.627 4.476-1.635z" />
        </svg>
      ),
    },
  ].filter((c) => c.url);

  return (
    <>
      {/* Overlay when open */}
      <AnimatePresence>
        {open && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="fixed inset-0 z-[999] bg-black/20 backdrop-blur-[2px]"
            onClick={() => setOpen(false)}
          />
        )}
      </AnimatePresence>

      {/* Fixed container */}
      <div className={`fixed ${bannerVisible ? 'bottom-[120px]' : 'bottom-[72px]'} sm:bottom-6 right-3 z-[1000] flex flex-col items-end gap-2 transition-all duration-300`}>
        {/* Channel buttons */}
        <AnimatePresence>
          {open &&
            channels.map((ch, i) => (
              <motion.a
                key={ch.key}
                href={ch.url}
                target="_blank"
                rel="noopener noreferrer"
                initial={{ opacity: 0, scale: 0.3, y: 30 }}
                animate={{ opacity: 1, scale: 1, y: 0 }}
                exit={{ opacity: 0, scale: 0.3, y: 30 }}
                transition={{ delay: i * 0.07, type: "spring", stiffness: 350, damping: 22 }}
                className="flex items-center gap-2"
              >
                <span
                  className="text-white text-xs font-bold px-3 py-1 rounded-full shadow-md whitespace-nowrap"
                  style={{ backgroundColor: ch.color }}
                >
                  {ch.label}
                </span>
                <div
                  className="h-12 w-12 rounded-full flex items-center justify-center shadow-lg hover:scale-110 active:scale-95 transition-transform duration-150"
                  style={{ backgroundColor: ch.color }}
                >
                  {ch.icon}
                </div>
              </motion.a>
            ))}
        </AnimatePresence>

        {/* Main toggle button */}
        <motion.button
          onClick={() => setOpen(!open)}
          whileTap={{ scale: 0.9 }}
          className="relative h-14 w-14 rounded-full bg-primary text-primary-foreground shadow-xl flex items-center justify-center"
        >
          {!open && (
            <span className="absolute inset-0 rounded-full bg-primary/40 animate-ping" style={{ animationDuration: '2.5s' }} />
          )}
          <AnimatePresence mode="wait" initial={false}>
            {open ? (
              <motion.div key="close" initial={{ rotate: -90, opacity: 0 }} animate={{ rotate: 0, opacity: 1 }} exit={{ rotate: 90, opacity: 0 }} transition={{ duration: 0.15 }}>
                <X className="h-6 w-6" />
              </motion.div>
            ) : (
              <motion.div key="open" initial={{ rotate: 90, opacity: 0 }} animate={{ rotate: 0, opacity: 1 }} exit={{ rotate: -90, opacity: 0 }} transition={{ duration: 0.15 }}>
                <MessageCircle className="h-6 w-6" />
              </motion.div>
            )}
          </AnimatePresence>
        </motion.button>
      </div>
    </>
  );
};

export default FloatingSupportButton;
