import { useState, useEffect, useRef } from "react";
import { createPortal } from "react-dom";
import { useNavigate } from "react-router-dom";
import { Search, ShoppingCart, Heart, Menu, X, ChevronDown, ChevronRight, User, LogOut, Home, Store, Package, Info, Headphones, BarChart3, ExternalLink } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarImage, AvatarFallback } from "@/components/ui/avatar";
import { useCart } from "@/contexts/CartContext";
import { useAuth } from "@/contexts/AuthContext";
import { supabase } from "@/integrations/supabase/client";
import type { Product } from "@/components/productsData";
import { useSiteLogo } from "@/hooks/useSiteLogo";
import { useSiteSettings } from "@/hooks/useSiteSettings";

const navLinks = [
  { label: "HOME", href: "/" },
  { label: "SHOP", href: "/shop" },
  { label: "BLOG", href: "/blog" },
  { label: "ORDER TRACKING", href: "/orders" },
  { label: "PRODUCT REQUEST", href: "/product-request" },
  { label: "ABOUT US", href: "/about" },
];

const Header = () => {
  const [mobileOpen, setMobileOpenState] = useState(false);
  const [headerHeight, setHeaderHeight] = useState(0);
  const headerRef = useRef<HTMLElement>(null);
  const setMobileOpen = (open: boolean) => {
    setMobileOpenState(open);
    window.dispatchEvent(new Event(open ? "mobile-sidebar-open" : "mobile-sidebar-close"));
  };
  const [mobileSearchHidden, setMobileSearchHidden] = useState(false);
  const lastScrollY = useRef(0);
  const [mobileTab, setMobileTab] = useState<"categories" | "menu">("categories");
  const [catOpen, setCatOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [searchResults, setSearchResults] = useState<Product[]>([]);
  const [showResults, setShowResults] = useState(false);
  const [allProducts, setAllProducts] = useState<Product[]>([]);
  const [categories, setCategories] = useState<{ label: string; id: string }[]>([]);
  const [avatarUrl, setAvatarUrl] = useState<string | null>(null);
  const [customLinks, setCustomLinks] = useState<{ label: string; url: string; icon: string }[]>([]);
  const { totalItems, setIsCartOpen } = useCart();
  const { user, signOut, isReseller } = useAuth();
  const navigate = useNavigate();
  const logoUrl = useSiteLogo();
  const { settings } = useSiteSettings();
  const siteName = settings.site_title || "TZSS";

  useEffect(() => {
    const fetchData = async () => {
      const [catRes, prodRes, linksRes] = await Promise.all([
        supabase.from("categories").select("name").eq("is_active", true).order("sort_order"),
        supabase.from("products").select("*"),
        supabase.from("site_settings").select("value").eq("key", "sidebar_custom_links").single(),
      ]);
      if (catRes.data) {
        setCategories(catRes.data.map((c) => ({ label: c.name, id: c.name.toLowerCase().replace(/\s+/g, "-") })));
      }
      if (prodRes.data) {
        setAllProducts(prodRes.data.map((p: any) => ({
          id: p.id,
          name: p.name,
          image: p.image_url || "/placeholder.svg",
          price: p.price,
          category: p.category,
        })));
      }
      if (linksRes.data?.value) {
        try { setCustomLinks(JSON.parse(linksRes.data.value)); } catch {}
      }
    };
    fetchData();
  }, []);

  useEffect(() => {
    if (!headerRef.current) return;
    const observer = new ResizeObserver(([entry]) => {
      setHeaderHeight(entry.contentRect.height);
    });
    observer.observe(headerRef.current);
    setHeaderHeight(headerRef.current.offsetHeight);
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    const handleScroll = () => {
      const currentY = window.scrollY;
      if (currentY > 60) {
        setMobileSearchHidden(true);
      } else {
        setMobileSearchHidden(false);
      }
      lastScrollY.current = currentY;
    };
    window.addEventListener("scroll", handleScroll, { passive: true });
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    if (!user) { setAvatarUrl(null); return; }
    supabase.from("profiles").select("avatar_url").eq("user_id", user.id).single()
      .then(({ data }) => setAvatarUrl(data?.avatar_url || null));
  }, [user]);

  const handleSearch = (query: string) => {
    setSearchQuery(query);
    if (query.trim().length > 0) {
      const filtered = allProducts.filter((p) =>
        p.name.toLowerCase().includes(query.toLowerCase()) ||
        p.category.toLowerCase().includes(query.toLowerCase())
      );
      setSearchResults(filtered);
      setShowResults(true);
    } else {
      setSearchResults([]);
      setShowResults(false);
    }
  };

  const handleProductClick = (productId: number | string) => {
    setShowResults(false);
    setSearchQuery("");
    setMobileOpen(false);
    navigate(`/product/${productId}`);
  };

  const handleCategoryClick = (id: string) => {
    setCatOpen(false);
    setMobileOpen(false);
    navigate(`/category/${id}`);
  };

  const handleNavClick = (href: string) => {
    setMobileOpen(false);
    navigate(href);
  };

  const sidebarPortal = mobileOpen
    ? createPortal(
        <div className="md:hidden fixed inset-0 z-[9990]" style={{ top: 0 }}>
          <div className="absolute inset-0 bg-black/40" onClick={() => setMobileOpen(false)} />
          <div className="absolute left-0 top-0 bottom-0 w-[85%] max-w-[340px] bg-background shadow-2xl flex flex-col animate-fade-in">
            {/* Close button */}
            <button
              onClick={() => setMobileOpen(false)}
              className="absolute top-3 right-3 z-10 p-1.5 rounded-full bg-muted hover:bg-muted/80 transition-colors"
            >
              <X className="h-5 w-5 text-foreground" />
            </button>
            <div className="flex border-b border-border">
              <button
                onClick={() => setMobileTab("categories")}
                className={`flex-1 py-3.5 text-sm font-bold tracking-wide transition-colors ${
                  mobileTab === "categories"
                    ? "text-primary border-b-[3px] border-primary"
                    : "text-muted-foreground"
                }`}
              >
                CATEGORIES
              </button>
              <button
                onClick={() => setMobileTab("menu")}
                className={`flex-1 py-3.5 text-sm font-bold tracking-wide transition-colors ${
                  mobileTab === "menu"
                    ? "text-primary border-b-[3px] border-primary"
                    : "text-muted-foreground"
                }`}
              >
                MENU
              </button>
            </div>
            <div className="flex-1 overflow-y-auto">
              {mobileTab === "categories" ? (
                <div className="py-1">
                  {categories.map((cat) => (
                    <button
                      key={cat.id}
                      onClick={() => handleCategoryClick(cat.id)}
                      className="flex items-center justify-between w-full px-5 py-3.5 text-sm font-medium text-foreground hover:bg-secondary/50 transition-colors border-b border-border/50"
                    >
                      <span>{cat.label}</span>
                      <ChevronRight className="h-4 w-4 text-muted-foreground" />
                    </button>
                  ))}
                </div>
              ) : (
                <div className="py-1">
                  {navLinks.map((link) => (
                    <button
                      key={link.label}
                      onClick={() => handleNavClick(link.href)}
                      className="flex items-center gap-3 w-full px-5 py-3.5 text-sm font-medium text-foreground hover:bg-secondary/50 transition-colors border-b border-border/50"
                    >
                      {link.label === "HOME" && <Home className="h-4 w-4 text-primary" />}
                      {link.label === "SHOP" && <Store className="h-4 w-4 text-primary" />}
                      {link.label === "BLOG" && <Package className="h-4 w-4 text-primary" />}
                      {link.label === "ORDER TRACKING" && <Search className="h-4 w-4 text-primary" />}
                      {link.label === "PRODUCT REQUEST" && <Headphones className="h-4 w-4 text-primary" />}
                      {link.label === "ABOUT US" && <Info className="h-4 w-4 text-primary" />}
                      <span>{link.label}</span>
                    </button>
                  ))}
                  {customLinks.map((link, i) => (
                    <a
                      key={`custom-${i}`}
                      href={link.url}
                      target="_blank"
                      rel="noopener noreferrer"
                      onClick={() => setMobileOpen(false)}
                      className="flex items-center gap-3 w-full px-5 py-3.5 text-sm font-medium text-foreground hover:bg-secondary/50 transition-colors border-b border-border/50"
                    >
                      <ExternalLink className="h-4 w-4 text-primary" />
                      <span>{link.label}</span>
                    </a>
                  ))}
                  <div className="border-t border-border mt-2 pt-2">
                    {user ? (
                      <>
                        <button
                          onClick={() => { setMobileOpen(false); navigate("/profile"); }}
                          className="flex items-center gap-3 w-full px-5 py-3.5 text-sm font-medium text-foreground hover:bg-secondary/50 transition-colors"
                        >
                          <User className="h-4 w-4 text-primary" />
                          <span>MY ACCOUNT</span>
                        </button>
                        {isReseller && (
                          <button
                            onClick={() => { setMobileOpen(false); navigate("/reseller"); }}
                            className="flex items-center gap-3 w-full px-5 py-3.5 text-sm font-medium text-foreground hover:bg-secondary/50 transition-colors"
                          >
                            <BarChart3 className="h-4 w-4 text-primary" />
                            <span>🏪 RESELLER DASHBOARD</span>
                          </button>
                        )}
                        <button
                          onClick={async () => { await signOut(); setMobileOpen(false); navigate("/"); }}
                          className="flex items-center gap-3 w-full px-5 py-3.5 text-sm font-medium text-destructive hover:bg-secondary/50 transition-colors"
                        >
                          <LogOut className="h-4 w-4" />
                          <span>LOGOUT</span>
                        </button>
                      </>
                    ) : (
                      <button
                        onClick={() => { setMobileOpen(false); navigate("/auth"); }}
                        className="flex items-center gap-3 w-full px-5 py-3.5 text-sm font-medium text-primary hover:bg-secondary/50 transition-colors"
                      >
                        <User className="h-4 w-4" />
                        <span>LOGIN / REGISTER</span>
                      </button>
                    )}
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>,
        document.body
      )
    : null;

  return (
    <>
    <header ref={headerRef} className="fixed top-0 left-0 right-0 z-50 bg-background/95 backdrop-blur-sm border-b border-border">
      {/* Mobile Header */}
      <div className="md:hidden">
        <div className="flex items-center justify-between px-3 py-2">
          <button onClick={() => setMobileOpen(!mobileOpen)}>
            {mobileOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
          </button>
          <a href="/" onClick={(e) => { e.preventDefault(); navigate("/"); }} className="absolute left-1/2 -translate-x-1/2">
            {logoUrl ? (
              <img src={logoUrl} alt={siteName} className="h-8 max-w-[180px] object-contain" />
            ) : (
              <h1 className="font-heading text-lg font-extrabold tracking-tight whitespace-nowrap">
                {siteName}
              </h1>
            )}
          </a>
          <div className="flex items-center gap-1">
            <Button variant="ghost" size="icon" className="relative" onClick={() => setIsCartOpen(true)}>
              <ShoppingCart className="h-5 w-5" />
              {totalItems > 0 && (
                <span className="absolute -top-1 -right-1 h-4 w-4 rounded-full bg-primary text-[10px] font-bold text-primary-foreground flex items-center justify-center">
                  {totalItems}
                </span>
              )}
            </Button>
            <button
              onClick={() => user ? navigate("/profile") : navigate("/auth")}
              className="flex items-center justify-center w-8 h-8 rounded-full border-2 border-primary/30 hover:border-primary transition-colors"
            >
              {user && avatarUrl ? (
                <Avatar className="h-6 w-6">
                  <AvatarImage src={avatarUrl} alt="Profile" />
                  <AvatarFallback className="bg-primary/10 text-primary text-xs"><User className="h-3.5 w-3.5" /></AvatarFallback>
                </Avatar>
              ) : (
                <User className="h-4 w-4 text-muted-foreground" />
              )}
            </button>
          </div>
        </div>
        {/* Mobile Search Bar */}
        <div className={`px-3 pb-2 relative transition-all duration-300 overflow-hidden ${mobileSearchHidden ? "max-h-0 pb-0 opacity-0" : "max-h-20 opacity-100"}`}>
          <div className="flex w-full border border-border rounded-full overflow-hidden bg-secondary/30">
            <input
              type="text"
              placeholder="Search for products"
              value={searchQuery}
              onChange={(e) => handleSearch(e.target.value)}
              onFocus={() => searchQuery.trim() && setShowResults(true)}
              onBlur={() => setTimeout(() => setShowResults(false), 200)}
              className="flex-1 px-4 py-2.5 text-sm bg-transparent text-foreground outline-none"
            />
            <button className="bg-primary text-primary-foreground rounded-full w-10 h-10 flex items-center justify-center mr-0.5 my-0.5">
              <Search className="h-4 w-4" />
            </button>
          </div>
          {showResults && searchResults.length > 0 && (
            <div className="absolute top-full left-4 right-4 mt-1 bg-card border border-border rounded-lg shadow-xl z-50 max-h-60 overflow-y-auto">
              {searchResults.map((product) => (
                <button
                  key={product.id}
                  onMouseDown={() => handleProductClick(product.id)}
                  className="flex items-center gap-3 w-full px-3 py-2 hover:bg-secondary transition-colors text-left"
                >
                  <img src={product.image} alt={product.name} className="w-8 h-8 object-contain rounded" />
                  <div className="flex-1 min-w-0">
                    <p className="text-sm font-medium text-foreground truncate">{product.name}</p>
                  </div>
                  <span className="text-xs font-bold text-primary">{product.price}</span>
                </button>
              ))}
            </div>
          )}
          {showResults && searchQuery.trim() && searchResults.length === 0 && (
            <div className="absolute top-full left-4 right-4 mt-1 bg-card border border-border rounded-lg shadow-xl z-50 p-4 text-center">
              <p className="text-sm text-muted-foreground">No products found for "{searchQuery}"</p>
            </div>
          )}
        </div>
      </div>

      {/* Desktop Header */}
      <div className="hidden md:block">
        <div className="container mx-auto px-4 py-3">
          <div className="flex items-center gap-4">
            <a href="/" onClick={(e) => { e.preventDefault(); navigate("/"); }} className="flex-shrink-0">
              {logoUrl ? (
                <img src={logoUrl} alt={siteName} className="h-10 max-w-[220px] object-contain" />
              ) : (
                <h1 className="font-heading text-xl font-extrabold tracking-tight">
                  {siteName}
                </h1>
              )}
            </a>
            <div className="flex flex-1 max-w-xl mx-auto relative">
              <div className="flex w-full border border-border rounded-md overflow-hidden">
                <input
                  type="text"
                  placeholder="I'm shopping for ..."
                  value={searchQuery}
                  onChange={(e) => handleSearch(e.target.value)}
                  onFocus={() => searchQuery.trim() && setShowResults(true)}
                  onBlur={() => setTimeout(() => setShowResults(false), 200)}
                  className="flex-1 px-4 py-2.5 text-sm bg-background text-foreground outline-none"
                />
                <button className="btn-primary-shop rounded-none flex items-center gap-2">
                  SEARCH <Search className="h-4 w-4" />
                </button>
              </div>
              {showResults && searchResults.length > 0 && (
                <div className="absolute top-full left-0 right-0 mt-1 bg-card border border-border rounded-lg shadow-xl z-50 max-h-80 overflow-y-auto">
                  {searchResults.map((product) => (
                    <button
                      key={product.id}
                      onMouseDown={() => handleProductClick(product.id)}
                      className="flex items-center gap-3 w-full px-4 py-3 hover:bg-secondary transition-colors text-left"
                    >
                      <img src={product.image} alt={product.name} className="w-10 h-10 object-contain rounded" />
                      <div className="flex-1 min-w-0">
                        <p className="text-sm font-medium text-foreground truncate">{product.name}</p>
                        <p className="text-xs text-muted-foreground">{product.category}</p>
                      </div>
                      <span className="text-sm font-bold text-primary flex-shrink-0">{product.price}</span>
                    </button>
                  ))}
                </div>
              )}
              {showResults && searchQuery.trim() && searchResults.length === 0 && (
                <div className="absolute top-full left-0 right-0 mt-1 bg-card border border-border rounded-lg shadow-xl z-50 p-4 text-center">
                  <p className="text-sm text-muted-foreground">No products found for "{searchQuery}"</p>
                </div>
              )}
            </div>
            <div className="flex items-center gap-2 ml-auto">
              <Button variant="ghost" size="icon" className="relative" onClick={() => setIsCartOpen(true)}>
                <ShoppingCart className="h-5 w-5" />
                {totalItems > 0 && (
                  <span className="absolute -top-1 -right-1 h-4 w-4 rounded-full bg-primary text-[10px] font-bold text-primary-foreground flex items-center justify-center">
                    {totalItems}
                  </span>
                )}
              </Button>
              {user ? (
                <div className="flex items-center gap-2">
                  {isReseller && (
                    <Button variant="outline" size="sm" className="gap-1.5 text-xs" onClick={() => navigate("/reseller")}>
                      <BarChart3 className="h-3.5 w-3.5" /> 🏪 Reseller
                    </Button>
                  )}
                  <button onClick={() => navigate("/profile")} className="flex items-center gap-2">
                    <div className="w-9 h-9 rounded-full border-2 border-primary/30 hover:border-primary transition-colors flex items-center justify-center">
                      {avatarUrl ? (
                        <Avatar className="h-7 w-7">
                          <AvatarImage src={avatarUrl} alt="Profile" />
                          <AvatarFallback className="bg-primary/10 text-primary text-xs"><User className="h-4 w-4" /></AvatarFallback>
                        </Avatar>
                      ) : (
                        <User className="h-4.5 w-4.5 text-muted-foreground" />
                      )}
                    </div>
                    <span className="text-sm text-muted-foreground hidden lg:inline">
                      {user.user_metadata?.name || user.email?.split("@")[0]}
                    </span>
                  </button>
                  <Button variant="ghost" size="icon" onClick={async () => { await signOut(); navigate("/"); }}>
                    <LogOut className="h-5 w-5" />
                  </Button>
                </div>
              ) : (
                <Button variant="default" size="sm" onClick={() => navigate("/auth")}>
                  Login
                </Button>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Navigation Bar */}
      <div className="border-t border-border bg-background hidden md:block">
        <div className="container mx-auto px-4 flex items-center">
          <nav className="flex items-center gap-1">
            {navLinks.map((link) => (
              <button
                key={link.label}
                onClick={() => handleNavClick(link.href)}
                className="px-4 py-3 text-sm font-semibold text-foreground hover:text-primary transition-colors"
              >
                {link.label}
              </button>
            ))}
          </nav>
        </div>
      </div>

    </header>
    <div style={{ height: headerHeight }} />
    {sidebarPortal}
    </>
  );
};

export default Header;
