import { useEffect, useRef, useState } from "react";
import { supabase } from "@/integrations/supabase/client";

interface Partner {
  id: string;
  name: string;
  logo_url: string | null;
  sort_order: number;
}

const PartnersSection = () => {
  const scrollRef = useRef<HTMLDivElement>(null);
  const [partners, setPartners] = useState<Partner[]>([]);

  useEffect(() => {
    const fetch = async () => {
      const { data } = await supabase
        .from("partners")
        .select("*")
        .eq("is_active", true)
        .order("sort_order");
      if (data) setPartners(data);
    };
    fetch();
  }, []);

  useEffect(() => {
    const el = scrollRef.current;
    if (!el || partners.length === 0) return;
    let frame: number;
    let pos = 0;
    const speed = 0.5;
    const animate = () => {
      pos += speed;
      if (pos >= el.scrollWidth / 2) pos = 0;
      el.scrollLeft = pos;
      frame = requestAnimationFrame(animate);
    };
    frame = requestAnimationFrame(animate);
    return () => cancelAnimationFrame(frame);
  }, [partners]);

  if (partners.length === 0) return null;

  return (
    <section className="bg-background py-10 border-t border-border">
      <div className="container mx-auto px-4 text-center">
        <h3 className="font-heading text-lg md:text-xl font-bold text-foreground mb-6">We Partner of</h3>
        <div ref={scrollRef} className="flex gap-8 overflow-hidden whitespace-nowrap">
          {[...partners, ...partners].map((p, i) => (
            <div
              key={`${p.id}-${i}`}
              className="flex-shrink-0 px-6 py-3 rounded-lg border border-border bg-card flex items-center justify-center min-w-[140px] h-[60px]"
            >
              {p.logo_url ? (
                <img src={p.logo_url} alt={p.name} className="max-h-[40px] max-w-[120px] object-contain" />
              ) : (
                <span className="font-heading font-bold text-sm text-foreground/70">{p.name}</span>
              )}
            </div>
          ))}
        </div>
      </div>
    </section>
  );
};

export default PartnersSection;
