import { useState, useEffect, forwardRef } from "react";
import { X } from "lucide-react";
import { supabase } from "@/integrations/supabase/client";
import { motion, AnimatePresence } from "framer-motion";

interface PopupData {
  id: string;
  title: string;
  description: string;
  image_url: string;
  button_text: string;
  button_link: string;
}

const PopupModal = () => {
  const [popup, setPopup] = useState<PopupData | null>(null);
  const [visible, setVisible] = useState(false);

  useEffect(() => {
    const dismissed = sessionStorage.getItem("popup_dismissed");
    if (dismissed) return;

    const fetchPopup = async () => {
      const { data } = await supabase
        .from("popups")
        .select("*")
        .eq("is_active", true)
        .limit(1)
        .single();

      if (data) {
        setPopup(data as PopupData);
        setTimeout(() => setVisible(true), 800);
      }
    };
    fetchPopup();
  }, []);

  const handleClose = () => {
    setVisible(false);
    sessionStorage.setItem("popup_dismissed", "true");
  };

  return (
    <AnimatePresence>
      {visible && popup && (
        <motion.div
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
          className="fixed inset-0 z-[9999] flex items-center justify-center p-4 bg-black/60 backdrop-blur-sm"
          onClick={handleClose}
        >
          <motion.div
            initial={{ scale: 0.8, opacity: 0, y: 40 }}
            animate={{ scale: 1, opacity: 1, y: 0 }}
            exit={{ scale: 0.8, opacity: 0, y: 40 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
            className="relative bg-background rounded-2xl shadow-2xl max-w-md w-full overflow-hidden"
            onClick={(e) => e.stopPropagation()}
          >
            {/* Close button */}
            <button
              onClick={handleClose}
              className="absolute top-3 right-3 z-10 p-1.5 rounded-full bg-black/40 hover:bg-black/60 text-white transition-colors"
            >
              <X className="h-5 w-5" />
            </button>

            {/* Image */}
            {popup.image_url && (
              <div className="w-full">
                <img
                  src={popup.image_url}
                  alt={popup.title}
                  className="w-full h-auto object-cover"
                />
              </div>
            )}

            {/* Content */}
            <div className="p-6 text-center">
              {popup.title && (
                <h2 className="text-xl font-bold text-foreground mb-3">
                  {popup.title}
                </h2>
              )}
              {popup.description && (
                <p className="text-muted-foreground text-sm leading-relaxed mb-5">
                  {popup.description}
                </p>
              )}
              {popup.button_text && popup.button_link && (
                <a
                  href={popup.button_link}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="block w-full py-3 rounded-xl font-bold text-primary-foreground text-base transition-all hover:opacity-90 bg-primary"
                >
                  {popup.button_text}
                </a>
              )}
            </div>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
};

export default PopupModal;
