import { Phone, Mail, MessageCircle, MapPin, Send } from "lucide-react";
import { useSiteSettings } from "@/hooks/useSiteSettings";
import { useState } from "react";
import { useToast } from "@/hooks/use-toast";
import { supabase } from "@/integrations/supabase/client";

const quickLinks = ["Delivery Information", "Privacy Policy", "Terms & Condition", "Search Terms", "Order & Return"];

const paymentMethods = [
  { name: "bKash", logo: "/payments/bkash.svg" },
  { name: "Nagad", logo: "/payments/nagad.png" },
  { name: "Visa", logo: "/payments/visa.svg" },
  { name: "Mastercard", logo: "/payments/mastercard.svg" },
  { name: "DBBL", logo: "/payments/dbbl.png" },
  { name: "SSL", logo: "/payments/ssl.png" },
];

const ShopFooter = () => {
  const { settings } = useSiteSettings();
  const [email, setEmail] = useState("");
  const { toast } = useToast();

  const handleSubscribe = async () => {
    if (!email.trim() || !email.includes("@")) {
      toast({ title: "সঠিক ইমেইল দিন", variant: "destructive" });
      return;
    }
    const { error } = await supabase.from("subscribers").insert({ email: email.trim().toLowerCase() });
    if (error) {
      if (error.code === "23505") {
        toast({ title: "এই ইমেইল আগে থেকেই সাবস্ক্রাইব করা আছে", variant: "destructive" });
      } else {
        toast({ title: "কিছু ভুল হয়েছে, আবার চেষ্টা করুন", variant: "destructive" });
      }
      return;
    }
    toast({ title: "✅ সাবস্ক্রাইব সফল হয়েছে!", description: "ধন্যবাদ! আমরা আপনাকে আপডেট পাঠাবো।" });
    setEmail("");
  };

  return (
    <footer className="relative overflow-hidden">
      <div className="bg-foreground text-background/80 pt-12 pb-6">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 md:grid-cols-12 gap-8 mb-10">
            {/* Brand + Contact */}
            <div className="md:col-span-4">
              {settings.logo_url ? (
                <div className="flex items-center gap-3 mb-4">
                  <img src={settings.logo_url} alt={settings.site_title} className="h-10 max-w-[180px] object-contain brightness-0 invert" />
                  <div className="flex gap-2 ml-2">
                    {settings.facebook_url && (
                      <a href={settings.facebook_url} target="_blank" rel="noopener noreferrer" className="w-8 h-8 rounded-full bg-background/10 flex items-center justify-center hover:bg-primary transition-all">
                        <svg className="h-3.5 w-3.5" fill="currentColor" viewBox="0 0 24 24"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>
                      </a>
                    )}
                    {settings.whatsapp_number && (
                      <a href={`https://wa.me/${settings.whatsapp_number.replace(/[^0-9]/g, "")}`} target="_blank" rel="noopener noreferrer" className="w-8 h-8 rounded-full bg-background/10 flex items-center justify-center hover:bg-primary transition-all">
                        <MessageCircle className="h-3.5 w-3.5" />
                      </a>
                    )}
                    {settings.email && (
                      <a href={`mailto:${settings.email}`} className="w-8 h-8 rounded-full bg-background/10 flex items-center justify-center hover:bg-primary transition-all">
                        <Mail className="h-3.5 w-3.5" />
                      </a>
                    )}
                  </div>
                </div>
              ) : (
                <h3 className="font-heading text-xl font-extrabold text-background mb-4">
                  <span className="text-primary">T</span>ZSS
                </h3>
              )}
              <div className="space-y-2 text-sm text-background/50">
                {settings.phone && <p className="flex items-center gap-2"><Phone className="h-3.5 w-3.5 text-primary" />{settings.phone}</p>}
                {settings.facebook_url && <p className="flex items-center gap-2"><svg className="h-3.5 w-3.5 text-primary" fill="currentColor" viewBox="0 0 24 24"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>Facebook Page</p>}
                {settings.email && <p className="flex items-center gap-2"><Mail className="h-3.5 w-3.5 text-primary" />{settings.email}</p>}
                {settings.address && <p className="flex items-center gap-2"><MapPin className="h-3.5 w-3.5 text-primary" />{settings.address}</p>}
              </div>
            </div>

            {/* Quick Links */}
            <div className="md:col-span-3">
              <ul className="space-y-2.5 mt-2">
                {quickLinks.map((link) => (
                  <li key={link}>
                    <a href="#" className="text-sm text-background/40 hover:text-primary hover:pl-1 transition-all duration-200">{link}</a>
                  </li>
                ))}
              </ul>
            </div>

            {/* Newsletter + Payment */}
            <div className="md:col-span-5">
              <div className="flex gap-0 mb-5">
                <input
                  type="email"
                  placeholder="Enter your email here"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  className="flex-1 px-4 py-2.5 bg-background/10 border border-background/20 rounded-l-md text-sm text-background placeholder:text-background/30 focus:outline-none focus:border-primary"
                />
                <button onClick={handleSubscribe} className="px-5 py-2.5 bg-primary text-primary-foreground font-semibold text-sm rounded-r-md hover:opacity-90 transition-opacity flex items-center gap-1.5">
                  <Send className="h-3.5 w-3.5" /> SUBSCRIBE
                </button>
              </div>
              
              {/* Payment Methods Banner */}
              <div className="rounded-lg overflow-hidden">
                <img src={settings.payment_banner_url || "/payments/payment-banner.png"} alt="Payment Methods - bKash, Nagad, Visa, Mastercard, DBBL, SSL and more" className="w-full h-auto object-contain" />
              </div>
            </div>
          </div>

          <div className="border-t border-background/10 pt-4 text-center space-y-1">
            <p className="text-xs text-background/30">
              © 2023 – 2026 <span className="text-primary">{settings.site_title}</span> All Right Reserved.
            </p>
            {settings.developer_name && (
              <p className="text-xs text-background/30">
                Developed by {settings.developer_url ? (
                  <a href={settings.developer_url} target="_blank" rel="noopener noreferrer" className="text-primary hover:underline">{settings.developer_name}</a>
                ) : (
                  <span className="text-primary">{settings.developer_name}</span>
                )}
              </p>
            )}
          </div>
        </div>
      </div>
    </footer>
  );
};

export default ShopFooter;
