import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { SectionCard } from "@/components/admin/AdminShared";
import { Save, Loader2, Shield, Mail, Lock, Globe } from "lucide-react";

const AdminAuthSettingsPanel = () => {
  const { toast } = useToast();
  const [saving, setSaving] = useState(false);
  const [loading, setLoading] = useState(true);

  // Login methods
  const [emailLoginEnabled, setEmailLoginEnabled] = useState(true);
  const [googleLoginEnabled, setGoogleLoginEnabled] = useState(true);
  const [requireEmailVerification, setRequireEmailVerification] = useState(true);

  // Registration
  const [registrationEnabled, setRegistrationEnabled] = useState(true);
  const [phoneRequired, setPhoneRequired] = useState(true);
  const [minPasswordLength, setMinPasswordLength] = useState("6");

  // Session
  const [sessionNote, setSessionNote] = useState("");

  // Maintenance
  const [maintenanceMode, setMaintenanceMode] = useState(false);
  const [maintenanceMessage, setMaintenanceMessage] = useState("সাইটটি রক্ষণাবেক্ষণের জন্য সাময়িকভাবে বন্ধ আছে।");

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const { data } = await supabase.from("site_settings").select("key, value");
      if (data) {
        const map: Record<string, string> = {};
        data.forEach(r => { if (r.value) map[r.key] = r.value; });
        if (map.auth_email_login !== undefined) setEmailLoginEnabled(map.auth_email_login === "true");
        if (map.auth_google_login !== undefined) setGoogleLoginEnabled(map.auth_google_login === "true");
        if (map.auth_require_email_verification !== undefined) setRequireEmailVerification(map.auth_require_email_verification === "true");
        if (map.auth_registration_enabled !== undefined) setRegistrationEnabled(map.auth_registration_enabled === "true");
        if (map.auth_phone_required !== undefined) setPhoneRequired(map.auth_phone_required === "true");
        if (map.auth_min_password_length) setMinPasswordLength(map.auth_min_password_length);
        if (map.auth_session_note) setSessionNote(map.auth_session_note);
        if (map.auth_maintenance_mode !== undefined) setMaintenanceMode(map.auth_maintenance_mode === "true");
        if (map.auth_maintenance_message) setMaintenanceMessage(map.auth_maintenance_message);
      }
    } catch { } finally { setLoading(false); }
  };

  const upsert = async (key: string, value: string) => {
    const { data: existing } = await supabase.from("site_settings").select("id").eq("key", key).maybeSingle();
    if (existing) await supabase.from("site_settings").update({ value, updated_at: new Date().toISOString() }).eq("key", key);
    else await supabase.from("site_settings").insert({ key, value });
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      await Promise.all([
        upsert("auth_email_login", String(emailLoginEnabled)),
        upsert("auth_google_login", String(googleLoginEnabled)),
        upsert("auth_require_email_verification", String(requireEmailVerification)),
        upsert("auth_registration_enabled", String(registrationEnabled)),
        upsert("auth_phone_required", String(phoneRequired)),
        upsert("auth_min_password_length", minPasswordLength),
        upsert("auth_session_note", sessionNote),
        upsert("auth_maintenance_mode", String(maintenanceMode)),
        upsert("auth_maintenance_message", maintenanceMessage),
      ]);
      toast({ title: "✅ Authentication সেটিংস সেভ হয়েছে!" });
    } catch {
      toast({ title: "সেভ করতে সমস্যা হয়েছে", variant: "destructive" });
    } finally { setSaving(false); }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center py-20">
        <Loader2 className="h-6 w-6 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="space-y-6 max-w-3xl">
      {/* Login Methods */}
      <SectionCard title="লগইন মেথড" description="কোন কোন মাধ্যমে ইউজাররা লগইন করতে পারবে সেটি নিয়ন্ত্রণ করুন" icon={Shield}>
        <div className="space-y-5">
          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
            <div className="flex items-center gap-3">
              <div className="h-10 w-10 rounded-xl bg-primary/10 flex items-center justify-center">
                <Mail className="h-5 w-5 text-primary" />
              </div>
              <div>
                <p className="font-medium text-sm">ইমেইল/পাসওয়ার্ড লগইন</p>
                <p className="text-xs text-muted-foreground">ইমেইল ও পাসওয়ার্ড দিয়ে লগইন</p>
              </div>
            </div>
            <Switch checked={emailLoginEnabled} onCheckedChange={setEmailLoginEnabled} />
          </div>

          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
            <div className="flex items-center gap-3">
              <div className="h-10 w-10 rounded-xl bg-red-500/10 flex items-center justify-center">
                <svg className="h-5 w-5" viewBox="0 0 24 24">
                  <path d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92a5.06 5.06 0 0 1-2.2 3.32v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.1z" fill="#4285F4"/>
                  <path d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z" fill="#34A853"/>
                  <path d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z" fill="#FBBC05"/>
                  <path d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z" fill="#EA4335"/>
                </svg>
              </div>
              <div>
                <p className="font-medium text-sm">Google লগইন</p>
                <p className="text-xs text-muted-foreground">Google অ্যাকাউন্ট দিয়ে লগইন</p>
              </div>
            </div>
            <Switch checked={googleLoginEnabled} onCheckedChange={setGoogleLoginEnabled} />
          </div>
        </div>
      </SectionCard>

      {/* Registration Settings */}
      <SectionCard title="রেজিস্ট্রেশন সেটিংস" description="নতুন অ্যাকাউন্ট তৈরির নিয়মাবলী কনফিগার করুন" icon={Globe}>
        <div className="space-y-5">
          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
            <div>
              <p className="font-medium text-sm">নতুন রেজিস্ট্রেশন অনুমতি</p>
              <p className="text-xs text-muted-foreground">বন্ধ করলে নতুন ইউজার সাইনআপ করতে পারবে না</p>
            </div>
            <Switch checked={registrationEnabled} onCheckedChange={setRegistrationEnabled} />
          </div>

          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
            <div>
              <p className="font-medium text-sm">ইমেইল ভেরিফিকেশন আবশ্যক</p>
              <p className="text-xs text-muted-foreground">সাইনআপের পর ইমেইল ভেরিফাই করতে হবে</p>
            </div>
            <Switch checked={requireEmailVerification} onCheckedChange={setRequireEmailVerification} />
          </div>

          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
            <div>
              <p className="font-medium text-sm">ফোন নম্বর আবশ্যক</p>
              <p className="text-xs text-muted-foreground">সাইনআপে ফোন নম্বর বাধ্যতামূলক</p>
            </div>
            <Switch checked={phoneRequired} onCheckedChange={setPhoneRequired} />
          </div>

          <div className="space-y-1.5">
            <Label className="text-xs font-medium flex items-center gap-1.5">
              <Lock className="h-3 w-3" /> মিনিমাম পাসওয়ার্ড দৈর্ঘ্য
            </Label>
            <Input
              type="number"
              min="4"
              max="32"
              value={minPasswordLength}
              onChange={e => setMinPasswordLength(e.target.value)}
              className="rounded-xl w-32"
            />
            <p className="text-[11px] text-muted-foreground">পাসওয়ার্ড কমপক্ষে এত ক্যারেক্টারের হতে হবে</p>
          </div>
        </div>
      </SectionCard>

      {/* Maintenance Mode */}
      <SectionCard title="মেইনটেন্যান্স মোড" description="সাইট সাময়িকভাবে বন্ধ রাখতে চাইলে এনাবল করুন" icon={Shield}>
        <div className="space-y-4">
          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
            <div>
              <p className="font-medium text-sm">মেইনটেন্যান্স মোড</p>
              <p className="text-xs text-muted-foreground">চালু করলে শুধু এডমিনরাই সাইটে ঢুকতে পারবে</p>
            </div>
            <Switch checked={maintenanceMode} onCheckedChange={setMaintenanceMode} />
          </div>
          {maintenanceMode && (
            <div className="space-y-1.5">
              <Label className="text-xs font-medium">মেইনটেন্যান্স মেসেজ</Label>
              <Input
                value={maintenanceMessage}
                onChange={e => setMaintenanceMessage(e.target.value)}
                placeholder="সাইটটি রক্ষণাবেক্ষণের জন্য বন্ধ আছে..."
                className="rounded-xl"
              />
            </div>
          )}
        </div>
      </SectionCard>

      {/* Session Note */}
      <SectionCard title="সেশন নোট" description="লগইন পেজে দেখানোর জন্য নোট/বার্তা (ঐচ্ছিক)" icon={Mail}>
        <div className="space-y-1.5">
          <Label className="text-xs font-medium">লগইন পেজ নোটিস</Label>
          <Input
            value={sessionNote}
            onChange={e => setSessionNote(e.target.value)}
            placeholder="যেমন: নতুন ফিচার যোগ হয়েছে! লগইন করে দেখুন।"
            className="rounded-xl"
          />
          <p className="text-[11px] text-muted-foreground">এই বার্তাটি লগইন/সাইনআপ পেজে দেখানো হবে</p>
        </div>
      </SectionCard>

      {/* Save Button */}
      <div className="sticky bottom-4 z-10">
        <Button onClick={handleSave} disabled={saving} className="gap-1.5 rounded-xl shadow-lg shadow-primary/20 w-full sm:w-auto">
          {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
          {saving ? "সেভ হচ্ছে..." : "সেটিংস সেভ করুন"}
        </Button>
      </div>
    </div>
  );
};

export default AdminAuthSettingsPanel;
