import { useState, useEffect, useRef } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { SectionCard, EmptyState } from "./AdminShared";
import { Plus, Edit2, Trash2, Save, Loader2, Upload, FileText, XCircle } from "lucide-react";
import { AnimatePresence, motion } from "framer-motion";

const AdminBlogPanel = ({ fetchData: refetch }: { fetchData: () => void }) => {
  const [posts, setPosts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showForm, setShowForm] = useState(false);
  const [editing, setEditing] = useState<any>(null);
  const [form, setForm] = useState({ title: "", slug: "", excerpt: "", content: "", image_url: "", is_published: false });
  const [saving, setSaving] = useState(false);
  const [uploadingImage, setUploadingImage] = useState(false);
  const imageRef = useRef<HTMLInputElement>(null);
  const { toast: blogToast } = useToast();

  const fetchPosts = async () => {
    setLoading(true);
    const { data } = await supabase.from("blog_posts").select("*").order("created_at", { ascending: false });
    if (data) setPosts(data);
    setLoading(false);
  };

  useEffect(() => { fetchPosts(); }, []);

  const handleImageUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    setUploadingImage(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `blog-${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("products").upload(fileName, file, { upsert: true });
      if (error) { blogToast({ title: "আপলোড এরর", variant: "destructive" }); return; }
      const { data: urlData } = supabase.storage.from("products").getPublicUrl(fileName);
      setForm(f => ({ ...f, image_url: urlData.publicUrl }));
    } finally { setUploadingImage(false); }
  };

  const generateSlug = (title: string) => title.toLowerCase().replace(/[^a-z0-9\u0980-\u09FF]+/g, "-").replace(/^-|-$/g, "");

  const handleSave = async () => {
    if (!form.title) { blogToast({ title: "টাইটেল দিন", variant: "destructive" }); return; }
    setSaving(true);
    const slug = form.slug || generateSlug(form.title);
    try {
      if (editing) {
        await supabase.from("blog_posts").update({ ...form, slug, updated_at: new Date().toISOString() }).eq("id", editing.id);
        blogToast({ title: "✅ ব্লগ আপডেট হয়েছে!" });
      } else {
        await supabase.from("blog_posts").insert({ ...form, slug });
        blogToast({ title: "✅ ব্লগ পোস্ট যোগ হয়েছে!" });
      }
      setForm({ title: "", slug: "", excerpt: "", content: "", image_url: "", is_published: false });
      setEditing(null); setShowForm(false); fetchPosts();
    } catch { blogToast({ title: "সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setSaving(false); }
  };

  const handleDelete = async (id: string) => {
    await supabase.from("blog_posts").delete().eq("id", id);
    blogToast({ title: "✅ ব্লগ ডিলিট হয়েছে!" });
    fetchPosts();
  };

  const startEdit = (post: any) => {
    setEditing(post);
    setForm({ title: post.title, slug: post.slug, excerpt: post.excerpt || "", content: post.content || "", image_url: post.image_url || "", is_published: post.is_published });
    setShowForm(true);
  };

  return (
    <div className="space-y-6 max-w-4xl">
      <div className="flex items-center justify-between">
        <h2 className="font-heading font-bold text-lg">ব্লগ পোস্ট</h2>
        <Button size="sm" className="rounded-xl gap-1.5" onClick={() => { setEditing(null); setForm({ title: "", slug: "", excerpt: "", content: "", image_url: "", is_published: false }); setShowForm(!showForm); }}>
          <Plus className="h-4 w-4" />{showForm ? "বাতিল" : "নতুন পোস্ট"}
        </Button>
      </div>

      {showForm && (
        <SectionCard title={editing ? "ব্লগ এডিট" : "নতুন ব্লগ পোস্ট"} icon={FileText}>
          <div className="space-y-4">
            <div className="space-y-1.5">
              <Label className="text-xs">টাইটেল *</Label>
              <Input value={form.title} onChange={e => setForm(f => ({ ...f, title: e.target.value, slug: generateSlug(e.target.value) }))} className="rounded-xl" />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Slug</Label>
              <Input value={form.slug} onChange={e => setForm(f => ({ ...f, slug: e.target.value }))} className="rounded-xl font-mono text-sm" />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">সংক্ষেপ (Excerpt)</Label>
              <Input value={form.excerpt} onChange={e => setForm(f => ({ ...f, excerpt: e.target.value }))} className="rounded-xl" />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">কন্টেন্ট</Label>
              <textarea value={form.content} onChange={e => setForm(f => ({ ...f, content: e.target.value }))}
                className="flex w-full rounded-xl border border-input bg-background px-3 py-2 text-sm min-h-[200px] resize-y ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring" />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">ছবি</Label>
              <div className="flex items-center gap-3 flex-wrap">
                {form.image_url && <img src={form.image_url} alt="" className="h-16 w-24 object-cover rounded-lg border border-border" />}
                <input type="file" accept="image/*" ref={imageRef} onChange={handleImageUpload} className="hidden" />
                <Button variant="outline" size="sm" className="rounded-xl gap-1.5" onClick={() => imageRef.current?.click()} disabled={uploadingImage}>
                  {uploadingImage ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                  {uploadingImage ? "আপলোড হচ্ছে..." : "ছবি আপলোড"}
                </Button>
              </div>
            </div>
            <div className="flex items-center gap-2">
              <Switch checked={form.is_published} onCheckedChange={v => setForm(f => ({ ...f, is_published: v }))} />
              <Label className="text-xs">পাবলিশ করুন</Label>
            </div>
            <Button onClick={handleSave} disabled={saving} className="gap-1.5 rounded-xl w-full sm:w-auto">
              {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
              {saving ? "সেভ হচ্ছে..." : editing ? "আপডেট করুন" : "পোস্ট করুন"}
            </Button>
          </div>
        </SectionCard>
      )}

      {loading ? (
        <div className="flex justify-center py-12"><Loader2 className="h-8 w-8 animate-spin text-primary" /></div>
      ) : posts.length === 0 ? (
        <EmptyState icon={FileText} title="কোনো ব্লগ পোস্ট নেই" description="নতুন ব্লগ পোস্ট যোগ করুন।" />
      ) : (
        <div className="space-y-3">
          {posts.map(post => (
            <div key={post.id} className="bg-card border border-border rounded-xl p-3 sm:p-4 flex flex-col sm:flex-row sm:items-start gap-3 sm:gap-4">
              {post.image_url && <img src={post.image_url} alt="" className="h-16 w-full sm:w-24 object-cover rounded-lg flex-shrink-0" />}
              <div className="flex-1 min-w-0">
                <div className="flex items-center gap-2 mb-1 flex-wrap">
                  <h4 className="font-semibold text-sm truncate">{post.title}</h4>
                  <span className={`text-[10px] px-2 py-0.5 rounded-full font-medium ${post.is_published ? "bg-primary/10 text-primary" : "bg-muted text-muted-foreground"}`}>
                    {post.is_published ? "Published" : "Draft"}
                  </span>
                </div>
                {post.excerpt && <p className="text-xs text-muted-foreground line-clamp-1">{post.excerpt}</p>}
                <p className="text-[11px] text-muted-foreground mt-1">{new Date(post.created_at).toLocaleDateString("bn-BD")}</p>
              </div>
              <div className="flex gap-1 flex-shrink-0">
                <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => startEdit(post)}><Edit2 className="h-3.5 w-3.5" /></Button>
                <Button variant="ghost" size="icon" className="h-8 w-8 text-destructive" onClick={() => handleDelete(post.id)}><Trash2 className="h-3.5 w-3.5" /></Button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default AdminBlogPanel;
