import { useState, useEffect, useMemo } from "react";
import { supabase } from "@/integrations/supabase/client";
import {
  Trash2, RefreshCw, Loader2, ShoppingCart, Clock, User, Phone, Mail,
  ArrowRightCircle, CheckSquare, Square, Download, TrendingUp, BarChart3,
  AlertCircle, CheckCircle2, XCircle, Filter
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { useToast } from "@/hooks/use-toast";
import { formatDistanceToNow, format, subDays, startOfDay, isAfter } from "date-fns";
import { bn } from "date-fns/locale";
import { motion, AnimatePresence } from "framer-motion";
import { BarChart, Bar, XAxis, YAxis, Tooltip, ResponsiveContainer, PieChart, Pie, Cell, AreaChart, Area } from "recharts";

interface IncompleteOrder {
  id: string;
  session_id: string;
  name: string;
  email: string;
  phone: string;
  cart_items: any[];
  cart_total: number;
  created_at: string;
  updated_at: string;
  is_converted: boolean;
}

type ViewMode = "analytics" | "list";
type FilterMode = "all" | "with_contact" | "no_contact";

const AdminIncompleteOrdersPanel = () => {
  const [allItems, setAllItems] = useState<IncompleteOrder[]>([]);
  const [loading, setLoading] = useState(true);
  const [deleting, setDeleting] = useState<string | null>(null);
  const [converting, setConverting] = useState<string | null>(null);
  const [selectedIds, setSelectedIds] = useState<Set<string>>(new Set());
  const [bulkLoading, setBulkLoading] = useState(false);
  const [viewMode, setViewMode] = useState<ViewMode>("analytics");
  const [filterMode, setFilterMode] = useState<FilterMode>("all");
  const [confirmDialog, setConfirmDialog] = useState<{
    type: "convert" | "delete" | "bulk_delete" | "bulk_convert" | "clear_all";
    id?: string;
    title: string;
    message: string;
  } | null>(null);
  const { toast } = useToast();

  // Separate active (not converted) and converted items
  const activeItems = useMemo(() => allItems.filter(i => !i.is_converted), [allItems]);
  const convertedItems = useMemo(() => allItems.filter(i => i.is_converted), [allItems]);

  // Filtered active items
  const filteredItems = useMemo(() => {
    if (filterMode === "with_contact") return activeItems.filter(i => i.name || i.email || i.phone);
    if (filterMode === "no_contact") return activeItems.filter(i => !i.name && !i.email && !i.phone);
    return activeItems;
  }, [activeItems, filterMode]);

  // ═══ Analytics Data ═══
  const analytics = useMemo(() => {
    const totalIncomplete = activeItems.length;
    const totalConverted = convertedItems.length;
    const totalAll = totalIncomplete + totalConverted;
    const conversionRate = totalAll > 0 ? Math.round((totalConverted / totalAll) * 100) : 0;
    const totalLostValue = activeItems.reduce((s, i) => s + (i.cart_total || 0), 0);
    const totalRecoveredValue = convertedItems.reduce((s, i) => s + (i.cart_total || 0), 0);

    // Daily trend (last 7 days)
    const dailyTrend: { date: string; incomplete: number; converted: number; recovered: number }[] = [];
    for (let i = 6; i >= 0; i--) {
      const day = startOfDay(subDays(new Date(), i));
      const nextDay = startOfDay(subDays(new Date(), i - 1));
      const dayLabel = format(day, "dd MMM", { locale: bn });
      const incompleteCount = allItems.filter(item => {
        const d = new Date(item.created_at);
        return isAfter(d, day) && !isAfter(d, nextDay) && !item.is_converted;
      }).length;
      const convertedCount = allItems.filter(item => {
        const d = new Date(item.updated_at);
        return isAfter(d, day) && !isAfter(d, nextDay) && item.is_converted;
      }).length;
      const recoveredAmt = allItems
        .filter(item => {
          const d = new Date(item.updated_at);
          return isAfter(d, day) && !isAfter(d, nextDay) && item.is_converted;
        })
        .reduce((s, i) => s + (i.cart_total || 0), 0);
      dailyTrend.push({ date: dayLabel, incomplete: incompleteCount, converted: convertedCount, recovered: recoveredAmt });
    }

    // Funnel data
    const withCart = activeItems.filter(i => i.cart_items && i.cart_items.length > 0).length;
    const withContact = activeItems.filter(i => i.name || i.email || i.phone).length;
    const funnel = [
      { stage: "চেকআউট শুরু", count: totalAll, fill: "hsl(var(--primary))" },
      { stage: "কার্ট আছে", count: withCart + totalConverted, fill: "hsl(var(--primary) / 0.7)" },
      { stage: "তথ্য দিয়েছে", count: withContact + totalConverted, fill: "hsl(var(--primary) / 0.5)" },
      { stage: "অর্ডার সম্পন্ন", count: totalConverted, fill: "hsl(var(--primary) / 0.3)" },
    ];

    return { totalIncomplete, totalConverted, conversionRate, totalLostValue, totalRecoveredValue, dailyTrend, funnel };
  }, [allItems, activeItems, convertedItems]);

  const fetchData = async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("incomplete_orders")
      .select("*")
      .order("updated_at", { ascending: false });

    if (!error && data) {
      setAllItems(data as unknown as IncompleteOrder[]);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
    const channel = supabase
      .channel("incomplete-orders-realtime")
      .on("postgres_changes", { event: "*", schema: "public", table: "incomplete_orders" }, () => fetchData())
      .subscribe();
    return () => { supabase.removeChannel(channel); };
  }, []);

  // ═══ Convert to Order ═══
  const handleConvert = async (item: IncompleteOrder) => {
    setConverting(item.id);
    try {
      const productName = item.cart_items?.map((ci: any) => `${ci.name}${ci.option ? ` (${ci.option})` : ""} x${ci.quantity}`).join(", ") || "N/A";
      const { error } = await supabase.from("orders").insert({
        customer_name: item.name || "Unknown",
        customer_phone: item.phone || null,
        product_name: productName,
        amount: `৳${item.cart_total || 0}`,
        status: "pending",
        payment_status: "unpaid",
        payment_method: "manual",
      });
      if (error) throw error;

      await supabase.from("incomplete_orders").update({ is_converted: true } as any).eq("id", item.id);
      toast({ title: "✅ অর্ডারে রূপান্তর হয়েছে!" });
      fetchData();
    } catch {
      toast({ title: "রূপান্তর ব্যর্থ হয়েছে", variant: "destructive" });
    }
    setConverting(null);
    setConfirmDialog(null);
  };

  const handleDelete = async (id: string) => {
    setDeleting(id);
    await supabase.from("incomplete_orders").delete().eq("id", id);
    setAllItems(prev => prev.filter(i => i.id !== id));
    setSelectedIds(prev => { const n = new Set(prev); n.delete(id); return n; });
    setDeleting(null);
    setConfirmDialog(null);
    toast({ title: "ডিলিট করা হয়েছে" });
  };

  // ═══ Bulk Actions ═══
  const handleBulkDelete = async () => {
    setBulkLoading(true);
    const ids = Array.from(selectedIds);
    for (const id of ids) {
      await supabase.from("incomplete_orders").delete().eq("id", id);
    }
    setAllItems(prev => prev.filter(i => !selectedIds.has(i.id)));
    setSelectedIds(new Set());
    setBulkLoading(false);
    setConfirmDialog(null);
    toast({ title: `${ids.length} টি ডিলিট হয়েছে` });
  };

  const handleBulkConvert = async () => {
    setBulkLoading(true);
    const ids = Array.from(selectedIds);
    const toConvert = filteredItems.filter(i => ids.includes(i.id));
    for (const item of toConvert) {
      const productName = item.cart_items?.map((ci: any) => `${ci.name}${ci.option ? ` (${ci.option})` : ""} x${ci.quantity}`).join(", ") || "N/A";
      await supabase.from("orders").insert({
        customer_name: item.name || "Unknown",
        customer_phone: item.phone || null,
        product_name: productName,
        amount: `৳${item.cart_total || 0}`,
        status: "pending",
        payment_status: "unpaid",
        payment_method: "manual",
      });
      await supabase.from("incomplete_orders").update({ is_converted: true } as any).eq("id", item.id);
    }
    setSelectedIds(new Set());
    setBulkLoading(false);
    setConfirmDialog(null);
    fetchData();
    toast({ title: `${toConvert.length} টি অর্ডারে রূপান্তর হয়েছে!` });
  };

  const handleClearAll = async () => {
    setLoading(true);
    await supabase.from("incomplete_orders").delete().eq("is_converted", false);
    setAllItems(prev => prev.filter(i => i.is_converted));
    setSelectedIds(new Set());
    setLoading(false);
    setConfirmDialog(null);
    toast({ title: "সব ইনকম্পলিট অর্ডার মুছে ফেলা হয়েছে" });
  };

  // ═══ CSV Export ═══
  const exportCSV = () => {
    const headers = ["Name", "Email", "Phone", "Cart Items", "Total (৳)", "Created At", "Status"];
    const rows = filteredItems.map(i => [
      i.name || "",
      i.email || "",
      i.phone || "",
      (i.cart_items || []).map((ci: any) => `${ci.name} x${ci.quantity}`).join("; "),
      i.cart_total || 0,
      format(new Date(i.created_at), "yyyy-MM-dd HH:mm"),
      i.is_converted ? "Converted" : "Incomplete",
    ]);
    const csv = [headers.join(","), ...rows.map(r => r.map(v => `"${v}"`).join(","))].join("\n");
    const blob = new Blob(["\uFEFF" + csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `incomplete-orders-${format(new Date(), "yyyy-MM-dd")}.csv`;
    a.click();
    URL.revokeObjectURL(url);
    toast({ title: "CSV ডাউনলোড হয়েছে" });
  };

  // Selection helpers
  const toggleSelect = (id: string) => {
    setSelectedIds(prev => {
      const n = new Set(prev);
      n.has(id) ? n.delete(id) : n.add(id);
      return n;
    });
  };
  const toggleSelectAll = () => {
    if (selectedIds.size === filteredItems.length) setSelectedIds(new Set());
    else setSelectedIds(new Set(filteredItems.map(i => i.id)));
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center py-20">
        <Loader2 className="h-6 w-6 animate-spin text-primary" />
      </div>
    );
  }

  const CHART_COLORS = ["hsl(var(--primary))", "hsl(var(--chart-2))", "hsl(var(--chart-3))", "hsl(var(--chart-4))"];

  return (
    <div className="space-y-5">
      {/* ═══ Confirmation Dialog ═══ */}
      <AnimatePresence>
        {confirmDialog && (
          <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
            className="fixed inset-0 z-[100] flex items-center justify-center bg-black/50 backdrop-blur-sm p-4">
            <motion.div initial={{ scale: 0.9, opacity: 0 }} animate={{ scale: 1, opacity: 1 }} exit={{ scale: 0.9, opacity: 0 }}
              className="bg-card rounded-2xl border border-border shadow-2xl p-6 max-w-sm w-full">
              <div className="flex items-center gap-3 mb-4">
                <div className={`h-10 w-10 rounded-full flex items-center justify-center ${confirmDialog.type.includes("delete") || confirmDialog.type === "clear_all" ? "bg-destructive/10" : "bg-primary/10"}`}>
                  {confirmDialog.type.includes("delete") || confirmDialog.type === "clear_all"
                    ? <AlertCircle className="h-5 w-5 text-destructive" />
                    : <ArrowRightCircle className="h-5 w-5 text-primary" />}
                </div>
                <div>
                  <h3 className="font-heading font-semibold">{confirmDialog.title}</h3>
                  <p className="text-sm text-muted-foreground">{confirmDialog.message}</p>
                </div>
              </div>
              <div className="flex gap-2 justify-end">
                <Button variant="outline" size="sm" onClick={() => setConfirmDialog(null)}>বাতিল</Button>
                {confirmDialog.type === "convert" && (
                  <Button size="sm" className="gap-1.5" disabled={converting !== null}
                    onClick={() => {
                      const item = filteredItems.find(i => i.id === confirmDialog.id);
                      if (item) handleConvert(item);
                    }}>
                    {converting ? <Loader2 className="h-3.5 w-3.5 animate-spin" /> : <ArrowRightCircle className="h-3.5 w-3.5" />}
                    রূপান্তর করুন
                  </Button>
                )}
                {confirmDialog.type === "delete" && (
                  <Button variant="destructive" size="sm" onClick={() => handleDelete(confirmDialog.id!)}>ডিলিট করুন</Button>
                )}
                {confirmDialog.type === "bulk_delete" && (
                  <Button variant="destructive" size="sm" disabled={bulkLoading} onClick={handleBulkDelete}>
                    {bulkLoading ? <Loader2 className="h-3.5 w-3.5 animate-spin mr-1" /> : null}
                    {selectedIds.size} টি ডিলিট করুন
                  </Button>
                )}
                {confirmDialog.type === "bulk_convert" && (
                  <Button size="sm" className="gap-1.5" disabled={bulkLoading} onClick={handleBulkConvert}>
                    {bulkLoading ? <Loader2 className="h-3.5 w-3.5 animate-spin" /> : <ArrowRightCircle className="h-3.5 w-3.5" />}
                    {selectedIds.size} টি রূপান্তর করুন
                  </Button>
                )}
                {confirmDialog.type === "clear_all" && (
                  <Button variant="destructive" size="sm" onClick={handleClearAll}>সব মুছুন</Button>
                )}
              </div>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* ═══ View Toggle & Actions ═══ */}
      <div className="flex flex-wrap items-center justify-between gap-3">
        <div className="flex items-center gap-2">
          <div className="bg-muted rounded-xl p-1 flex gap-0.5">
            <button onClick={() => setViewMode("analytics")}
              className={`px-3 py-1.5 text-xs font-medium rounded-lg transition-all ${viewMode === "analytics" ? "bg-card shadow text-foreground" : "text-muted-foreground hover:text-foreground"}`}>
              <BarChart3 className="h-3.5 w-3.5 inline mr-1" />অ্যানালিটিক্স
            </button>
            <button onClick={() => setViewMode("list")}
              className={`px-3 py-1.5 text-xs font-medium rounded-lg transition-all ${viewMode === "list" ? "bg-card shadow text-foreground" : "text-muted-foreground hover:text-foreground"}`}>
              <ShoppingCart className="h-3.5 w-3.5 inline mr-1" />তালিকা ({activeItems.length})
            </button>
          </div>
        </div>
        <div className="flex items-center gap-2">
          <Button variant="outline" size="sm" onClick={fetchData} className="h-8">
            <RefreshCw className="h-3.5 w-3.5 mr-1" />রিফ্রেশ
          </Button>
          <Button variant="outline" size="sm" onClick={exportCSV} className="h-8" disabled={filteredItems.length === 0}>
            <Download className="h-3.5 w-3.5 mr-1" />CSV
          </Button>
          {activeItems.length > 0 && (
            <Button variant="destructive" size="sm" className="h-8"
              onClick={() => setConfirmDialog({ type: "clear_all", title: "সব মুছে ফেলবেন?", message: `${activeItems.length} টি ইনকম্পলিট অর্ডার স্থায়ীভাবে মুছে যাবে।` })}>
              <Trash2 className="h-3.5 w-3.5 mr-1" />সব মুছুন
            </Button>
          )}
        </div>
      </div>

      {/* ═══ Analytics View ═══ */}
      {viewMode === "analytics" && (
        <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} className="space-y-5">
          {/* Stat Cards */}
          <div className="grid grid-cols-2 lg:grid-cols-5 gap-3">
            <StatMini label="ইনকম্পলিট" value={analytics.totalIncomplete} icon={XCircle} color="bg-amber-500/10 text-amber-600" />
            <StatMini label="রিকভার্ড" value={analytics.totalConverted} icon={CheckCircle2} color="bg-emerald-500/10 text-emerald-600" />
            <StatMini label="কনভার্শন রেট" value={`${analytics.conversionRate}%`} icon={TrendingUp} color="bg-primary/10 text-primary" />
            <StatMini label="হারানো মূল্য" value={`৳${analytics.totalLostValue}`} icon={AlertCircle} color="bg-destructive/10 text-destructive" />
            <StatMini label="রিকভার্ড মূল্য" value={`৳${analytics.totalRecoveredValue}`} icon={CheckCircle2} color="bg-emerald-500/10 text-emerald-600" />
          </div>

          {/* Charts Row */}
          <div className="grid md:grid-cols-2 gap-4">
            {/* Daily Trend */}
            <div className="bg-card border border-border rounded-2xl p-4">
              <h4 className="text-sm font-semibold font-heading mb-3">৭ দিনের ট্রেন্ড</h4>
              <div className="h-52">
                <ResponsiveContainer width="100%" height="100%">
                  <AreaChart data={analytics.dailyTrend}>
                    <defs>
                      <linearGradient id="incGrad" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="5%" stopColor="hsl(var(--destructive))" stopOpacity={0.3} />
                        <stop offset="95%" stopColor="hsl(var(--destructive))" stopOpacity={0} />
                      </linearGradient>
                      <linearGradient id="convGrad" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="5%" stopColor="hsl(var(--primary))" stopOpacity={0.3} />
                        <stop offset="95%" stopColor="hsl(var(--primary))" stopOpacity={0} />
                      </linearGradient>
                    </defs>
                    <XAxis dataKey="date" tick={{ fontSize: 11 }} tickLine={false} axisLine={false} />
                    <YAxis tick={{ fontSize: 11 }} tickLine={false} axisLine={false} allowDecimals={false} />
                    <Tooltip contentStyle={{ borderRadius: 12, border: "1px solid hsl(var(--border))", background: "hsl(var(--card))", fontSize: 12 }} />
                    <Area type="monotone" dataKey="incomplete" name="ইনকম্পলিট" stroke="hsl(var(--destructive))" fill="url(#incGrad)" strokeWidth={2} />
                    <Area type="monotone" dataKey="converted" name="রিকভার্ড" stroke="hsl(var(--primary))" fill="url(#convGrad)" strokeWidth={2} />
                  </AreaChart>
                </ResponsiveContainer>
              </div>
            </div>

            {/* Conversion Funnel */}
            <div className="bg-card border border-border rounded-2xl p-4">
              <h4 className="text-sm font-semibold font-heading mb-3">কনভার্শন ফানেল</h4>
              <div className="space-y-3 mt-2">
                {analytics.funnel.map((stage, idx) => {
                  const maxCount = analytics.funnel[0].count || 1;
                  const pct = Math.round((stage.count / maxCount) * 100);
                  return (
                    <div key={idx}>
                      <div className="flex justify-between text-xs mb-1">
                        <span className="text-muted-foreground">{stage.stage}</span>
                        <span className="font-semibold">{stage.count} ({pct}%)</span>
                      </div>
                      <div className="h-6 bg-muted rounded-lg overflow-hidden">
                        <motion.div
                          initial={{ width: 0 }}
                          animate={{ width: `${pct}%` }}
                          transition={{ duration: 0.8, delay: idx * 0.15 }}
                          className="h-full rounded-lg"
                          style={{ background: stage.fill }}
                        />
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>
          </div>

          {/* Recovered Amount Chart */}
          <div className="bg-card border border-border rounded-2xl p-4">
            <h4 className="text-sm font-semibold font-heading mb-3">দৈনিক রিকভার্ড মূল্য (৳)</h4>
            <div className="h-48">
              <ResponsiveContainer width="100%" height="100%">
                <BarChart data={analytics.dailyTrend}>
                  <XAxis dataKey="date" tick={{ fontSize: 11 }} tickLine={false} axisLine={false} />
                  <YAxis tick={{ fontSize: 11 }} tickLine={false} axisLine={false} />
                  <Tooltip contentStyle={{ borderRadius: 12, border: "1px solid hsl(var(--border))", background: "hsl(var(--card))", fontSize: 12 }} />
                  <Bar dataKey="recovered" name="রিকভার্ড (৳)" fill="hsl(var(--primary))" radius={[6, 6, 0, 0]} />
                </BarChart>
              </ResponsiveContainer>
            </div>
          </div>
        </motion.div>
      )}

      {/* ═══ List View ═══ */}
      {viewMode === "list" && (
        <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} className="space-y-4">
          {/* Filter & Bulk Actions */}
          <div className="flex flex-wrap items-center gap-2">
            <div className="bg-muted rounded-xl p-1 flex gap-0.5">
              {([["all", "সব"], ["with_contact", "তথ্য আছে"], ["no_contact", "তথ্য নেই"]] as [FilterMode, string][]).map(([key, label]) => (
                <button key={key} onClick={() => { setFilterMode(key); setSelectedIds(new Set()); }}
                  className={`px-2.5 py-1 text-xs font-medium rounded-lg transition-all ${filterMode === key ? "bg-card shadow text-foreground" : "text-muted-foreground"}`}>
                  {label}
                </button>
              ))}
            </div>

            {filteredItems.length > 0 && (
              <button onClick={toggleSelectAll}
                className="flex items-center gap-1.5 text-xs text-muted-foreground hover:text-foreground transition-colors px-2 py-1">
                {selectedIds.size === filteredItems.length
                  ? <CheckSquare className="h-3.5 w-3.5 text-primary" />
                  : <Square className="h-3.5 w-3.5" />}
                সব সিলেক্ট
              </button>
            )}

            {selectedIds.size > 0 && (
              <div className="flex items-center gap-2 ml-auto">
                <span className="text-xs text-muted-foreground">{selectedIds.size} টি সিলেক্ট</span>
                <Button size="sm" variant="outline" className="h-7 text-xs gap-1"
                  onClick={() => setConfirmDialog({ type: "bulk_convert", title: "বাল্ক রূপান্তর", message: `${selectedIds.size} টি ইনকম্পলিট অর্ডার অর্ডারে রূপান্তর হবে।` })}>
                  <ArrowRightCircle className="h-3 w-3" />রূপান্তর
                </Button>
                <Button size="sm" variant="destructive" className="h-7 text-xs gap-1"
                  onClick={() => setConfirmDialog({ type: "bulk_delete", title: "বাল্ক ডিলিট", message: `${selectedIds.size} টি স্থায়ীভাবে ডিলিট হবে।` })}>
                  <Trash2 className="h-3 w-3" />ডিলিট
                </Button>
              </div>
            )}
          </div>

          {/* Cards */}
          {filteredItems.length === 0 ? (
            <div className="text-center py-16 text-muted-foreground">
              <ShoppingCart className="h-12 w-12 mx-auto mb-3 opacity-30" />
              <p className="text-sm">কোনো ইনকম্পলিট অর্ডার নেই</p>
            </div>
          ) : (
            <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-3">
              {filteredItems.map((item) => {
                const isSelected = selectedIds.has(item.id);
                return (
                  <motion.div
                    key={item.id}
                    layout
                    initial={{ opacity: 0, scale: 0.95 }}
                    animate={{ opacity: 1, scale: 1 }}
                    className={`bg-card border rounded-xl p-4 space-y-3 hover:shadow-md transition-all cursor-pointer ${isSelected ? "border-primary ring-1 ring-primary/20" : "border-border"}`}
                    onClick={() => toggleSelect(item.id)}
                  >
                    {/* Header */}
                    <div className="flex items-start justify-between">
                      <div className="flex items-center gap-2">
                        <div className={`h-5 w-5 rounded flex items-center justify-center ${isSelected ? "bg-primary text-primary-foreground" : "border border-border"}`}>
                          {isSelected && <CheckSquare className="h-3.5 w-3.5" />}
                        </div>
                        <p className="text-xs text-muted-foreground">
                          {formatDistanceToNow(new Date(item.updated_at), { addSuffix: true, locale: bn })}
                        </p>
                      </div>
                      <div className="flex gap-1" onClick={e => e.stopPropagation()}>
                        <Button variant="ghost" size="icon" className="h-7 w-7 text-primary hover:bg-primary/10"
                          title="অর্ডারে রূপান্তর"
                          disabled={converting === item.id}
                          onClick={() => setConfirmDialog({ type: "convert", id: item.id, title: "অর্ডারে রূপান্তর?", message: `"${item.name || "Unknown"}" এর ইনকম্পলিট অর্ডারটি সরাসরি অর্ডারে রূপান্তর হবে।` })}>
                          {converting === item.id
                            ? <Loader2 className="h-3.5 w-3.5 animate-spin" />
                            : <ArrowRightCircle className="h-3.5 w-3.5" />}
                        </Button>
                        <Button variant="ghost" size="icon" className="h-7 w-7 text-muted-foreground hover:text-destructive"
                          disabled={deleting === item.id}
                          onClick={() => setConfirmDialog({ type: "delete", id: item.id, title: "ডিলিট করবেন?", message: "এটি স্থায়ীভাবে মুছে যাবে।" })}>
                          {deleting === item.id
                            ? <Loader2 className="h-3.5 w-3.5 animate-spin" />
                            : <Trash2 className="h-3.5 w-3.5" />}
                        </Button>
                      </div>
                    </div>

                    {/* Customer Info */}
                    <div className="space-y-1.5">
                      {item.name && (
                        <div className="flex items-center gap-2 text-sm">
                          <User className="h-3.5 w-3.5 text-muted-foreground" />
                          <span className="font-medium">{item.name}</span>
                        </div>
                      )}
                      {item.email && (
                        <div className="flex items-center gap-2 text-sm">
                          <Mail className="h-3.5 w-3.5 text-muted-foreground" />
                          <span className="text-muted-foreground truncate">{item.email}</span>
                        </div>
                      )}
                      {item.phone && (
                        <div className="flex items-center gap-2 text-sm">
                          <Phone className="h-3.5 w-3.5 text-muted-foreground" />
                          <span className="text-muted-foreground">{item.phone}</span>
                        </div>
                      )}
                      {!item.name && !item.email && !item.phone && (
                        <p className="text-xs text-muted-foreground italic">কোনো তথ্য দেওয়া হয়নি</p>
                      )}
                    </div>

                    {/* Cart Items */}
                    {item.cart_items && item.cart_items.length > 0 && (
                      <div className="border-t border-border pt-2 space-y-1">
                        {item.cart_items.map((ci: any, idx: number) => (
                          <div key={idx} className="flex justify-between text-xs">
                            <span className="text-muted-foreground truncate flex-1">{ci.name} x{ci.quantity}</span>
                            <span className="font-medium text-primary ml-2">{ci.price}</span>
                          </div>
                        ))}
                        {item.cart_total > 0 && (
                          <div className="flex justify-between text-sm font-semibold pt-1 border-t border-dashed border-border">
                            <span>মোট</span>
                            <span className="text-primary">৳{item.cart_total}</span>
                          </div>
                        )}
                      </div>
                    )}
                  </motion.div>
                );
              })}
            </div>
          )}
        </motion.div>
      )}
    </div>
  );
};

// Mini stat card
const StatMini = ({ label, value, icon: Icon, color }: { label: string; value: string | number; icon: any; color: string }) => (
  <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}
    className="bg-card border border-border rounded-xl p-3 flex items-center gap-3">
    <div className={`h-9 w-9 rounded-lg ${color} flex items-center justify-center shrink-0`}>
      <Icon className="h-4 w-4" />
    </div>
    <div className="min-w-0">
      <p className="text-[11px] text-muted-foreground truncate">{label}</p>
      <p className="text-sm font-bold font-heading truncate">{value}</p>
    </div>
  </motion.div>
);

export default AdminIncompleteOrdersPanel;
