import { useState, useEffect, useRef } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { SectionCard, EmptyState } from "./AdminShared";
import { Plus, Edit2, Trash2, Save, Loader2, Upload, Handshake, XCircle } from "lucide-react";
import { AnimatePresence, motion } from "framer-motion";

interface DBPartner {
  id: string; name: string; logo_url: string | null; sort_order: number; is_active: boolean;
}

const AdminPartnersPanel = ({ fetchData: refetch }: { fetchData: () => void }) => {
  const [partners, setPartners] = useState<DBPartner[]>([]);
  const [loading, setLoading] = useState(true);
  const [showAdd, setShowAdd] = useState(false);
  const [newName, setNewName] = useState("");
  const [newLogoUrl, setNewLogoUrl] = useState("");
  const [newSortOrder, setNewSortOrder] = useState(0);
  const [editingPartner, setEditingPartner] = useState<DBPartner | null>(null);
  const [uploading, setUploading] = useState(false);
  const [editUploading, setEditUploading] = useState(false);
  const fileRef = useRef<HTMLInputElement>(null);
  const editFileRef = useRef<HTMLInputElement>(null);
  const { toast: pToast } = useToast();

  const fetchPartners = async () => {
    setLoading(true);
    const { data } = await supabase.from("partners").select("*").order("sort_order", { ascending: true });
    if (data) setPartners(data as DBPartner[]);
    setLoading(false);
  };

  useEffect(() => { fetchPartners(); }, []);

  const uploadLogo = async (file: File, setUrl: (url: string) => void, setUpl: (v: boolean) => void) => {
    setUpl(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `partner-${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("partners").upload(fileName, file, { upsert: true });
      if (error) { pToast({ title: "আপলোড এরর", variant: "destructive" }); return; }
      const { data: urlData } = supabase.storage.from("partners").getPublicUrl(fileName);
      setUrl(urlData.publicUrl);
    } finally { setUpl(false); }
  };

  const handleAdd = async () => {
    if (!newName.trim()) { pToast({ title: "নাম দিন", variant: "destructive" }); return; }
    const { error } = await supabase.from("partners").insert({ name: newName, logo_url: newLogoUrl || null, sort_order: newSortOrder });
    if (error) { pToast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    pToast({ title: "✅ পার্টনার যোগ হয়েছে!" });
    setNewName(""); setNewLogoUrl(""); setNewSortOrder(0); setShowAdd(false); fetchPartners();
  };

  const handleUpdate = async () => {
    if (!editingPartner) return;
    const { error } = await supabase.from("partners").update({
      name: editingPartner.name, logo_url: editingPartner.logo_url, sort_order: editingPartner.sort_order, is_active: editingPartner.is_active,
    }).eq("id", editingPartner.id);
    if (error) { pToast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    pToast({ title: "✅ পার্টনার আপডেট হয়েছে!" });
    setEditingPartner(null); fetchPartners();
  };

  const handleDelete = async (id: string) => {
    const { error } = await supabase.from("partners").delete().eq("id", id);
    if (error) { pToast({ title: "এরর", variant: "destructive" }); return; }
    pToast({ title: "🗑️ পার্টনার ডিলিট হয়েছে!" });
    fetchPartners();
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h2 className="font-heading font-bold text-lg">পার্টনার ম্যানেজমেন্ট</h2>
        <Button size="sm" className="gap-1.5 rounded-xl" onClick={() => { setShowAdd(!showAdd); setEditingPartner(null); }}>
          {showAdd ? <XCircle className="h-4 w-4" /> : <Plus className="h-4 w-4" />}
          {showAdd ? "বন্ধ করুন" : "নতুন পার্টনার"}
        </Button>
      </div>

      <AnimatePresence>
        {showAdd && (
          <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
            <SectionCard title="নতুন পার্টনার যোগ করুন" icon={Handshake}>
              <div className="space-y-4">
                <div className="space-y-1.5">
                  <Label className="text-xs">পার্টনারের নাম</Label>
                  <Input value={newName} onChange={e => setNewName(e.target.value)} placeholder="যেমন: Google, Microsoft" className="rounded-xl" />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">লোগো</Label>
                  <div className="flex items-center gap-3 flex-wrap">
                    {newLogoUrl && <img src={newLogoUrl} alt="Preview" className="h-12 max-w-[120px] object-contain rounded-lg border border-border p-1" />}
                    <input type="file" accept="image/*" ref={fileRef} onChange={e => { const f = e.target.files?.[0]; if (f) uploadLogo(f, setNewLogoUrl, setUploading); }} className="hidden" />
                    <Button variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploading} onClick={() => fileRef.current?.click()}>
                      {uploading ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                      {uploading ? "আপলোড হচ্ছে..." : "লোগো আপলোড"}
                    </Button>
                  </div>
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">সর্ট অর্ডার</Label>
                  <Input type="number" value={newSortOrder} onChange={e => setNewSortOrder(Number(e.target.value))} className="rounded-xl w-24" />
                </div>
                <Button onClick={handleAdd} className="gap-1.5 rounded-xl w-full sm:w-auto"><Save className="h-4 w-4" />যোগ করুন</Button>
              </div>
            </SectionCard>
          </motion.div>
        )}
      </AnimatePresence>

      <AnimatePresence>
        {editingPartner && (
          <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
            <SectionCard title="পার্টনার এডিট করুন" icon={Edit2}>
              <div className="space-y-4">
                <div className="space-y-1.5">
                  <Label className="text-xs">নাম</Label>
                  <Input value={editingPartner.name} onChange={e => setEditingPartner(p => p ? { ...p, name: e.target.value } : null)} className="rounded-xl" />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">লোগো</Label>
                  <div className="flex items-center gap-3 flex-wrap">
                    {editingPartner.logo_url && <img src={editingPartner.logo_url} alt="Preview" className="h-12 max-w-[120px] object-contain rounded-lg border border-border p-1" />}
                    <input type="file" accept="image/*" ref={editFileRef} onChange={e => { const f = e.target.files?.[0]; if (f) uploadLogo(f, (url) => setEditingPartner(p => p ? { ...p, logo_url: url } : null), setEditUploading); }} className="hidden" />
                    <Button variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={editUploading} onClick={() => editFileRef.current?.click()}>
                      {editUploading ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                      {editUploading ? "আপলোড হচ্ছে..." : "লোগো পরিবর্তন"}
                    </Button>
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  <Switch checked={editingPartner.is_active} onCheckedChange={checked => setEditingPartner(p => p ? { ...p, is_active: checked } : null)} />
                  <Label className="text-xs">{editingPartner.is_active ? "অ্যাক্টিভ" : "ইনঅ্যাক্টিভ"}</Label>
                </div>
                <div className="flex gap-2">
                  <Button onClick={handleUpdate} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />আপডেট</Button>
                  <Button variant="outline" onClick={() => setEditingPartner(null)} className="rounded-xl">বাতিল</Button>
                </div>
              </div>
            </SectionCard>
          </motion.div>
        )}
      </AnimatePresence>

      {loading ? (
        <div className="flex justify-center py-12"><Loader2 className="h-6 w-6 animate-spin text-primary" /></div>
      ) : partners.length === 0 ? (
        <EmptyState icon={Handshake} title="কোনো পার্টনার নেই" description="হোম পেজে পার্টনার লোগো দেখাতে নতুন পার্টনার যোগ করুন।"
          action={<Button size="sm" className="rounded-xl gap-1.5" onClick={() => setShowAdd(true)}><Plus className="h-4 w-4" />পার্টনার যোগ করুন</Button>} />
      ) : (
        <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3 sm:gap-4">
          {partners.map(p => (
            <motion.div key={p.id} initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }}
              className="bg-card rounded-xl border border-border overflow-hidden group">
              <div className="h-20 flex items-center justify-center p-3 bg-muted/30">
                {p.logo_url ? <img src={p.logo_url} alt={p.name} className="max-h-[50px] max-w-full object-contain" /> : <span className="font-heading font-bold text-foreground/50">{p.name}</span>}
              </div>
              <div className="p-2 sm:p-3">
                <div className="flex items-center justify-between mb-2">
                  <span className="text-xs font-medium truncate">{p.name}</span>
                  <span className={`text-[10px] px-1.5 py-0.5 rounded-lg ${p.is_active ? "bg-primary/10 text-primary" : "bg-muted text-muted-foreground"}`}>
                    {p.is_active ? "অ্যাক্টিভ" : "অফ"}
                  </span>
                </div>
                <div className="flex gap-1">
                  <Button variant="ghost" size="sm" className="h-7 text-[11px] rounded-lg flex-1 gap-1" onClick={() => { setEditingPartner(p); setShowAdd(false); }}>
                    <Edit2 className="h-3 w-3" />এডিট
                  </Button>
                  <Button variant="ghost" size="sm" className="h-7 text-[11px] rounded-lg text-destructive hover:text-destructive flex-1 gap-1" onClick={() => handleDelete(p.id)}>
                    <Trash2 className="h-3 w-3" />ডিলিট
                  </Button>
                </div>
              </div>
            </motion.div>
          ))}
        </div>
      )}
    </div>
  );
};

export default AdminPartnersPanel;
