import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { SectionCard } from "@/components/admin/AdminShared";
import { Save, Upload, Loader2, Image, Trash2 } from "lucide-react";

interface PopupRow {
  id: string;
  title: string;
  description: string;
  image_url: string;
  button_text: string;
  button_link: string;
  is_active: boolean;
}

const AdminPopupPanel = () => {
  const [popup, setPopup] = useState<PopupRow | null>(null);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [uploading, setUploading] = useState(false);
  const { toast } = useToast();

  // Form state
  const [title, setTitle] = useState("");
  const [description, setDescription] = useState("");
  const [imageUrl, setImageUrl] = useState("");
  const [buttonText, setButtonText] = useState("Join Now");
  const [buttonLink, setButtonLink] = useState("#");
  const [isActive, setIsActive] = useState(false);

  useEffect(() => {
    fetchPopup();
  }, []);

  const fetchPopup = async () => {
    setLoading(true);
    const { data } = await supabase.from("popups").select("*").limit(1).single();
    if (data) {
      const p = data as PopupRow;
      setPopup(p);
      setTitle(p.title);
      setDescription(p.description || "");
      setImageUrl(p.image_url || "");
      setButtonText(p.button_text || "Join Now");
      setButtonLink(p.button_link || "#");
      setIsActive(p.is_active);
    }
    setLoading(false);
  };

  const handleImageUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    setUploading(true);
    const ext = file.name.split(".").pop();
    const path = `popups/${Date.now()}.${ext}`;
    const { error } = await supabase.storage.from("banners").upload(path, file, { upsert: true });
    if (error) {
      toast({ title: "আপলোড ব্যর্থ", description: error.message, variant: "destructive" });
    } else {
      const { data: urlData } = supabase.storage.from("banners").getPublicUrl(path);
      setImageUrl(urlData.publicUrl);
      toast({ title: "ইমেজ আপলোড হয়েছে" });
    }
    setUploading(false);
  };

  const handleSave = async () => {
    setSaving(true);
    const payload = {
      title,
      description,
      image_url: imageUrl,
      button_text: buttonText,
      button_link: buttonLink,
      is_active: isActive,
      updated_at: new Date().toISOString(),
    };

    if (popup) {
      await supabase.from("popups").update(payload).eq("id", popup.id);
    } else {
      const { data } = await supabase.from("popups").insert(payload).select().single();
      if (data) setPopup(data as PopupRow);
    }
    toast({ title: "পপআপ সেভ হয়েছে ✅" });
    setSaving(false);
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center py-20">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <SectionCard title="পপআপ সেটিংস" icon={Image}>
        <div className="space-y-5">
          {/* Enable/Disable */}
          <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50">
            <div>
              <p className="font-semibold text-foreground">পপআপ সক্রিয়</p>
              <p className="text-xs text-muted-foreground">ওয়েবসাইটে ঢুকলে পপআপ দেখাবে</p>
            </div>
            <Switch checked={isActive} onCheckedChange={setIsActive} />
          </div>

          {/* Image Upload */}
          <div className="space-y-2">
            <Label>পপআপ ইমেজ</Label>
            {imageUrl && (
              <div className="relative rounded-xl overflow-hidden border border-border">
                <img src={imageUrl} alt="Popup" className="w-full h-48 object-cover" />
                <button
                  onClick={() => setImageUrl("")}
                  className="absolute top-2 right-2 p-1.5 rounded-full bg-destructive text-white hover:opacity-80"
                >
                  <Trash2 className="h-4 w-4" />
                </button>
              </div>
            )}
            <label className="flex items-center gap-2 cursor-pointer px-4 py-3 rounded-xl border-2 border-dashed border-border hover:border-primary/50 transition-colors">
              {uploading ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
              <span className="text-sm">{uploading ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড করুন"}</span>
              <input type="file" accept="image/*" className="hidden" onChange={handleImageUpload} disabled={uploading} />
            </label>
            <Input
              placeholder="অথবা ইমেজ URL পেস্ট করুন"
              value={imageUrl}
              onChange={(e) => setImageUrl(e.target.value)}
            />
          </div>

          {/* Title */}
          <div className="space-y-2">
            <Label>টাইটেল</Label>
            <Input value={title} onChange={(e) => setTitle(e.target.value)} placeholder="পপআপের শিরোনাম" />
          </div>

          {/* Description */}
          <div className="space-y-2">
            <Label>বিবরণ</Label>
            <Textarea
              value={description}
              onChange={(e) => setDescription(e.target.value)}
              placeholder="পপআপের বিস্তারিত লিখুন..."
              rows={4}
            />
          </div>

          {/* Button Text */}
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label>বাটন টেক্সট</Label>
              <Input value={buttonText} onChange={(e) => setButtonText(e.target.value)} placeholder="Join Now" />
            </div>
            <div className="space-y-2">
              <Label>বাটন লিঙ্ক</Label>
              <Input value={buttonLink} onChange={(e) => setButtonLink(e.target.value)} placeholder="https://..." />
            </div>
          </div>

          {/* Preview */}
          <div className="space-y-2">
            <Label>প্রিভিউ</Label>
            <div className="border border-border rounded-2xl overflow-hidden bg-background shadow-lg max-w-sm mx-auto">
              {imageUrl && <img src={imageUrl} alt="Preview" className="w-full h-auto object-cover" />}
              <div className="p-5 text-center">
                {title && <h3 className="font-bold text-lg text-foreground mb-2">{title}</h3>}
                {description && <p className="text-muted-foreground text-sm mb-4">{description}</p>}
                {buttonText && (
                  <div
                    className="py-2.5 rounded-xl font-bold text-white text-sm bg-primary"
                  >
                    {buttonText}
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* Save */}
          <Button onClick={handleSave} disabled={saving} className="w-full">
            {saving ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Save className="h-4 w-4 mr-2" />}
            সেভ করুন
          </Button>
        </div>
      </SectionCard>
    </div>
  );
};

export default AdminPopupPanel;
