import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { SectionCard } from "@/components/admin/AdminShared";
import { Phone, Save, Loader2, TestTube, Volume2 } from "lucide-react";

const AdminVoiceCallPanel = () => {
  const [enabled, setEnabled] = useState(false);
  const [apiToken, setApiToken] = useState("");
  const [voiceName, setVoiceName] = useState("");
  const [senderNumber, setSenderNumber] = useState("");
  const [saving, setSaving] = useState(false);
  const [testPhone, setTestPhone] = useState("");
  const [testing, setTesting] = useState(false);
  const { toast } = useToast();

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    const { data } = await supabase
      .from("site_settings")
      .select("key, value")
      .in("key", ["awaj_enabled", "awaj_api_token", "awaj_voice_name", "awaj_sender_number"]);

    data?.forEach((s) => {
      if (s.key === "awaj_enabled") setEnabled(s.value === "true");
      if (s.key === "awaj_api_token" && s.value) setApiToken(s.value);
      if (s.key === "awaj_voice_name" && s.value) setVoiceName(s.value);
      if (s.key === "awaj_sender_number" && s.value) setSenderNumber(s.value);
    });
  };

  const handleSave = async () => {
    setSaving(true);
    const settings = [
      { key: "awaj_enabled", value: enabled ? "true" : "false" },
      { key: "awaj_api_token", value: apiToken },
      { key: "awaj_voice_name", value: voiceName },
      { key: "awaj_sender_number", value: senderNumber },
    ];

    for (const s of settings) {
      const { data: existing } = await supabase
        .from("site_settings")
        .select("id")
        .eq("key", s.key)
        .maybeSingle();

      if (existing) {
        await supabase.from("site_settings").update({ value: s.value }).eq("key", s.key);
      } else {
        await supabase.from("site_settings").insert({ key: s.key, value: s.value });
      }
    }

    setSaving(false);
    toast({ title: "✅ ভয়েস কল সেটিংস সেভ হয়েছে!" });
  };

  const handleTest = async () => {
    if (!testPhone.trim()) {
      toast({ title: "ফোন নম্বর দিন", variant: "destructive" });
      return;
    }
    setTesting(true);
    try {
      const { data, error } = await supabase.functions.invoke("send-voice-call", {
        body: { phone_number: testPhone, customer_name: "টেস্ট কাস্টমার" },
      });

      if (error) {
        toast({ title: "❌ কল ব্যর্থ", description: error.message, variant: "destructive" });
      } else if (data?.success) {
        toast({ title: "✅ টেস্ট কল পাঠানো হয়েছে!" });
      } else {
        toast({
          title: "❌ কল ব্যর্থ",
          description: data?.error ? JSON.stringify(data.error) : data?.message || "অজানা সমস্যা",
          variant: "destructive",
        });
      }
    } catch (err: any) {
      toast({ title: "❌ কল ব্যর্থ", description: err.message, variant: "destructive" });
    }
    setTesting(false);
  };

  return (
    <div className="space-y-6 max-w-3xl">
      <SectionCard
        title="AwajDigital ভয়েস কল"
        description="অর্ডারের পর কাস্টমারকে অটোমেটিক ভয়েস কল পাঠান"
        icon={Volume2}
      >
        <div className="space-y-5">
          {/* Enable/Disable */}
          <div className="flex items-center justify-between p-3 rounded-lg bg-muted/50 border border-border">
            <div>
              <p className="text-sm font-medium">ভয়েস কল সক্রিয়</p>
              <p className="text-xs text-muted-foreground">
                অন করলে প্রতিটি সফল অর্ডারের পর কাস্টমারকে অটো কল যাবে
              </p>
            </div>
            <Switch checked={enabled} onCheckedChange={setEnabled} />
          </div>

          {/* API Token */}
          <div className="space-y-1.5">
            <Label className="text-xs">API Token</Label>
            <Input
              type="password"
              value={apiToken}
              onChange={(e) => setApiToken(e.target.value)}
              placeholder="আপনার AwajDigital API Token দিন"
              className="rounded-xl"
            />
            <p className="text-[11px] text-muted-foreground">
              AwajDigital ড্যাশবোর্ডের Account Settings থেকে API Token পাবেন
            </p>
          </div>

          {/* Voice Name */}
          <div className="space-y-1.5">
            <Label className="text-xs">Voice Name</Label>
            <Input
              value={voiceName}
              onChange={(e) => setVoiceName(e.target.value)}
              placeholder="আপনার অ্যাপ্রুভড ভয়েসের নাম"
              className="rounded-xl"
            />
            <p className="text-[11px] text-muted-foreground">
              AwajDigital-এ তৈরি করা এবং অ্যাপ্রুভড ভয়েসের নাম দিন
            </p>
          </div>

          {/* Sender Number */}
          <div className="space-y-1.5">
            <Label className="text-xs">Sender Number (Caller ID)</Label>
            <Input
              value={senderNumber}
              onChange={(e) => setSenderNumber(e.target.value)}
              placeholder="8801XXXXXXXXX"
              className="rounded-xl"
            />
            <p className="text-[11px] text-muted-foreground">
              আপনার অ্যাক্টিভ কলার সেন্ডার নম্বর (8801XXXXXXXXX ফরম্যাটে)
            </p>
          </div>

          <Button onClick={handleSave} disabled={saving} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
            {saving ? "সেভ হচ্ছে..." : "সেটিংস সেভ করুন"}
          </Button>
        </div>
      </SectionCard>

      {/* Test Call */}
      <SectionCard
        title="টেস্ট কল"
        description="সেটিংস ঠিক আছে কিনা যাচাই করতে টেস্ট কল পাঠান"
        icon={TestTube}
      >
        <div className="flex gap-2 items-end">
          <div className="flex-1 space-y-1.5">
            <Label className="text-xs">ফোন নম্বর</Label>
            <Input
              value={testPhone}
              onChange={(e) => setTestPhone(e.target.value)}
              placeholder="01XXXXXXXXX"
              className="rounded-xl"
            />
          </div>
          <Button
            onClick={handleTest}
            disabled={testing || !testPhone.trim()}
            variant="outline"
            className="gap-1.5 rounded-xl h-10"
          >
            {testing ? <Loader2 className="h-4 w-4 animate-spin" /> : <Phone className="h-4 w-4" />}
            {testing ? "কল হচ্ছে..." : "টেস্ট কল"}
          </Button>
        </div>
      </SectionCard>
    </div>
  );
};

export default AdminVoiceCallPanel;
