import { useEffect, useRef, useCallback } from "react";
import { supabase } from "@/integrations/supabase/client";

// Helper to read _fbp / _fbc cookies
const getCookie = (name: string): string | undefined => {
  const match = document.cookie.match(new RegExp("(^| )" + name + "=([^;]+)"));
  return match ? match[2] : undefined;
};

export interface PixelEvent {
  key: string;
  label: string;
  enabled: boolean;
}

export interface PixelConfig {
  pixel_enabled: boolean;
  pixel_id: string;
  access_token: string;
  test_event_code: string;
  events: PixelEvent[];
  custom_events: { name: string; enabled: boolean }[];
}

const DEFAULT_EVENTS: PixelEvent[] = [
  { key: "PageView", label: "PageView — পেজ লোড হলে", enabled: true },
  { key: "AddToCart", label: "Add to Cart — কার্টে যোগ হলে", enabled: true },
  { key: "InitiateCheckout", label: "Initiate Checkout — চেকআউট শুরু হলে", enabled: true },
  { key: "Purchase", label: "Purchase — পেমেন্ট সফল হলে", enabled: true },
  { key: "TimeOnPage", label: "TimeOnPage — ৩০ সেকেন্ড থাকলে", enabled: true },
  { key: "PageScroll", label: "PageScroll — ৫০% স্ক্রল হলে", enabled: true },
  { key: "ViewContent", label: "ViewContent — প্রোডাক্ট দেখলে", enabled: true },
  { key: "InternalClick", label: "InternalClick — ইন্টারনাল লিংক ক্লিক", enabled: true },
];

let pixelConfigCache: PixelConfig | null = null;
let pixelLoaded = false;

const loadPixelScript = (pixelId: string) => {
  if (pixelLoaded || !pixelId) return;
  pixelLoaded = true;

  // Facebook Pixel base code
  const f = window as any;
  if (f.fbq) return;
  const n: any = (f.fbq = function () {
    n.callMethod ? n.callMethod.apply(n, arguments) : n.queue.push(arguments);
  });
  if (!f._fbq) f._fbq = n;
  n.push = n;
  n.loaded = true;
  n.version = "2.0";
  n.queue = [];

  const s = document.createElement("script");
  s.async = true;
  s.src = "https://connect.facebook.net/en_US/fbevents.js";
  document.head.appendChild(s);

  (window as any).fbq("init", pixelId);
};

const fireEvent = (eventName: string, params?: Record<string, any>) => {
  if (!(window as any).fbq) return;
  (window as any).fbq("track", eventName, params);
};

const fireCustomEvent = (eventName: string, params?: Record<string, any>) => {
  if (!(window as any).fbq) return;
  (window as any).fbq("trackCustom", eventName, params);
};

// Send event to server-side CAPI via edge function
const sendCAPIEvent = async (eventName: string, params?: Record<string, any>, eventId?: string) => {
  try {
    await supabase.functions.invoke("facebook-capi", {
      body: {
        data: {
          event_name: eventName,
          event_source_url: window.location.href,
          event_id: eventId || crypto.randomUUID(),
          user_data: {
            fbp: getCookie("_fbp"),
            fbc: getCookie("_fbc"),
          },
          custom_data: params || {},
        },
      },
    });
  } catch (e) {
    console.warn("CAPI send failed:", e);
  }
};

export const getPixelConfig = async (): Promise<PixelConfig> => {
  if (pixelConfigCache) return pixelConfigCache;
  const { data } = await supabase.from("site_settings").select("key, value").in("key", [
    "pixel_enabled", "pixel_id", "pixel_access_token", "pixel_test_event_code", "pixel_events", "pixel_custom_events",
  ]);
  const map: Record<string, string> = {};
  data?.forEach((r) => { if (r.value) map[r.key] = r.value; });

  let events = DEFAULT_EVENTS;
  try { if (map.pixel_events) events = JSON.parse(map.pixel_events); } catch {}

  let custom_events: { name: string; enabled: boolean }[] = [];
  try { if (map.pixel_custom_events) custom_events = JSON.parse(map.pixel_custom_events); } catch {}

  pixelConfigCache = {
    pixel_enabled: map.pixel_enabled === "true",
    pixel_id: map.pixel_id || "",
    access_token: map.pixel_access_token || "",
    test_event_code: map.pixel_test_event_code || "",
    events,
    custom_events,
  };
  return pixelConfigCache;
};

export const clearPixelCache = () => { pixelConfigCache = null; };

export const isEventEnabled = (config: PixelConfig, eventKey: string): boolean => {
  const ev = config.events.find((e) => e.key === eventKey);
  return ev?.enabled ?? false;
};

// Track a standard event (browser + server-side CAPI)
export const trackPixelEvent = async (eventName: string, params?: Record<string, any>) => {
  const config = await getPixelConfig();
  if (!config.pixel_enabled || !config.pixel_id) return;
  loadPixelScript(config.pixel_id);

  if (!isEventEnabled(config, eventName)) return;
  const eventId = crypto.randomUUID();
  fireEvent(eventName, { ...params, eventID: eventId });
  // Also send server-side if access_token is configured
  if (config.access_token) {
    sendCAPIEvent(eventName, params, eventId);
  }
};

// Track a custom event (browser + server-side CAPI)
export const trackCustomPixelEvent = async (eventName: string, params?: Record<string, any>) => {
  const config = await getPixelConfig();
  if (!config.pixel_enabled || !config.pixel_id) return;
  loadPixelScript(config.pixel_id);

  const ce = config.custom_events.find((e) => e.name === eventName);
  if (!ce?.enabled) return;
  const eventId = crypto.randomUUID();
  fireCustomEvent(eventName, { ...params, eventID: eventId });
  if (config.access_token) {
    sendCAPIEvent(eventName, params, eventId);
  }
};

// Hook: auto PageView + TimeOnPage + PageScroll
export const usePixelPageTracking = () => {
  const scrollTracked = useRef(false);
  const timeTracked = useRef(false);

  useEffect(() => {
    // PageView
    trackPixelEvent("PageView");

    // TimeOnPage (30s)
    const timer = setTimeout(() => {
      if (!timeTracked.current) {
        timeTracked.current = true;
        trackPixelEvent("TimeOnPage", { seconds: 30 });
      }
    }, 30000);

    // PageScroll (50%)
    const handleScroll = () => {
      if (scrollTracked.current) return;
      const scrollPercent = (window.scrollY / (document.body.scrollHeight - window.innerHeight)) * 100;
      if (scrollPercent >= 50) {
        scrollTracked.current = true;
        trackPixelEvent("PageScroll", { percent: 50 });
      }
    };
    window.addEventListener("scroll", handleScroll, { passive: true });

    return () => {
      clearTimeout(timer);
      window.removeEventListener("scroll", handleScroll);
    };
  }, []);
};

// Hook: internal click tracking
export const usePixelInternalClickTracking = () => {
  useEffect(() => {
    const handler = (e: MouseEvent) => {
      const anchor = (e.target as HTMLElement).closest("a");
      if (!anchor) return;
      const href = anchor.getAttribute("href");
      if (href && !href.startsWith("http") && !href.startsWith("//")) {
        trackPixelEvent("InternalClick", { destination: href });
      }
    };
    document.addEventListener("click", handler);
    return () => document.removeEventListener("click", handler);
  }, []);
};

export { DEFAULT_EVENTS };
