/**
 * Parse a duration option string like "1 মাস - ৳299" into label and price.
 * Supports formats:
 * - "1 মাস - ৳299"
 * - "1 Month - ৳299"
 * - "Shared" (no price)
 */
export const parseDurationOption = (d: string): { label: string; price: string | null } => {
  // Match patterns like "label - ৳price" or "label - ৳1,299"
  const match = d.match(/^(.+?)\s*[-–—]\s*(৳[\d,.\s]+)$/);
  if (match) {
    return { label: match[1].trim(), price: match[2].trim() };
  }
  return { label: d.trim(), price: null };
};

/**
 * Extract numeric value from price string like "৳299" or "৳1,299"
 */
export const extractNumericPrice = (priceStr: string): number => {
  const num = priceStr.replace(/[৳,\s]/g, "");
  return parseFloat(num) || 0;
};

/**
 * Get price range string from duration options.
 * Returns "৳min – ৳max" if options have prices, otherwise returns null.
 */
export const getPriceRangeFromDurations = (durations: string[]): string | null => {
  const prices = durations
    .map((d) => parseDurationOption(d).price)
    .filter((p): p is string => p !== null)
    .map(extractNumericPrice)
    .filter((n) => !isNaN(n) && n > 0);

  if (prices.length === 0) return null;

  const min = Math.min(...prices);
  const max = Math.max(...prices);

  if (min === max) return `৳${min.toLocaleString("en-IN")}`;
  return `৳${min.toLocaleString("en-IN")} – ৳${max.toLocaleString("en-IN")}`;
};
