import { useState, useEffect, useCallback, useRef, useMemo } from "react";
import { useNavigate } from "react-router-dom";
import {
  Package, ShoppingCart, BarChart3, LogOut, Menu, X, Plus, Edit2, Trash2,
  Eye, Save, XCircle, Image, Upload, Loader2, Tag, Settings, TrendingUp,
  Clock, CheckCircle2, AlertCircle, Search, ChevronDown, Shield, Globe,
  CreditCard, Phone, Mail, MapPin, Facebook, MessageCircle, ExternalLink,
  LayoutDashboard, ArrowUpRight, DollarSign, RefreshCw, Handshake,
  FileText, Inbox, Users, RotateCcw, Archive, PlusCircle, CheckSquare, Star, HelpCircle
} from "lucide-react";
import { getPriceRangeFromDurations, parseDurationOption } from "@/lib/priceUtils";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { motion, AnimatePresence } from "framer-motion";
import AdminProductKeys from "@/components/AdminProductKeys";
import AdminPromoPanel from "@/components/AdminPromoPanel";
import AdminSmsPanel from "@/components/admin/AdminSmsPanel";
import AdminWhatsAppPanel from "@/components/admin/AdminWhatsAppPanel";
import AdminBlogPanel from "@/components/admin/AdminBlogPanel";
import AdminPartnersPanel from "@/components/admin/AdminPartnersPanel";
import AdminProductRequestsPanel from "@/components/admin/AdminProductRequestsPanel";
import { ConfirmDialog, StatCard, EmptyState, SectionCard } from "@/components/admin/AdminShared";
import type { DBCategory, DBProduct, DBOrder, DBBanner } from "@/components/admin/adminTypes";
import AdminUsersPanel from "@/components/admin/AdminUsersPanel";
import AdminSubscribersPanel from "@/components/admin/AdminSubscribersPanel";
import AdminVoiceCallPanel from "@/components/admin/AdminVoiceCallPanel";
import AdminAuthSettingsPanel from "@/components/admin/AdminAuthSettingsPanel";
import AdminPixelPanel from "@/components/admin/AdminPixelPanel";
import AdminReviewsPanel from "@/components/admin/AdminReviewsPanel";
import AdminFaqPanel from "@/components/admin/AdminFaqPanel";
import AdminPopupPanel from "@/components/admin/AdminPopupPanel";
import AdminIncompleteOrdersPanel from "@/components/admin/AdminIncompleteOrdersPanel";

type Tab = "dashboard" | "products" | "orders" | "banner" | "categories" | "settings" | "sms" | "whatsapp" | "email" | "product_keys" | "partners" | "blog" | "product_requests" | "promo_codes" | "users" | "subscribers" | "voice_call" | "auth_settings" | "pixel" | "reviews" | "faq" | "popup" | "incomplete_orders";


// ── Email / SMTP Settings Panel ──
const AdminEmailPanel = () => {
  const [smtpHost, setSmtpHost] = useState("");
  const [smtpPort, setSmtpPort] = useState("465");
  const [smtpUser, setSmtpUser] = useState("");
  const [smtpPassword, setSmtpPassword] = useState("");
  const [smtpEncryption, setSmtpEncryption] = useState<"ssl" | "tls" | "none">("ssl");
  const [smtpFromName, setSmtpFromName] = useState("");
  const [saving, setSaving] = useState(false);
  const [testEmail, setTestEmail] = useState("");
  const [testSending, setTestSending] = useState(false);
  const [savingTemplates, setSavingTemplates] = useState(false);
  const { toast: emailToast } = useToast();

  // Email templates
  interface EmailTemplate {
    key: string;
    label: string;
    description: string;
    recipient: string;
    enabled: boolean;
    subject: string;
    body: string;
    defaultSubject: string;
    defaultBody: string;
    selectedTheme: string;
    tags: string;
  }

  // WordPress WooCommerce-style email body builder
  const makeBody = (headerColor: string, headerTitle: string, headerSub: string, contentHtml: string) =>
    `<div style="font-family:'Segoe UI',Arial,sans-serif;max-width:600px;margin:0 auto;background:#f7f7f7;padding:20px 0">
      <div style="background:#fff;border-radius:8px;overflow:hidden;border:1px solid #e0e0e0;margin:0 20px">
        <!-- Header -->
        <div style="background:${headerColor};padding:28px 30px;text-align:center">
          <h1 style="color:#fff;margin:0;font-size:22px;font-weight:700;letter-spacing:0.5px">${headerTitle}</h1>
          <p style="color:rgba(255,255,255,.85);margin:8px 0 0;font-size:14px">${headerSub}</p>
        </div>
        <!-- Content -->
        <div style="padding:30px 30px 20px">${contentHtml}</div>
        <!-- Footer -->
        <div style="border-top:1px solid #e5e7eb;padding:20px 30px;text-align:center;background:#fafafa">
          <p style="color:#999;font-size:12px;margin:0 0 8px">এই ইমেইলটি স্বয়ংক্রিয়ভাবে পাঠানো হয়েছে।</p>
          <p style="color:#999;font-size:12px;margin:0 0 8px">© ${new Date().getFullYear()} {shop_name} | সকল স্বত্ব সংরক্ষিত</p>
          <div style="margin-top:12px">
            <a href="{shop_url}" style="color:#6366f1;font-size:12px;text-decoration:none;margin:0 8px">🌐 ওয়েবসাইট</a>
            <a href="mailto:{support_email}" style="color:#6366f1;font-size:12px;text-decoration:none;margin:0 8px">✉️ সাপোর্ট</a>
          </div>
        </div>
      </div>
    </div>`;

  const purpleGrad = "linear-gradient(135deg,#6366f1,#8b5cf6)";
  const greenGrad = "linear-gradient(135deg,#059669,#10b981)";
  const redGrad = "linear-gradient(135deg,#dc2626,#ef4444)";
  const blueGrad = "linear-gradient(135deg,#1e40af,#3b82f6)";
  const orangeGrad = "linear-gradient(135deg,#ea580c,#f97316)";
  const darkBg = "#1a1a2e";

  // Reusable order details table
  const orderTable = `
    <table style="width:100%;border-collapse:collapse;margin:16px 0;font-size:14px">
      <thead>
        <tr style="background:#f3f4f6">
          <th style="text-align:left;padding:10px 12px;border-bottom:2px solid #e5e7eb;color:#374151;font-size:13px">প্রোডাক্ট</th>
          <th style="text-align:right;padding:10px 12px;border-bottom:2px solid #e5e7eb;color:#374151;font-size:13px">মূল্য</th>
        </tr>
      </thead>
      <tbody>
        <tr>
          <td style="padding:12px;border-bottom:1px solid #f3f4f6;color:#1f2937">{product}</td>
          <td style="padding:12px;border-bottom:1px solid #f3f4f6;text-align:right;color:#1f2937;font-weight:600">{total} ৳</td>
        </tr>
      </tbody>
      <tfoot>
        <tr style="background:#f9fafb">
          <td style="padding:12px;font-weight:700;color:#111827;border-top:2px solid #e5e7eb">মোট</td>
          <td style="padding:12px;text-align:right;font-weight:700;color:#6366f1;font-size:16px;border-top:2px solid #e5e7eb">{total} ৳</td>
        </tr>
      </tfoot>
    </table>`;

  // Billing info block
  const billingBlock = `
    <div style="margin:20px 0;padding:16px;background:#f9fafb;border:1px solid #e5e7eb;border-radius:8px">
      <h3 style="margin:0 0 12px;font-size:14px;color:#374151;font-weight:700;border-bottom:1px solid #e5e7eb;padding-bottom:8px">📋 বিলিং তথ্য</h3>
      <table style="width:100%;font-size:13px;color:#4b5563">
        <tr><td style="padding:4px 0;width:120px;color:#6b7280">কাস্টমার:</td><td style="padding:4px 0;font-weight:600">{name}</td></tr>
        <tr><td style="padding:4px 0;color:#6b7280">ফোন:</td><td style="padding:4px 0">{phone}</td></tr>
        <tr><td style="padding:4px 0;color:#6b7280">ইমেইল:</td><td style="padding:4px 0">{email}</td></tr>
        <tr><td style="padding:4px 0;color:#6b7280">অর্ডার ID:</td><td style="padding:4px 0;font-family:monospace;font-size:12px">{order_id}</td></tr>
      </table>
    </div>`;

  // Status badge builder
  const statusBadge = (emoji: string, text: string, bgColor: string, textColor: string) =>
    `<div style="display:inline-block;padding:6px 16px;border-radius:20px;background:${bgColor};color:${textColor};font-size:13px;font-weight:600;margin:8px 0">${emoji} ${text}</div>`;

  // Theme variants for each template
  const themeVariants = [
    { name: "💜 পার্পল", color: purpleGrad },
    { name: "🟢 গ্রিন", color: greenGrad },
    { name: "🔵 ব্লু", color: blueGrad },
    { name: "🌙 ডার্ক", color: darkBg },
  ];

  const allEmailTypes: Omit<EmailTemplate, "enabled" | "subject" | "body" | "selectedTheme">[] = [
    { key: "new_order", label: "New order", description: "নতুন অর্ডার পেলে অ্যাডমিনকে নোটিফাই করে", recipient: "Admin",
      defaultSubject: "🛒 [অর্ডার #{order_id}] নতুন অর্ডার - {product}",
      defaultBody: makeBody(purpleGrad, "{shop_name}", "🛒 নতুন অর্ডার পেয়েছেন!",
        `<p style="font-size:15px;color:#1f2937">আপনি একটি নতুন অর্ডার পেয়েছেন। নিচে অর্ডারের বিস্তারিত দেখুন:</p>
        ${orderTable}
        ${billingBlock}
        <div style="text-align:center;margin:24px 0">
          <a href="{admin_url}" style="background:linear-gradient(135deg,#6366f1,#8b5cf6);color:#fff;padding:12px 28px;border-radius:8px;text-decoration:none;font-weight:600;font-size:14px;display:inline-block">📦 অর্ডার ম্যানেজ করুন</a>
        </div>`),
      tags: "{name}, {product}, {total}, {phone}, {email}, {order_id}, {admin_url}" },

    { key: "order_confirmation", label: "Order confirmation", description: "পেমেন্ট সফল হলে কাস্টমারকে কনফার্মেশন পাঠায়", recipient: "Customer",
      defaultSubject: "✅ অর্ডার কনফার্মেশন #{order_id} - পেমেন্ট সফল",
      defaultBody: makeBody(purpleGrad, "{shop_name}", "✅ পেমেন্ট সফল হয়েছে!",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডারের পেমেন্ট সফলভাবে সম্পন্ন হয়েছে! ধন্যবাদ আপনার কেনাকাটার জন্য। 🎉</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("✅", "পেমেন্ট সফল", "#dcfce7", "#166534")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="margin:16px 0;padding:12px 16px;background:#eff6ff;border-left:4px solid #3b82f6;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#1e40af"><strong>Transaction ID:</strong> {transaction_id}</p>
        </div>
        <p style="color:#6b7280;font-size:13px;margin-top:20px">আমরা শীঘ্রই আপনার অর্ডার প্রসেস করবো এবং আপডেট জানাবো।</p>`),
      tags: "{name}, {product}, {total}, {transaction_id}, {email}, {order_id}, {phone}" },

    { key: "processing", label: "Processing order", description: "অর্ডার প্রসেসিং হলে কাস্টমারকে জানায়", recipient: "Customer",
      defaultSubject: "⚙️ অর্ডার #{order_id} প্রসেসিং হচ্ছে - {product}",
      defaultBody: makeBody(blueGrad, "{shop_name}", "⚙️ আপনার অর্ডার প্রসেস হচ্ছে",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডার এখন প্রসেস করা হচ্ছে। আমরা শীঘ্রই আপনাকে আপডেট জানাবো।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("⚙️", "প্রসেসিং", "#dbeafe", "#1e40af")}</div>
        ${orderTable}
        ${billingBlock}`),
      tags: "{name}, {product}, {order_id}, {total}, {phone}, {email}" },

    { key: "completed", label: "Completed order", description: "অর্ডার সম্পন্ন হলে কাস্টমারকে জানায়", recipient: "Customer",
      defaultSubject: "🎉 অর্ডার #{order_id} সম্পন্ন - ধন্যবাদ!",
      defaultBody: makeBody(greenGrad, "{shop_name}", "🎉 অর্ডার সম্পন্ন হয়েছে!",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডার সফলভাবে সম্পন্ন হয়েছে! আপনার প্রোডাক্ট ডেলিভারি করা হয়েছে। 🎉</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("✅", "সম্পন্ন", "#dcfce7", "#166534")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="margin:20px 0;padding:16px;background:#f0fdf4;border:1px solid #bbf7d0;border-radius:8px;text-align:center">
          <p style="margin:0 0 8px;font-size:14px;color:#166534;font-weight:600">📦 আপনার প্রোডাক্ট কী</p>
          <p style="margin:0;font-size:13px;color:#4b5563">প্রোফাইল পেজ থেকে আপনার প্রোডাক্ট কী দেখুন</p>
          <a href="{profile_url}" style="display:inline-block;margin-top:12px;background:linear-gradient(135deg,#059669,#10b981);color:#fff;padding:10px 24px;border-radius:8px;text-decoration:none;font-weight:600;font-size:13px">🔑 কী দেখুন</a>
        </div>
        <p style="color:#6b7280;font-size:13px;margin-top:16px;text-align:center">ধন্যবাদ! আবার কেনাকাটার আমন্ত্রণ রইলো। 💜</p>`),
      tags: "{name}, {product}, {order_id}, {total}, {phone}, {email}, {profile_url}" },

    { key: "cancelled", label: "Cancelled order", description: "অর্ডার বাতিল হলে জানায়", recipient: "Customer",
      defaultSubject: "❌ অর্ডার #{order_id} বাতিল করা হয়েছে",
      defaultBody: makeBody(redGrad, "{shop_name}", "❌ অর্ডার বাতিল",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">দুঃখিত, আপনার অর্ডারটি বাতিল করা হয়েছে।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("❌", "বাতিল", "#fee2e2", "#991b1b")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="margin:20px 0;padding:16px;background:#fef2f2;border-left:4px solid #ef4444;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#991b1b">কোনো সমস্যা থাকলে আমাদের সাথে যোগাযোগ করুন।</p>
        </div>`),
      tags: "{name}, {product}, {order_id}, {total}, {phone}, {email}" },

    { key: "failed", label: "Failed order", description: "অর্ডার ব্যর্থ হলে জানায়", recipient: "Customer",
      defaultSubject: "⚠️ অর্ডার #{order_id} ব্যর্থ হয়েছে",
      defaultBody: makeBody(redGrad, "{shop_name}", "⚠️ অর্ডার ব্যর্থ",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">দুঃখিত, আপনার অর্ডার প্রসেস করতে সমস্যা হয়েছে।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("⚠️", "ব্যর্থ", "#fee2e2", "#991b1b")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="text-align:center;margin:24px 0">
          <a href="{shop_url}" style="background:linear-gradient(135deg,#dc2626,#ef4444);color:#fff;padding:12px 28px;border-radius:8px;text-decoration:none;font-weight:600;font-size:14px;display:inline-block">🔄 আবার চেষ্টা করুন</a>
        </div>`),
      tags: "{name}, {product}, {order_id}, {total}, {phone}, {email}, {shop_url}" },

    { key: "on_hold", label: "Order on-hold", description: "অর্ডার হোল্ডে থাকলে কাস্টমারকে জানায়", recipient: "Customer",
      defaultSubject: "⏸️ অর্ডার #{order_id} হোল্ডে আছে",
      defaultBody: makeBody(orangeGrad, "{shop_name}", "⏸️ অর্ডার হোল্ডে",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডার সাময়িকভাবে হোল্ডে রাখা হয়েছে। পেমেন্ট যাচাই করা হচ্ছে।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("⏸️", "হোল্ড", "#ffedd5", "#9a3412")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="margin:20px 0;padding:16px;background:#fff7ed;border-left:4px solid #f97316;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#9a3412">পেমেন্ট সম্পন্ন করুন অথবা সাপোর্টে যোগাযোগ করুন।</p>
        </div>`),
      tags: "{name}, {product}, {order_id}, {total}, {phone}, {email}" },

    { key: "refunded", label: "Refunded order", description: "রিফান্ড হলে কাস্টমারকে জানায়", recipient: "Customer",
      defaultSubject: "💰 অর্ডার #{order_id} রিফান্ড সম্পন্ন",
      defaultBody: makeBody(blueGrad, "{shop_name}", "💰 রিফান্ড সম্পন্ন",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডারের রিফান্ড সফলভাবে প্রসেস করা হয়েছে।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("💰", "রিফান্ড সম্পন্ন", "#dbeafe", "#1e40af")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="margin:20px 0;padding:16px;background:#eff6ff;border-left:4px solid #3b82f6;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#1e40af">রিফান্ড আপনার অ্যাকাউন্টে ৩-৭ কার্যদিবসের মধ্যে পৌঁছাবে।</p>
        </div>`),
      tags: "{name}, {product}, {total}, {order_id}, {phone}, {email}" },

    { key: "password_reset", label: "Reset password", description: "পাসওয়ার্ড রিসেট লিংক পাঠায়", recipient: "Customer",
      defaultSubject: "🔐 পাসওয়ার্ড রিসেট - {shop_name}",
      defaultBody: makeBody(purpleGrad, "{shop_name}", "🔐 পাসওয়ার্ড রিসেট",
        `<p style="font-size:15px;color:#1f2937">আপনি আপনার পাসওয়ার্ড রিসেটের অনুরোধ করেছেন।</p>
        <p style="color:#4b5563">নিচের বাটনে ক্লিক করে নতুন পাসওয়ার্ড সেট করুন:</p>
        <div style="text-align:center;margin:30px 0">
          <a href="{reset_link}" style="background:linear-gradient(135deg,#6366f1,#8b5cf6);color:#fff;padding:14px 36px;border-radius:8px;text-decoration:none;font-weight:700;font-size:16px;display:inline-block;box-shadow:0 4px 14px rgba(99,102,241,.4)">🔑 পাসওয়ার্ড রিসেট করুন</a>
        </div>
        <div style="margin:20px 0;padding:16px;background:#fef3c7;border-left:4px solid #f59e0b;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#92400e">⚠️ এই লিংক ১ ঘণ্টার মধ্যে মেয়াদোত্তীর্ণ হবে।</p>
          <p style="margin:4px 0 0;font-size:12px;color:#a16207">আপনি যদি এই অনুরোধ না করে থাকেন, এই ইমেইল উপেক্ষা করুন।</p>
        </div>`),
      tags: "{reset_link}" },

    { key: "new_account", label: "New account", description: "নতুন অ্যাকাউন্ট তৈরি হলে স্বাগতম ইমেইল পাঠায়", recipient: "Customer",
      defaultSubject: "🎉 স্বাগতম {name} - {shop_name}",
      defaultBody: makeBody(purpleGrad, "{shop_name}", "🎉 স্বাগতম!",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">{shop_name}-এ আপনাকে স্বাগতম! আপনার অ্যাকাউন্ট সফলভাবে তৈরি হয়েছে। 🎉</p>
        <div style="margin:20px 0;padding:20px;background:#f0fdf4;border:1px solid #bbf7d0;border-radius:8px">
          <h3 style="margin:0 0 12px;font-size:14px;color:#166534">✨ আপনি এখন যা করতে পারবেন:</h3>
          <ul style="margin:0;padding:0 0 0 20px;color:#4b5563;font-size:13px;line-height:2">
            <li>প্রিমিয়াম সফটওয়্যার ও ডিজিটাল প্রোডাক্ট কিনুন</li>
            <li>অর্ডারের ট্র্যাকিং ও হিস্ট্রি দেখুন</li>
            <li>প্রোডাক্ট কী সংরক্ষণ ও ম্যানেজ করুন</li>
            <li>এক্সক্লুসিভ অফার ও ডিসকাউন্ট পান</li>
          </ul>
        </div>
        <div style="text-align:center;margin:24px 0">
          <a href="{shop_link}" style="background:linear-gradient(135deg,#6366f1,#8b5cf6);color:#fff;padding:14px 36px;border-radius:8px;text-decoration:none;font-weight:700;font-size:16px;display:inline-block;box-shadow:0 4px 14px rgba(99,102,241,.4)">🛍️ শপিং শুরু করুন</a>
        </div>`),
      tags: "{name}, {email}, {shop_link}" },

    { key: "customer_note", label: "Customer note", description: "কাস্টমারকে নোট/মেসেজ পাঠায়", recipient: "Customer",
      defaultSubject: "📝 আপনার জন্য একটি বার্তা - {shop_name}",
      defaultBody: makeBody(purpleGrad, "{shop_name}", "📝 আপনার জন্য বার্তা",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডার সম্পর্কে একটি আপডেট আছে:</p>
        <div style="margin:20px 0;padding:20px;background:#f9fafb;border-left:4px solid #6366f1;border-radius:4px">
          <p style="margin:0;font-size:14px;color:#1f2937;line-height:1.7">{note}</p>
        </div>
        <p style="color:#6b7280;font-size:13px;margin-top:20px">কোনো প্রশ্ন থাকলে আমাদের সাথে যোগাযোগ করুন।</p>`),
      tags: "{name}, {note}" },

    { key: "cancelled_admin", label: "Cancelled order (Admin)", description: "অর্ডার বাতিল হলে অ্যাডমিনকে নোটিফাই করে", recipient: "Admin",
      defaultSubject: "❌ [অর্ডার #{order_id}] অর্ডার বাতিল হয়েছে - {product}",
      defaultBody: makeBody(redGrad, "{shop_name}", "❌ একটি অর্ডার বাতিল হয়েছে",
        `<p style="font-size:15px;color:#1f2937">নিচের অর্ডারটি বাতিল করা হয়েছে:</p>
        ${orderTable}
        ${billingBlock}
        <div style="text-align:center;margin:24px 0">
          <a href="{admin_url}" style="background:linear-gradient(135deg,#dc2626,#ef4444);color:#fff;padding:12px 28px;border-radius:8px;text-decoration:none;font-weight:600;font-size:14px;display:inline-block">📦 অর্ডার দেখুন</a>
        </div>`),
      tags: "{name}, {product}, {total}, {phone}, {email}, {order_id}, {admin_url}" },

    { key: "failed_admin", label: "Failed order (Admin)", description: "অর্ডার ব্যর্থ হলে অ্যাডমিনকে নোটিফাই করে", recipient: "Admin",
      defaultSubject: "⚠️ [অর্ডার #{order_id}] অর্ডার ব্যর্থ হয়েছে - {product}",
      defaultBody: makeBody(redGrad, "{shop_name}", "⚠️ একটি অর্ডার ব্যর্থ হয়েছে",
        `<p style="font-size:15px;color:#1f2937">নিচের অর্ডারটি ব্যর্থ হয়েছে। দয়া করে চেক করুন:</p>
        ${orderTable}
        ${billingBlock}
        <div style="text-align:center;margin:24px 0">
          <a href="{admin_url}" style="background:linear-gradient(135deg,#dc2626,#ef4444);color:#fff;padding:12px 28px;border-radius:8px;text-decoration:none;font-weight:600;font-size:14px;display:inline-block">📦 অর্ডার ম্যানেজ করুন</a>
        </div>`),
      tags: "{name}, {product}, {total}, {phone}, {email}, {order_id}, {admin_url}" },

    { key: "order_details", label: "Order details", description: "কাস্টমারকে অর্ডারের বিস্তারিত তথ্য পাঠায়", recipient: "Customer",
      defaultSubject: "📋 অর্ডার #{order_id} এর বিস্তারিত - {shop_name}",
      defaultBody: makeBody(blueGrad, "{shop_name}", "📋 অর্ডারের বিস্তারিত",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডারের বিস্তারিত তথ্য নিচে দেওয়া হলো:</p>
        ${orderTable}
        ${billingBlock}
        <div style="margin:16px 0;padding:12px 16px;background:#eff6ff;border-left:4px solid #3b82f6;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#1e40af"><strong>Transaction ID:</strong> {transaction_id}</p>
        </div>
        <div style="text-align:center;margin:24px 0">
          <a href="{profile_url}" style="background:linear-gradient(135deg,#1e40af,#3b82f6);color:#fff;padding:12px 28px;border-radius:8px;text-decoration:none;font-weight:600;font-size:14px;display:inline-block">📦 আমার অর্ডারস</a>
        </div>`),
      tags: "{name}, {product}, {total}, {order_id}, {transaction_id}, {phone}, {email}, {profile_url}" },

    { key: "license_keys", label: "Order License Keys", description: "প্রোডাক্ট কী/লাইসেন্স কী কাস্টমারকে পাঠায়", recipient: "Customer",
      defaultSubject: "🔑 লাইসেন্স কী - অর্ডার #{order_id}",
      defaultBody: makeBody(greenGrad, "{shop_name}", "🔑 আপনার লাইসেন্স কী",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার অর্ডারের প্রোডাক্ট কী নিচে দেওয়া হলো। দয়া করে নিরাপদে সংরক্ষণ করুন।</p>
        <div style="margin:20px 0;padding:20px;background:#f0fdf4;border:1px solid #bbf7d0;border-radius:8px">
          <p style="margin:0 0 8px;font-size:14px;color:#166534;font-weight:600">🔑 প্রোডাক্ট কী</p>
          <div style="background:#fff;border:1px solid #d1d5db;border-radius:6px;padding:12px;font-family:monospace;font-size:14px;word-break:break-all;color:#1f2937">{license_key}</div>
        </div>
        ${orderTable}
        <div style="text-align:center;margin:24px 0">
          <a href="{profile_url}" style="background:linear-gradient(135deg,#059669,#10b981);color:#fff;padding:12px 28px;border-radius:8px;text-decoration:none;font-weight:600;font-size:14px;display:inline-block">🔑 প্রোফাইলে কী দেখুন</a>
        </div>`),
      tags: "{name}, {product}, {total}, {order_id}, {license_key}, {profile_url}" },

    { key: "pos_completed", label: "POS completed order", description: "POS থেকে সম্পন্ন অর্ডারের নোটিফিকেশন", recipient: "Customer",
      defaultSubject: "✅ POS অর্ডার #{order_id} সম্পন্ন",
      defaultBody: makeBody(greenGrad, "{shop_name}", "✅ POS অর্ডার সম্পন্ন",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার POS অর্ডার সফলভাবে সম্পন্ন হয়েছে।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("✅", "POS সম্পন্ন", "#dcfce7", "#166534")}</div>
        ${orderTable}
        ${billingBlock}`),
      tags: "{name}, {product}, {total}, {order_id}, {phone}, {email}" },

    { key: "pos_refunded", label: "POS refunded order", description: "POS থেকে রিফান্ড হওয়া অর্ডারের নোটিফিকেশন", recipient: "Customer",
      defaultSubject: "💰 POS অর্ডার #{order_id} রিফান্ড সম্পন্ন",
      defaultBody: makeBody(blueGrad, "{shop_name}", "💰 POS রিফান্ড সম্পন্ন",
        `<p style="font-size:15px;color:#1f2937">প্রিয় <strong>{name}</strong>,</p>
        <p style="color:#4b5563">আপনার POS অর্ডারের রিফান্ড সফলভাবে প্রসেস করা হয়েছে।</p>
        <div style="text-align:center;margin:16px 0">${statusBadge("💰", "POS রিফান্ড", "#dbeafe", "#1e40af")}</div>
        ${orderTable}
        ${billingBlock}
        <div style="margin:20px 0;padding:16px;background:#eff6ff;border-left:4px solid #3b82f6;border-radius:4px">
          <p style="margin:0;font-size:13px;color:#1e40af">রিফান্ড আপনার অ্যাকাউন্টে ৩-৭ কার্যদিবসের মধ্যে পৌঁছাবে।</p>
        </div>`),
      tags: "{name}, {product}, {total}, {order_id}, {phone}, {email}" },
  ];

  const defaultEmailTemplates: EmailTemplate[] = allEmailTypes.map(t => ({
    ...t,
    enabled: ["order_confirmation", "password_reset", "completed"].includes(t.key),
    subject: "",
    body: "",
    selectedTheme: "💜 পার্পল",
  }));

  const [emailTemplates, setEmailTemplates] = useState<EmailTemplate[]>(defaultEmailTemplates);
  const [managingTemplate, setManagingTemplate] = useState<string | null>(null);
  const [previewTemplate, setPreviewTemplate] = useState<string | null>(null);

  useEffect(() => {
    const load = async () => {
      const { data } = await supabase.from("site_settings").select("key, value");
      if (data) {
        const map: Record<string, string> = {};
        data.forEach(r => { if (r.value) map[r.key] = r.value; });
        if (map.smtp_host) setSmtpHost(map.smtp_host);
        if (map.smtp_port) setSmtpPort(map.smtp_port);
        if (map.smtp_user) setSmtpUser(map.smtp_user);
        if (map.smtp_password) setSmtpPassword(map.smtp_password);
        if (map.smtp_encryption) setSmtpEncryption(map.smtp_encryption as "ssl" | "tls" | "none");
        if (map.smtp_from_name) setSmtpFromName(map.smtp_from_name);
        if (map.email_templates) {
          try {
            const saved = JSON.parse(map.email_templates) as Partial<EmailTemplate>[];
            setEmailTemplates(defaultEmailTemplates.map(dt => {
              const found = saved.find(s => s.key === dt.key);
              return found ? { ...dt, ...found, defaultSubject: dt.defaultSubject, defaultBody: dt.defaultBody, label: dt.label, description: dt.description, recipient: dt.recipient, tags: dt.tags } : dt;
            }));
          } catch {}
        }
      }
    };
    load();
  }, []);

  const upsert = async (key: string, value: string) => {
    const { data: existing } = await supabase.from("site_settings").select("id").eq("key", key).maybeSingle();
    if (existing) await supabase.from("site_settings").update({ value, updated_at: new Date().toISOString() }).eq("key", key);
    else await supabase.from("site_settings").insert({ key, value });
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      await Promise.all([
        upsert("smtp_host", smtpHost),
        upsert("smtp_port", smtpPort),
        upsert("smtp_user", smtpUser),
        upsert("smtp_password", smtpPassword),
        upsert("smtp_encryption", smtpEncryption),
        upsert("smtp_from_name", smtpFromName),
      ]);
      emailToast({ title: "✅ SMTP সেটিংস সেভ হয়েছে!" });
    } catch {
      emailToast({ title: "সেভ করতে সমস্যা হয়েছে", variant: "destructive" });
    } finally {
      setSaving(false);
    }
  };

  const handleSaveTemplates = async () => {
    setSavingTemplates(true);
    try {
      const templateData = emailTemplates.map(t => ({ key: t.key, enabled: t.enabled, subject: t.subject, body: t.body, selectedTheme: t.selectedTheme }));
      await upsert("email_templates", JSON.stringify(templateData));
      emailToast({ title: "✅ ইমেইল টেমপ্লেট সেভ হয়েছে!" });
    } catch {
      emailToast({ title: "সেভ করতে সমস্যা হয়েছে", variant: "destructive" });
    } finally {
      setSavingTemplates(false);
    }
  };

  const handleTestEmail = async () => {
    if (!testEmail) { emailToast({ title: "ইমেইল অ্যাড্রেস দিন", variant: "destructive" }); return; }
    setTestSending(true);
    try {
      const testHtml = makeBody(purpleGrad, "Test", "Test Email", `<p>🎉 আপনার SMTP কনফিগারেশন সঠিকভাবে কাজ করছে।</p>`)
        .replace(/\{shop_name\}/g, smtpFromName || "TZSS")
        .replace(/\[shop_name\]/g, smtpFromName || "TZSS")
        .replace(/\{shop_url\}/g, window.location.origin)
        .replace(/\{support_email\}/g, smtpUser || "info@tzssbd.com");
      const { data, error } = await supabase.functions.invoke("send-email", {
        body: { to: testEmail, subject: "✅ Test Email - SMTP", html: testHtml, text: "SMTP working!" },
      });
      if (error) throw error;
      if (data?.success) emailToast({ title: "✅ টেস্ট ইমেইল পাঠানো হয়েছে!" });
      else emailToast({ title: data?.error || "ইমেইল পাঠাতে সমস্যা", variant: "destructive" });
    } catch (e: any) {
      emailToast({ title: e?.message || "ইমেইল পাঠাতে সমস্যা", variant: "destructive" });
    } finally { setTestSending(false); }
  };

  const updateTemplate = (key: string, field: Partial<EmailTemplate>) => {
    setEmailTemplates(prev => prev.map(t => t.key === key ? { ...t, ...field } : t));
  };

  const applyTheme = (templateKey: string, variant: { name: string; color: string }) => {
    const tpl = allEmailTypes.find(t => t.key === templateKey);
    if (!tpl) return;
    // Re-build body with selected color
    const newBody = tpl.defaultBody.replace(/linear-gradient\(135deg,[^)]+\)|#1a1a2e/g, variant.color);
    updateTemplate(templateKey, { body: newBody, selectedTheme: variant.name });
  };

  const getPreviewHtml = (template: EmailTemplate) => {
    return (template.body || template.defaultBody)
      .replace(/\{name\}/g, "রাহিম আহমেদ").replace(/\{product\}/g, "ChatGPT Plus - 1 Month")
      .replace(/\{total\}/g, "1,200").replace(/\{transaction_id\}/g, "TXN-ABC12345")
      .replace(/\{email\}/g, "rahim@example.com").replace(/\{reset_link\}/g, "#")
      .replace(/\{status\}/g, "সম্পন্ন ✅").replace(/\{order_id\}/g, "ORD-7X3K9M")
      .replace(/\{phone\}/g, "01712345678").replace(/\{note\}/g, "আপনার প্রোডাক্ট রেডি আছে। প্রোফাইল থেকে কী কপি করে নিন।")
      .replace(/\{shop_name\}/g, smtpFromName || "TZSS").replace(/\[shop_name\]/g, smtpFromName || "TZSS")
      .replace(/\{shop_link\}/g, "#").replace(/\{shop_url\}/g, "#").replace(/\{support_email\}/g, smtpUser || "info@tzssbd.com")
      .replace(/\{admin_url\}/g, "#").replace(/\{profile_url\}/g, "#");
  };

  return (
    <div className="space-y-6 max-w-3xl">
      {/* SMTP Connection Settings */}
      <SectionCard title="SMTP Connection Settings" description="ইমেইল পাঠানোর জন্য SMTP সেটিংস কনফিগার করুন" icon={Mail}>
        <div className="space-y-4">
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div className="space-y-1.5">
              <Label className="text-xs font-medium">SMTP Host</Label>
              <Input value={smtpHost} onChange={e => setSmtpHost(e.target.value)} placeholder="smtp.hostinger.com" className="rounded-xl" />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs font-medium">SMTP Port</Label>
              <Input value={smtpPort} onChange={e => setSmtpPort(e.target.value)} placeholder="465" className="rounded-xl" />
            </div>
          </div>
          <div className="space-y-1.5">
            <Label className="text-xs font-medium">Encryption</Label>
            <div className="flex gap-3 flex-wrap">
              {([
                { value: "ssl", label: "SSL (Port 465)", port: "465" },
                { value: "tls", label: "TLS/STARTTLS (Port 587)", port: "587" },
                { value: "none", label: "None", port: smtpPort },
              ] as const).map(opt => (
                <button key={opt.value} onClick={() => { setSmtpEncryption(opt.value); setSmtpPort(opt.port); }}
                  className={`px-4 py-2 rounded-xl text-sm font-medium transition-all border ${smtpEncryption === opt.value ? "bg-primary text-primary-foreground border-primary" : "bg-muted text-muted-foreground border-border"}`}>
                  {opt.label}
                </button>
              ))}
            </div>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div className="space-y-1.5">
              <Label className="text-xs font-medium">Username / Email</Label>
              <Input value={smtpUser} onChange={e => setSmtpUser(e.target.value)} placeholder="info@tzssbd.com" className="rounded-xl" />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs font-medium">Password</Label>
              <Input type="password" value={smtpPassword} onChange={e => setSmtpPassword(e.target.value)} placeholder="••••••••" className="rounded-xl" />
            </div>
          </div>
          <div className="space-y-1.5">
            <Label className="text-xs font-medium">Email From Name (প্রেরকের নাম)</Label>
            <Input value={smtpFromName} onChange={e => setSmtpFromName(e.target.value)} placeholder="TZSS" className="rounded-xl" />
            <p className="text-[11px] text-muted-foreground">ইমেইলে From হিসেবে যে নাম দেখাবে। খালি রাখলে সাইট টাইটেল ব্যবহার হবে।</p>
          </div>
          <Button onClick={handleSave} disabled={saving} className="gap-1.5 rounded-xl">
            {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
            {saving ? "সেভ হচ্ছে..." : "Save Connection Settings"}
          </Button>
        </div>
      </SectionCard>

      {/* WooCommerce-style Email List */}
      <SectionCard title="Email Notifications" description="প্রতিটি ইমেইল টাইপ Enable/Disable ও Manage করুন" icon={Mail}>
        <div className="border border-border rounded-xl overflow-hidden">
          {/* Table Header */}
          <div className="grid grid-cols-[1fr_auto_auto_auto] gap-2 px-4 py-2.5 bg-muted/70 border-b border-border text-xs font-semibold text-muted-foreground">
            <span>Email</span>
            <span className="w-20 text-center">Recipient</span>
            <span className="w-12 text-center">Status</span>
            <span className="w-20 text-center">Action</span>
          </div>

          {/* Email Rows */}
          {emailTemplates.map((tpl, i) => (
            <div key={tpl.key}>
              <div className={`grid grid-cols-[1fr_auto_auto_auto] gap-2 px-4 py-3 items-center ${i < emailTemplates.length - 1 ? "border-b border-border" : ""} hover:bg-muted/30 transition-colors`}>
                <div>
                  <p className="text-sm font-medium text-foreground">{tpl.label}</p>
                  <p className="text-[11px] text-muted-foreground">{tpl.description}</p>
                </div>
                <span className="w-20 text-center">
                  <span className={`text-[10px] px-2 py-0.5 rounded-full font-medium ${tpl.recipient === "Admin" ? "bg-orange-100 text-orange-700" : "bg-blue-100 text-blue-700"}`}>
                    {tpl.recipient}
                  </span>
                </span>
                <div className="w-12 flex justify-center">
                  <Switch checked={tpl.enabled} onCheckedChange={(checked) => updateTemplate(tpl.key, { enabled: checked })} />
                </div>
                <div className="w-20 flex justify-center">
                  <Button variant="outline" size="sm" className="rounded-lg text-xs h-8 px-3"
                    onClick={() => setManagingTemplate(managingTemplate === tpl.key ? null : tpl.key)}>
                    {managingTemplate === tpl.key ? "Close" : "Manage"}
                  </Button>
                </div>
              </div>

              {/* Expanded Manage Panel */}
              <AnimatePresence>
                {managingTemplate === tpl.key && (
                  <motion.div
                    initial={{ opacity: 0, height: 0 }}
                    animate={{ opacity: 1, height: "auto" }}
                    exit={{ opacity: 0, height: 0 }}
                    className="overflow-hidden"
                  >
                    <div className="px-4 py-4 bg-muted/20 border-b border-border space-y-4">
                      {/* Theme Selector */}
                      <div>
                        <Label className="text-xs font-medium mb-2 block">থিম সিলেক্ট করুন:</Label>
                        <div className="flex gap-2 flex-wrap">
                          {themeVariants.map((v) => (
                            <button key={v.name} onClick={() => applyTheme(tpl.key, v)}
                              className={`px-3 py-2 rounded-xl text-xs font-medium transition-all border-2 flex items-center gap-1.5 ${
                                (tpl.selectedTheme || "💜 পার্পল") === v.name
                                  ? "border-primary bg-primary/5"
                                  : "border-border hover:border-primary/40"
                              }`}>
                              {(tpl.selectedTheme || "💜 পার্পল") === v.name && <CheckCircle2 className="h-3 w-3 text-primary" />}
                              {v.name}
                            </button>
                          ))}
                        </div>
                      </div>

                      {/* Subject */}
                      <div className="space-y-1.5">
                        <Label className="text-xs">Subject</Label>
                        <Input value={tpl.subject || tpl.defaultSubject} onChange={e => updateTemplate(tpl.key, { subject: e.target.value })} className="rounded-xl text-sm" />
                      </div>

                      {/* Tags */}
                      <p className="text-[10px] text-muted-foreground">Tags: {tpl.tags}</p>

                      {/* Preview */}
                      <div>
                        <Button variant="outline" size="sm" className="gap-1.5 rounded-xl text-xs mb-2"
                          onClick={() => setPreviewTemplate(previewTemplate === tpl.key ? null : tpl.key)}>
                          <Eye className="h-3.5 w-3.5" />
                          {previewTemplate === tpl.key ? "প্রিভিউ বন্ধ" : "প্রিভিউ দেখুন"}
                        </Button>

                        <AnimatePresence>
                          {previewTemplate === tpl.key && (
                            <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }} className="overflow-hidden">
                              <div className="rounded-xl border border-border overflow-hidden bg-muted/30">
                                <div className="px-3 py-2 bg-muted border-b border-border flex items-center gap-2">
                                  <div className="flex gap-1">
                                    <div className="w-2.5 h-2.5 rounded-full bg-red-400" />
                                    <div className="w-2.5 h-2.5 rounded-full bg-yellow-400" />
                                    <div className="w-2.5 h-2.5 rounded-full bg-green-400" />
                                  </div>
                                  <span className="text-[10px] text-muted-foreground font-mono">Email Preview</span>
                                </div>
                                <div className="p-4 bg-background">
                                  <p className="text-xs text-muted-foreground mb-2"><strong>Subject:</strong> {(tpl.subject || tpl.defaultSubject).replace(/\{product\}/g, "ChatGPT Plus").replace(/\{order_id\}/g, "ORD-12345")}</p>
                                  <div className="border border-border rounded-lg overflow-hidden" dangerouslySetInnerHTML={{ __html: getPreviewHtml(tpl) }} />
                                </div>
                              </div>
                            </motion.div>
                          )}
                        </AnimatePresence>
                      </div>
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          ))}
        </div>

        <Button onClick={handleSaveTemplates} disabled={savingTemplates} className="gap-1.5 rounded-xl mt-4">
          {savingTemplates ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
          {savingTemplates ? "সেভ হচ্ছে..." : "Save All Templates"}
        </Button>
      </SectionCard>

      {/* Send Test Email */}
      <SectionCard title="Send Test Email" description="SMTP কনফিগারেশন টেস্ট করুন" icon={Mail}>
        <div className="flex gap-2 items-end">
          <div className="flex-1 space-y-1.5">
            <Label className="text-xs font-medium">ইমেইল অ্যাড্রেস</Label>
            <Input type="email" value={testEmail} onChange={e => setTestEmail(e.target.value)} placeholder="test@example.com" className="rounded-xl" />
          </div>
          <Button onClick={handleTestEmail} disabled={testSending} className="gap-1.5 rounded-xl">
            {testSending ? <Loader2 className="h-4 w-4 animate-spin" /> : <Mail className="h-4 w-4" />}
            {testSending ? "পাঠানো হচ্ছে..." : "Send Test"}
          </Button>
        </div>
      </SectionCard>

      {/* Help */}
      <SectionCard title="সাহায্য" description="SMTP গাইড" icon={AlertCircle}>
        <div className="space-y-2 text-sm text-muted-foreground">
          <p><strong className="text-foreground">Hostinger:</strong> Host: <code className="bg-muted px-1 rounded text-xs">smtp.hostinger.com</code>, Port: <code className="bg-muted px-1 rounded text-xs">465</code> (SSL) / <code className="bg-muted px-1 rounded text-xs">587</code> (TLS)</p>
          <p className="text-xs">⚠️ SSL কাজ না করলে TLS ব্যবহার করুন।</p>
        </div>
      </SectionCard>
    </div>
  );
};


const Admin = () => {
  const [activeTab, setActiveTab] = useState<Tab>("dashboard");
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [products, setProducts] = useState<DBProduct[]>([]);
  const [orders, setOrders] = useState<DBOrder[]>([]);
  const [banners, setBanners] = useState<DBBanner[]>([]);
  const [categories, setCategories] = useState<DBCategory[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingProduct, setEditingProduct] = useState<DBProduct | null>(null);
  const [editingBanner, setEditingBanner] = useState<DBBanner | null>(null);
  const [editingCategory, setEditingCategory] = useState<DBCategory | null>(null);
  const [showAddForm, setShowAddForm] = useState(false);
  const [showAddBanner, setShowAddBanner] = useState(false);
  const [showAddCategory, setShowAddCategory] = useState(false);
  const [newCategory, setNewCategory] = useState({ name: "", sort_order: 0 });
  const [newProduct, setNewProduct] = useState({ name: "", price: "", category: "", description: "", long_description: "", image_url: "", original_price: "", reseller_price: "", duration_options: [] as string[], custom_fields: [] as { label: string; placeholder: string; required: boolean; helper_text: string }[], features: [] as string[] });
  const [newFeatureText, setNewFeatureText] = useState("");
  const [editFeatureText, setEditFeatureText] = useState("");
  const [newProductResellerPrices, setNewProductResellerPrices] = useState<Record<string, string>>({});
  const [newVersionLabel, setNewVersionLabel] = useState("");
  const [newVersionPrice, setNewVersionPrice] = useState("");
  const [newVersionResellerPrice, setNewVersionResellerPrice] = useState("");
  const [editVersionLabel, setEditVersionLabel] = useState("");
  const [editVersionPrice, setEditVersionPrice] = useState("");
  const [editVersionResellerPrice, setEditVersionResellerPrice] = useState("");
  const [editingVersionIdx, setEditingVersionIdx] = useState<number | null>(null);
  const [editingVersionLabelInline, setEditingVersionLabelInline] = useState("");
  const [editingVersionPriceInline, setEditingVersionPriceInline] = useState("");
  const [editingVersionResellerPriceInline, setEditingVersionResellerPriceInline] = useState("");
  const [editProductResellerPrices, setEditProductResellerPrices] = useState<Record<string, string>>({});
  const [uploadingProductImage, setUploadingProductImage] = useState(false);
  const productImageRef = useRef<HTMLInputElement>(null);
  const editProductImageRef = useRef<HTMLInputElement>(null);
  const [newBanner, setNewBanner] = useState({ title: "", subtitle: "", description: "", cta_text: "Shop Now", cta_link: "#products", secondary_cta_text: "Best Selling", secondary_cta_link: "#best-selling", image_url: "" });
  const [searchQuery, setSearchQuery] = useState("");
  const [orderFilter, setOrderFilter] = useState<"all" | "pending" | "hold" | "completed" | "cancelled" | "trashed">("all");
  const [selectedOrders, setSelectedOrders] = useState<Set<string>>(new Set());
  const [viewingOrder, setViewingOrder] = useState<DBOrder | null>(null);
  const [viewingOrderEmail, setViewingOrderEmail] = useState<string | null>(null);
  const [showAddOrder, setShowAddOrder] = useState(false);
  const [newOrder, setNewOrder] = useState({ customer_name: "", customer_phone: "", product_name: "", amount: "", payment_method: "manual", payment_status: "unpaid", status: "processing" });
  const [confirmDelete, setConfirmDelete] = useState<{ type: string; id: string; name: string } | null>(null);
  const [customLinks, setCustomLinks] = useState<{ label: string; url: string; icon: string }[]>([]);
  const [newCustomLink, setNewCustomLink] = useState({ label: "", url: "", icon: "🔗" });
  const [showAddCustomLink, setShowAddCustomLink] = useState(false);
  const navigate = useNavigate();
  const { toast } = useToast();

  const checkAuth = useCallback(async () => {
    const { data: { session } } = await supabase.auth.getSession();
    if (!session) { navigate("/admin-login"); return; }
    const { data: roleData } = await supabase
      .from("user_roles").select("role").eq("user_id", session.user.id).eq("role", "admin").maybeSingle();
    if (!roleData) { navigate("/admin-login"); return; }
    fetchData();
  }, [navigate]);

  const fetchData = async () => {
    setLoading(true);
    const [prodRes, orderRes, bannerRes, catRes] = await Promise.all([
      supabase.from("products").select("*").order("created_at", { ascending: false }),
      supabase.from("orders").select("*").order("created_at", { ascending: false }),
      supabase.from("banners").select("*").order("created_at", { ascending: false }),
      supabase.from("categories").select("*").order("sort_order", { ascending: true }),
    ]);
    if (prodRes.data) setProducts(prodRes.data as DBProduct[]);
    if (orderRes.data) setOrders(orderRes.data as DBOrder[]);
    if (bannerRes.data) setBanners(bannerRes.data as DBBanner[]);
    if (catRes.data) setCategories(catRes.data as DBCategory[]);
    setLoading(false);
  };

  useEffect(() => { checkAuth(); }, [checkAuth]);

  const handleLogout = async () => {
    await supabase.auth.signOut();
    navigate("/admin-login");
  };

  // ── Product Image Upload ──
  const uploadProductImage = async (file: File): Promise<string | null> => {
    setUploadingProductImage(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `product-${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("products").upload(fileName, file, { upsert: true });
      if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return null; }
      const { data: urlData } = supabase.storage.from("products").getPublicUrl(fileName);
      return urlData.publicUrl;
    } finally { setUploadingProductImage(false); }
  };

  const handleNewProductImage = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    const url = await uploadProductImage(file);
    if (url) setNewProduct(p => ({ ...p, image_url: url }));
  };

  const handleEditProductImage = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    const url = await uploadProductImage(file);
    if (url) setEditingProduct(p => p ? { ...p, image_url: url } : null);
  };

  // ── Product CRUD ──
  const handleAddProduct = async () => {
    if (!newProduct.name || !newProduct.category) {
      toast({ title: "নাম ও ক্যাটাগরি পূরণ করুন", variant: "destructive" }); return;
    }
    // Auto-derive price from first duration option if no manual price
    const autoPrice = newProduct.duration_options.length > 0 ? (parseDurationOption(newProduct.duration_options[0]).price || newProduct.price || "০") : (newProduct.price || "০");
    let slug = newProduct.name.toLowerCase().replace(/[^a-z0-9\u0980-\u09FF\s-]/g, '').replace(/\s+/g, '-').replace(/^-+|-+$/g, '');
    if (!slug) slug = `product-${Date.now()}`;
    const { error } = await supabase.from("products").insert({
      name: newProduct.name, price: autoPrice, category: newProduct.category,
      description: newProduct.description, long_description: newProduct.long_description || null,
      image_url: newProduct.image_url || null,
      original_price: newProduct.original_price || null,
      reseller_price: newProduct.reseller_price || null,
      reseller_prices: Object.keys(newProductResellerPrices).length > 0 ? newProductResellerPrices : null,
      duration_options: newProduct.duration_options.length > 0 ? newProduct.duration_options : null,
      features: newProduct.features.filter(f => f?.trim()).length > 0 ? newProduct.features.filter(f => f?.trim()) : null,
      custom_fields: newProduct.custom_fields.length > 0 ? newProduct.custom_fields : null,
      slug,
    } as any);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ প্রোডাক্ট যোগ হয়েছে!" });
    setNewProduct({ name: "", price: "", category: "", description: "", long_description: "", image_url: "", original_price: "", reseller_price: "", duration_options: [], custom_fields: [], features: [] });
    setNewFeatureText("");
    setNewProductResellerPrices({});
    setNewVersionLabel(""); setNewVersionPrice(""); setNewVersionResellerPrice("");
    setShowAddForm(false);
    fetchData();
  };

  const handleDeleteProduct = async (id: string) => {
    const { error } = await supabase.from("products").delete().eq("id", id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "🗑️ প্রোডাক্ট ডিলিট হয়েছে!" });
    setConfirmDelete(null);
    fetchData();
  };

  const handleUpdateProduct = async () => {
    if (!editingProduct) return;
    const dOpts = editingProduct.duration_options || [];
    const feats = (editingProduct.features || []).map((f: string) => f?.trim()).filter(Boolean);
    const autoPrice = dOpts.length > 0 ? (parseDurationOption(dOpts[0]).price || editingProduct.price || "০") : (editingProduct.price || "০");
    const { error } = await supabase.from("products").update({
      name: editingProduct.name, price: autoPrice, category: editingProduct.category,
      description: editingProduct.description, long_description: (editingProduct as any).long_description || null,
      image_url: editingProduct.image_url,
      original_price: editingProduct.original_price,
      reseller_price: null,
      reseller_prices: Object.keys(editProductResellerPrices).length > 0 ? editProductResellerPrices : null,
      duration_options: dOpts.length > 0 ? dOpts : null,
      features: feats.length > 0 ? feats : null,
      custom_fields: (editingProduct as any).custom_fields?.length > 0 ? (editingProduct as any).custom_fields : null,
    } as any).eq("id", editingProduct.id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ প্রোডাক্ট আপডেট হয়েছে!" });
    setEditingProduct(null);
    setEditVersionLabel(""); setEditVersionPrice(""); setEditVersionResellerPrice("");
    setEditProductResellerPrices({});
    fetchData();
  };

  // ── Order ──
  const sendSms = async (number: string, message: string) => {
    try {
      const { data, error } = await supabase.functions.invoke("send-sms", {
        body: { number, message },
      });
      if (error) console.error("SMS error:", error);
      return data;
    } catch (e) {
      console.error("SMS send failed:", e);
    }
  };

  const handleOrderStatus = async (id: string, status: string) => {
    const order = orders.find(o => o.id === id);
    const { error } = await supabase.from("orders").update({ status }).eq("id", id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    const statusLabels: Record<string, string> = { completed: "সম্পন্ন", cancelled: "বাতিল", processing: "প্রসেসিং", pending: "পেন্ডিং", hold: "হোল্ড" };
    toast({ title: `✅ অর্ডার ${statusLabels[status] || status} হয়েছে!` });

    // Send SMS based on saved templates
    if (order?.customer_phone) {
      try {
        const { data: templateSetting } = await supabase.from("site_settings").select("value").eq("key", "sms_templates").maybeSingle();
        let templates: { key: string; enabled: boolean; message: string }[] = [];
        if (templateSetting?.value) {
          try { templates = JSON.parse(templateSetting.value); } catch {}
        }
        const template = templates.find(t => t.key === status);
        if (template?.enabled) {
          const smsReplacements: Record<string, string> = {
            name: order.customer_name, order_id: order.id.slice(0, 8),
            items: order.product_name, total: order.amount,
            phone: order.customer_phone || "", email: "",
            billing_email: "", billing_phone: order.customer_phone || "",
            status: statusLabels[status] || status,
            shop_name: siteFields.site_title, shop_email: siteFields.email,
            shop_url: window.location.origin, shop_phone: siteFields.phone,
            order_status: statusLabels[status] || status, order_amount: order.amount,
            order_product: order.product_name, order_product_with_qty: order.product_name,
            order_currency: "৳",
          };
          let msg = template.message || "";
          for (const [key, val] of Object.entries(smsReplacements)) {
            msg = msg.replace(new RegExp(`\\{${key}\\}|\\[${key}\\]`, "g"), val);
          }
          if (msg) sendSms(order.customer_phone, msg);
        } else if (!template || templates.length === 0) {
          // Only send fallback if NO templates are saved at all (first-time setup)
          const smsMsg = status === "completed"
            ? `প্রিয় ${order.customer_name}, আপনার অর্ডার "${order.product_name}" সম্পন্ন হয়েছে। ধন্যবাদ!`
            : null;
          if (smsMsg) sendSms(order.customer_phone, smsMsg);
        }
        // If template exists but is disabled, do NOT send any SMS
      } catch (e) {
        console.error("Template SMS error:", e);
      }

      // Also send WhatsApp notification based on templates
      try {
        const { data: waTemplateSetting } = await supabase.from("site_settings").select("value").eq("key", "whatsapp_templates").maybeSingle();
        let waTemplates: { key: string; enabled: boolean; message: string }[] = [];
        if (waTemplateSetting?.value) {
          try { waTemplates = JSON.parse(waTemplateSetting.value); } catch {}
        }
        const waTemplate = waTemplates.find(t => t.key === status);
        if (waTemplate?.enabled) {
          const waReplacements: Record<string, string> = {
            name: order.customer_name, order_id: order.id.slice(0, 8),
            items: order.product_name, total: order.amount,
            phone: order.customer_phone || "", email: "",
            billing_email: "", billing_phone: order.customer_phone || "",
            status: statusLabels[status] || status,
            shop_name: siteFields.site_title, shop_email: siteFields.email,
            shop_url: window.location.origin, shop_phone: siteFields.phone,
            order_status: statusLabels[status] || status, order_amount: order.amount,
            order_product: order.product_name, order_product_with_qty: order.product_name,
            order_currency: "৳",
          };
          let waMsg = waTemplate.message || "";
          for (const [key, val] of Object.entries(waReplacements)) {
            waMsg = waMsg.replace(new RegExp(`\\{${key}\\}|\\[${key}\\]`, "g"), val);
          }
          if (waMsg) {
            supabase.functions.invoke("send-whatsapp", {
              body: { number: order.customer_phone, message: waMsg },
            }).catch(e => console.error("WhatsApp send error:", e));
          }
        }
      } catch (e) {
        console.error("WhatsApp template error:", e);
      }
    }

    fetchData();
  };

  // ── Banner ──
  const [uploadingImage, setUploadingImage] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const editFileInputRef = useRef<HTMLInputElement>(null);

  const uploadBannerImage = async (file: File): Promise<string | null> => {
    setUploadingImage(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("banners").upload(fileName, file, { upsert: true });
      if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return null; }
      const { data: urlData } = supabase.storage.from("banners").getPublicUrl(fileName);
      return urlData.publicUrl;
    } finally { setUploadingImage(false); }
  };

  const handleNewBannerImage = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    const url = await uploadBannerImage(file);
    if (url) setNewBanner(b => ({ ...b, image_url: url }));
  };

  const handleEditBannerImage = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    const url = await uploadBannerImage(file);
    if (url) setEditingBanner(b => b ? { ...b, image_url: url } : null);
  };

  const handleAddBanner = async () => {
    if (!newBanner.title) { toast({ title: "টাইটেল দিন", variant: "destructive" }); return; }
    const { error } = await supabase.from("banners").insert({
      title: newBanner.title, subtitle: newBanner.subtitle || null, description: newBanner.description || null,
      cta_text: newBanner.cta_text || null, cta_link: newBanner.cta_link || null,
      secondary_cta_text: newBanner.secondary_cta_text || null, secondary_cta_link: newBanner.secondary_cta_link || null,
      image_url: newBanner.image_url || null,
    });
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ ব্যানার যোগ হয়েছে!" });
    setNewBanner({ title: "", subtitle: "", description: "", cta_text: "Shop Now", cta_link: "#products", secondary_cta_text: "Best Selling", secondary_cta_link: "#best-selling", image_url: "" });
    setShowAddBanner(false);
    fetchData();
  };

  const handleUpdateBanner = async () => {
    if (!editingBanner) return;
    const { error } = await supabase.from("banners").update({
      title: editingBanner.title, subtitle: editingBanner.subtitle, description: editingBanner.description,
      cta_text: editingBanner.cta_text, cta_link: editingBanner.cta_link,
      secondary_cta_text: editingBanner.secondary_cta_text, secondary_cta_link: editingBanner.secondary_cta_link,
      image_url: editingBanner.image_url, is_active: editingBanner.is_active,
    }).eq("id", editingBanner.id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ ব্যানার আপডেট হয়েছে!" });
    setEditingBanner(null);
    fetchData();
  };

  const handleDeleteBanner = async (id: string) => {
    const { error } = await supabase.from("banners").delete().eq("id", id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "🗑️ ব্যানার ডিলিট হয়েছে!" });
    setConfirmDelete(null);
    fetchData();
  };

  const handleToggleBannerActive = async (banner: DBBanner) => {
    const { error } = await supabase.from("banners").update({ is_active: !banner.is_active }).eq("id", banner.id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    fetchData();
  };

  // ── Categories ──
  const handleAddCategory = async () => {
    if (!newCategory.name.trim()) { toast({ title: "ক্যাটাগরির নাম দিন", variant: "destructive" }); return; }
    const { error } = await supabase.from("categories").insert({ name: newCategory.name, sort_order: newCategory.sort_order });
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ ক্যাটাগরি যোগ হয়েছে!" });
    setNewCategory({ name: "", sort_order: 0 });
    setShowAddCategory(false);
    fetchData();
  };

  const handleUpdateCategory = async () => {
    if (!editingCategory) return;
    const { error } = await supabase.from("categories").update({
      name: editingCategory.name, sort_order: editingCategory.sort_order, is_active: editingCategory.is_active,
    }).eq("id", editingCategory.id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ ক্যাটাগরি আপডেট হয়েছে!" });
    setEditingCategory(null);
    fetchData();
  };

  const handleDeleteCategory = async (id: string) => {
    const { error } = await supabase.from("categories").delete().eq("id", id);
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "🗑️ ক্যাটাগরি ডিলিট হয়েছে!" });
    setConfirmDelete(null);
    fetchData();
  };

  // ── Settings ──
  const [logoUrl, setLogoUrl] = useState<string | null>(null);
  const [paymentBannerUrl, setPaymentBannerUrl] = useState<string | null>(null);
  const [trustBadgeUrl, setTrustBadgeUrl] = useState<string | null>(null);
  const [uploadingLogo, setUploadingLogo] = useState(false);
  const [uploadingFavicon, setUploadingFavicon] = useState(false);
  const [uploadingPaymentBanner, setUploadingPaymentBanner] = useState(false);
  const [savingSettings, setSavingSettings] = useState(false);
  const [devCreditUnlocked, setDevCreditUnlocked] = useState(false);
  const [devCreditPassword, setDevCreditPassword] = useState("");
  const [storedDevPassword, setStoredDevPassword] = useState<string | null>(null);
  const logoFileRef = useRef<HTMLInputElement>(null);
  const faviconFileRef = useRef<HTMLInputElement>(null);
  const paymentBannerRef = useRef<HTMLInputElement>(null);
  const appIconRef = useRef<HTMLInputElement>(null);
  const pwaIconRef = useRef<HTMLInputElement>(null);
  const [uploadingAppIcon, setUploadingAppIcon] = useState(false);
  const [uploadingPwaIcon, setUploadingPwaIcon] = useState(false);
  const [siteFields, setSiteFields] = useState({
    site_title: "TZSS", site_description: "Unlock unlimited Digital Entertainment",
    favicon_url: "", phone: "+880 1851-182700", email: "info@tzssbd.com",
    facebook_url: "", whatsapp_number: "", address: "Dhaka, Bangladesh",
    tzsmmpay_api_key: "", payment_gateway_url: "pay.tzssbd.com",
    developer_name: "TEAM RONY", developer_url: "https://hanifahmedrony.site",
    telegram_bot_token: "", telegram_chat_id: "",
    support_whatsapp: "", support_messenger: "", support_telegram: "",
    app_install_enabled: "", app_install_title: "", app_install_subtitle: "", app_install_link: "", app_install_image: "",
    pwa_icon_url: "",
  });

  useEffect(() => {
    const fetchSettings = async () => {
      const { data } = await supabase.from("site_settings").select("key, value");
      if (data) {
        const map: Record<string, string> = {};
        data.forEach((r) => { if (r.value) map[r.key] = r.value; });
        if (map.logo_url) setLogoUrl(map.logo_url);
        if (map.payment_banner_url) setPaymentBannerUrl(map.payment_banner_url);
        if (map.trust_badge_url) setTrustBadgeUrl(map.trust_badge_url);
        setSiteFields(f => ({
          ...f, site_title: map.site_title || f.site_title, site_description: map.site_description || f.site_description,
          favicon_url: map.favicon_url || "", phone: map.phone || f.phone, email: map.email || f.email,
          facebook_url: map.facebook_url || "", whatsapp_number: map.whatsapp_number || "",
          address: map.address || f.address, tzsmmpay_api_key: map.tzsmmpay_api_key || "",
          payment_gateway_url: map.payment_gateway_url || f.payment_gateway_url,
          developer_name: map.developer_name || f.developer_name,
          developer_url: map.developer_url || f.developer_url,
          telegram_bot_token: map.telegram_bot_token || "",
          telegram_chat_id: map.telegram_chat_id || "",
          support_whatsapp: map.support_whatsapp || "",
          support_messenger: map.support_messenger || "",
          support_telegram: map.support_telegram || "",
          app_install_enabled: map.app_install_enabled || "",
          app_install_title: map.app_install_title || "",
          app_install_subtitle: map.app_install_subtitle || "",
          app_install_link: map.app_install_link || "",
          app_install_image: map.app_install_image || "",
          pwa_icon_url: map.pwa_icon_url || "",
        }));
        if (map.sidebar_custom_links) {
          try { setCustomLinks(JSON.parse(map.sidebar_custom_links)); } catch {}
        }
        if (map.developer_password) setStoredDevPassword(map.developer_password);
      }
    };
    fetchSettings();
  }, []);

  const handleLogoUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    setUploadingLogo(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `logo-${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("logos").upload(fileName, file, { upsert: true });
      if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return; }
      const { data: urlData } = supabase.storage.from("logos").getPublicUrl(fileName);
      const url = urlData.publicUrl;
      const { data: existing } = await supabase.from("site_settings").select("id").eq("key", "logo_url").maybeSingle();
      if (existing) { await supabase.from("site_settings").update({ value: url, updated_at: new Date().toISOString() }).eq("key", "logo_url"); }
      else { await supabase.from("site_settings").insert({ key: "logo_url", value: url }); }
      setLogoUrl(url);
      toast({ title: "✅ লোগো আপডেট হয়েছে!" });
    } finally { setUploadingLogo(false); }
  };

  const handleRemoveLogo = async () => {
    await supabase.from("site_settings").delete().eq("key", "logo_url");
    setLogoUrl(null);
    toast({ title: "🗑️ লোগো রিমুভ হয়েছে!" });
  };

  const handleFaviconUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    setUploadingFavicon(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `favicon-${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("logos").upload(fileName, file, { upsert: true });
      if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return; }
      const { data: urlData } = supabase.storage.from("logos").getPublicUrl(fileName);
      setSiteFields(f => ({ ...f, favicon_url: urlData.publicUrl }));
    } finally { setUploadingFavicon(false); }
  };

  const handlePaymentBannerUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]; if (!file) return;
    setUploadingPaymentBanner(true);
    try {
      const ext = file.name.split(".").pop();
      const fileName = `payment-banner-${Date.now()}.${ext}`;
      const { error } = await supabase.storage.from("logos").upload(fileName, file, { upsert: true });
      if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return; }
      const { data: urlData } = supabase.storage.from("logos").getPublicUrl(fileName);
      const url = urlData.publicUrl;
      const { data: existing } = await supabase.from("site_settings").select("id").eq("key", "payment_banner_url").maybeSingle();
      if (existing) { await supabase.from("site_settings").update({ value: url, updated_at: new Date().toISOString() }).eq("key", "payment_banner_url"); }
      else { await supabase.from("site_settings").insert({ key: "payment_banner_url", value: url }); }
      setPaymentBannerUrl(url);
      toast({ title: "✅ পেমেন্ট ব্যানার আপলোড হয়েছে!" });
    } finally { setUploadingPaymentBanner(false); }
  };

  const upsertSetting = async (key: string, value: string) => {
    const { data: existing } = await supabase.from("site_settings").select("id").eq("key", key).maybeSingle();
    if (existing) { await supabase.from("site_settings").update({ value, updated_at: new Date().toISOString() }).eq("key", key); }
    else { await supabase.from("site_settings").insert({ key, value }); }
  };

  const handleSaveSettings = async () => {
    setSavingSettings(true);
    try {
      for (const [key, value] of Object.entries(siteFields)) {
        if (value) await upsertSetting(key, value);
      }
      if (paymentBannerUrl) await upsertSetting("payment_banner_url", paymentBannerUrl);
      if (trustBadgeUrl) await upsertSetting("trust_badge_url", trustBadgeUrl);
      toast({ title: "✅ সেটিংস সেভ হয়েছে!" });
    } catch { toast({ title: "এরর হয়েছে", variant: "destructive" }); }
    finally { setSavingSettings(false); }
  };

  // ── Computed ──
  const completedOrders = orders.filter(o => o.status === "completed" && !o.is_trashed).length;
  const pendingOrders = orders.filter(o => o.status === "pending" && !o.is_trashed).length;
  const totalRevenue = orders.filter(o => o.status === "completed" && !o.is_trashed).reduce((sum, o) => {
    const num = parseFloat(o.amount.replace(/[^\d.]/g, ""));
    return sum + (isNaN(num) ? 0 : num);
  }, 0);

  const filteredProducts = useMemo(() => {
    if (!searchQuery) return products;
    return products.filter(p => p.name.toLowerCase().includes(searchQuery.toLowerCase()) || p.category.toLowerCase().includes(searchQuery.toLowerCase()));
  }, [products, searchQuery]);

  const filteredOrders = useMemo(() => {
    let filtered = orders;
    if (orderFilter === "trashed") filtered = filtered.filter(o => o.is_trashed);
    else {
      filtered = filtered.filter(o => !o.is_trashed);
      if (orderFilter !== "all") filtered = filtered.filter(o => o.status === orderFilter);
    }
    if (searchQuery) filtered = filtered.filter(o => o.customer_name.toLowerCase().includes(searchQuery.toLowerCase()) || o.product_name.toLowerCase().includes(searchQuery.toLowerCase()));
    return filtered;
  }, [orders, orderFilter, searchQuery]);

  const trashedCount = orders.filter(o => o.is_trashed).length;
  const activeOrders = orders.filter(o => !o.is_trashed);

  const toggleSelectOrder = (id: string) => {
    setSelectedOrders(prev => {
      const next = new Set(prev);
      if (next.has(id)) next.delete(id); else next.add(id);
      return next;
    });
  };

  const toggleSelectAll = () => {
    if (selectedOrders.size === filteredOrders.length) setSelectedOrders(new Set());
    else setSelectedOrders(new Set(filteredOrders.map(o => o.id)));
  };

  const handleBulkAction = async (action: "trash" | "restore" | "delete" | "completed" | "cancelled") => {
    if (selectedOrders.size === 0) return;
    const ids = Array.from(selectedOrders);
    if (action === "trash") {
      await supabase.from("orders").update({ is_trashed: true } as any).in("id", ids);
      toast({ title: `🗑️ ${ids.length}টি অর্ডার ট্র্যাশে সরানো হয়েছে` });
    } else if (action === "restore") {
      await supabase.from("orders").update({ is_trashed: false } as any).in("id", ids);
      toast({ title: `♻️ ${ids.length}টি অর্ডার রিস্টোর হয়েছে` });
    } else if (action === "delete") {
      await supabase.from("orders").delete().in("id", ids);
      toast({ title: `🗑️ ${ids.length}টি অর্ডার স্থায়ীভাবে ডিলিট হয়েছে` });
    } else {
      for (const id of ids) await handleOrderStatus(id, action);
    }
    setSelectedOrders(new Set());
    fetchData();
  };

  const handleTrashOrder = async (id: string) => {
    await supabase.from("orders").update({ is_trashed: true } as any).eq("id", id);
    toast({ title: "🗑️ অর্ডার ট্র্যাশে সরানো হয়েছে" });
    fetchData();
  };

  const handleRestoreOrder = async (id: string) => {
    await supabase.from("orders").update({ is_trashed: false } as any).eq("id", id);
    toast({ title: "♻️ অর্ডার রিস্টোর হয়েছে" });
    fetchData();
  };

  const handlePermanentDelete = async (id: string) => {
    await supabase.from("orders").delete().eq("id", id);
    toast({ title: "🗑️ অর্ডার স্থায়ীভাবে ডিলিট হয়েছে" });
    fetchData();
  };

  const handleAddOrder = async () => {
    if (!newOrder.customer_name || !newOrder.product_name || !newOrder.amount) {
      toast({ title: "কাস্টমার নাম, প্রোডাক্ট এবং মূল্য আবশ্যক", variant: "destructive" }); return;
    }
    const { error } = await supabase.from("orders").insert({
      customer_name: newOrder.customer_name,
      customer_phone: newOrder.customer_phone || null,
      product_name: newOrder.product_name,
      amount: newOrder.amount,
      payment_method: newOrder.payment_method,
      payment_status: newOrder.payment_status,
      status: newOrder.status,
    });
    if (error) { toast({ title: "এরর", description: error.message, variant: "destructive" }); return; }
    toast({ title: "✅ অর্ডার যোগ হয়েছে!" });
    setNewOrder({ customer_name: "", customer_phone: "", product_name: "", amount: "", payment_method: "manual", payment_status: "unpaid", status: "processing" });
    setShowAddOrder(false);
    fetchData();
  };

  const statusColor = (s: string) =>
    s === "completed" ? "bg-primary/10 text-primary" : s === "pending" ? "bg-amber-500/10 text-amber-600" : s === "hold" ? "bg-orange-500/10 text-orange-600" : s === "processing" ? "bg-blue-500/10 text-blue-600" : "bg-destructive/10 text-destructive";
  const statusText = (s: string) =>
    s === "completed" ? "সম্পন্ন" : s === "pending" ? "পেন্ডিং" : s === "hold" ? "হোল্ড" : s === "processing" ? "প্রসেসিং" : "বাতিল";

  const navItems: { tab: Tab; label: string; icon: typeof Package }[] = [
    { tab: "dashboard", label: "ড্যাশবোর্ড", icon: LayoutDashboard },
    { tab: "products", label: "প্রোডাক্ট", icon: Package },
    { tab: "categories", label: "ক্যাটাগরি", icon: Tag },
    { tab: "orders", label: "অর্ডার", icon: ShoppingCart },
    { tab: "incomplete_orders", label: "ইনকম্পলিট অর্ডার", icon: AlertCircle },
    { tab: "banner", label: "ব্যানার", icon: Image },
    { tab: "popup", label: "পপআপ", icon: Eye },
    { tab: "product_keys", label: "প্রোডাক্ট কী", icon: CreditCard },
    { tab: "partners", label: "পার্টনার", icon: Handshake },
    { tab: "blog", label: "ব্লগ", icon: FileText },
    { tab: "product_requests", label: "রিকোয়েস্ট", icon: Inbox },
    { tab: "promo_codes", label: "প্রোমো কোড", icon: Tag },
    { tab: "reviews", label: "রিভিউ", icon: Star },
    { tab: "faq", label: "FAQ", icon: HelpCircle },
    { tab: "users", label: "ইউজার", icon: Users },
    { tab: "subscribers", label: "সাবস্ক্রাইবার", icon: Mail },
    { tab: "sms", label: "SMS", icon: MessageCircle },
    { tab: "whatsapp", label: "WhatsApp", icon: Phone },
    { tab: "email", label: "ইমেইল", icon: Mail },
    { tab: "voice_call", label: "ভয়েস কল", icon: Phone },
    { tab: "auth_settings", label: "অথেন্টিকেশন", icon: Shield },
    { tab: "pixel", label: "Facebook Pixel", icon: Eye },
    { tab: "settings", label: "সেটিংস", icon: Settings },
  ];

  const handleConfirmDelete = () => {
    if (!confirmDelete) return;
    if (confirmDelete.type === "product") handleDeleteProduct(confirmDelete.id);
    else if (confirmDelete.type === "banner") handleDeleteBanner(confirmDelete.id);
    else if (confirmDelete.type === "category") handleDeleteCategory(confirmDelete.id);
  };

  return (
    <div className="min-h-screen bg-muted/30 flex">
      {/* ── Confirmation Dialog ── */}
      <ConfirmDialog
        open={!!confirmDelete}
        title="ডিলিট নিশ্চিত করুন"
        message={`"${confirmDelete?.name}" ডিলিট করতে চান? এটি পূর্বাবস্থায় ফেরানো যাবে না।`}
        onConfirm={handleConfirmDelete}
        onCancel={() => setConfirmDelete(null)}
      />

      {/* ── Sidebar ── */}
      <aside className={`fixed inset-y-0 left-0 z-50 w-72 bg-card border-r border-border transform transition-transform duration-300 ease-in-out lg:translate-x-0 ${sidebarOpen ? "translate-x-0" : "-translate-x-full"}`}>
        <div className="flex flex-col h-full">
          {/* Brand */}
          <div className="p-5 border-b border-border">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <div className="h-10 w-10 rounded-xl bg-primary flex items-center justify-center">
                  <Shield className="h-5 w-5 text-primary-foreground" />
                </div>
                <div>
                  <h1 className="font-heading text-base font-bold leading-tight">Admin Panel</h1>
                  <p className="text-[11px] text-muted-foreground">{siteFields.site_title}</p>
                </div>
              </div>
              <button className="lg:hidden p-1 rounded-lg hover:bg-muted" onClick={() => setSidebarOpen(false)}>
                <X className="h-5 w-5" />
              </button>
            </div>
          </div>

          {/* Nav */}
          <nav className="flex-1 p-3 space-y-1 overflow-y-auto">
            <p className="px-3 py-2 text-[10px] font-semibold text-muted-foreground uppercase tracking-wider">মেইন মেনু</p>
            {navItems.map((item) => (
              <button
                key={item.tab}
                onClick={() => { setActiveTab(item.tab); setSidebarOpen(false); setSearchQuery(""); }}
                className={`w-full flex items-center gap-3 px-3 py-2.5 rounded-xl text-sm font-medium transition-all duration-200
                  ${activeTab === item.tab
                    ? "bg-primary text-primary-foreground shadow-md shadow-primary/20"
                    : "text-muted-foreground hover:bg-muted hover:text-foreground"
                  }`}
              >
                <item.icon className="h-[18px] w-[18px]" />
                <span>{item.label}</span>
                {item.tab === "orders" && pendingOrders > 0 && (
                  <span className="ml-auto bg-amber-500 text-white text-[10px] font-bold px-1.5 py-0.5 rounded-full">{pendingOrders}</span>
                )}
              </button>
            ))}
          </nav>

          {/* Custom Links */}
          {customLinks.length > 0 && (
            <div className="px-3 pb-2">
              <p className="px-3 py-2 text-[10px] font-semibold text-muted-foreground uppercase tracking-wider">কাস্টম লিংক</p>
              <div className="space-y-1">
                {customLinks.map((link, i) => (
                  <a
                    key={i}
                    href={link.url}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="flex items-center gap-3 px-3 py-2.5 text-sm text-muted-foreground hover:text-foreground rounded-xl hover:bg-muted transition-all"
                  >
                    <span className="text-base">{link.icon}</span>
                    <span>{link.label}</span>
                    <ExternalLink className="h-3 w-3 ml-auto opacity-50" />
                  </a>
                ))}
              </div>
            </div>
          )}

          {/* Bottom */}
          <div className="p-3 border-t border-border space-y-1">
            <a href="/" target="_blank" rel="noopener noreferrer"
              className="flex items-center gap-3 px-3 py-2.5 text-sm text-muted-foreground hover:text-foreground rounded-xl hover:bg-muted transition-all">
              <ExternalLink className="h-[18px] w-[18px]" />
              <span>সাইট দেখুন</span>
            </a>
            <button onClick={handleLogout}
              className="w-full flex items-center gap-3 px-3 py-2.5 text-sm text-destructive hover:bg-destructive/10 rounded-xl transition-all">
              <LogOut className="h-[18px] w-[18px]" />
              <span>লগআউট</span>
            </button>
          </div>
        </div>
      </aside>

      {sidebarOpen && <div className="fixed inset-0 bg-black/40 backdrop-blur-sm z-40 lg:hidden" onClick={() => setSidebarOpen(false)} />}

      {/* ── Main ── */}
      <main className="flex-1 lg:ml-72 min-h-screen">
        {/* Header */}
        <header className="sticky top-0 z-30 bg-card/80 backdrop-blur-xl border-b border-border px-4 md:px-6 py-3">
          <div className="flex items-center justify-between gap-4">
            <div className="flex items-center gap-3">
              <button className="lg:hidden p-2 rounded-xl hover:bg-muted" onClick={() => setSidebarOpen(true)}>
                <Menu className="h-5 w-5" />
              </button>
              <div>
                <h2 className="font-heading font-bold text-lg">{navItems.find(n => n.tab === activeTab)?.label}</h2>
              </div>
            </div>
            <div className="flex items-center gap-2">
              {(activeTab === "products" || activeTab === "orders") && (
                <div className="relative hidden sm:block">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                  <Input
                    placeholder={activeTab === "products" ? "প্রোডাক্ট খুঁজুন..." : "অর্ডার খুঁজুন..."}
                    value={searchQuery}
                    onChange={e => setSearchQuery(e.target.value)}
                    className="pl-9 w-64 h-9 rounded-xl bg-muted/50 border-transparent focus:border-border"
                  />
                </div>
              )}
              <Button variant="outline" size="icon" className="h-9 w-9 rounded-xl" onClick={fetchData} title="রিফ্রেশ">
                <RefreshCw className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </header>

        <div className="p-4 md:p-6 max-w-7xl mx-auto">
          {loading ? (
            <div className="flex flex-col items-center justify-center py-32 gap-3">
              <div className="h-12 w-12 rounded-2xl bg-primary/10 flex items-center justify-center">
                <Loader2 className="h-6 w-6 text-primary animate-spin" />
              </div>
              <p className="text-sm text-muted-foreground">ডাটা লোড হচ্ছে...</p>
            </div>
          ) : (
            <AnimatePresence mode="wait">
              <motion.div key={activeTab} initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} exit={{ opacity: 0 }} transition={{ duration: 0.2 }}>

                {/* ═══ Dashboard ═══ */}
                {activeTab === "dashboard" && (
                  <div className="space-y-6">
                    <div className="grid grid-cols-2 lg:grid-cols-4 gap-4">
                      <StatCard label="মোট প্রোডাক্ট" value={products.length} icon={Package} color="bg-primary/10 text-primary" />
                      <StatCard label="মোট অর্ডার" value={orders.length} icon={ShoppingCart} color="bg-blue-500/10 text-blue-600" />
                      <StatCard label="সম্পন্ন অর্ডার" value={completedOrders} icon={CheckCircle2} color="bg-emerald-500/10 text-emerald-600" />
                      <StatCard label="পেন্ডিং অর্ডার" value={pendingOrders} icon={Clock} color="bg-amber-500/10 text-amber-600" />
                    </div>

                    {/* Revenue Card */}
                    <div className="bg-gradient-to-br from-primary/10 via-primary/5 to-transparent rounded-2xl border border-primary/20 p-6">
                      <div className="flex items-center gap-3 mb-2">
                        <div className="h-10 w-10 rounded-xl bg-primary/20 flex items-center justify-center">
                          <DollarSign className="h-5 w-5 text-primary" />
                        </div>
                        <div>
                          <p className="text-sm text-muted-foreground">মোট আয় (সম্পন্ন অর্ডার)</p>
                          <p className="text-2xl font-bold font-heading">৳{totalRevenue.toLocaleString()}</p>
                        </div>
                      </div>
                    </div>

                    {/* Recent Orders */}
                    <SectionCard title="সাম্প্রতিক অর্ডার" description="সর্বশেষ ৫টি অর্ডার" icon={ShoppingCart}>
                      <div className="overflow-x-auto -mx-6 -mb-6">
                        <table className="w-full text-sm">
                          <thead>
                            <tr className="border-b border-border text-muted-foreground">
                              <th className="text-left px-6 py-3 font-medium">কাস্টমার</th>
                              <th className="text-left px-6 py-3 font-medium hidden md:table-cell">প্রোডাক্ট</th>
                              <th className="text-left px-6 py-3 font-medium">মূল্য</th>
                              <th className="text-left px-6 py-3 font-medium">স্ট্যাটাস</th>
                            </tr>
                          </thead>
                          <tbody>
                            {orders.slice(0, 5).map((o) => (
                              <tr key={o.id} className="border-b border-border last:border-0 hover:bg-muted/30 transition-colors">
                                <td className="px-6 py-3.5 font-medium">{o.customer_name}</td>
                                <td className="px-6 py-3.5 hidden md:table-cell text-muted-foreground">{o.product_name}</td>
                                <td className="px-6 py-3.5 font-semibold">{o.amount}</td>
                                <td className="px-6 py-3.5">
                                  <span className={`inline-flex items-center px-2.5 py-1 rounded-lg text-xs font-semibold ${statusColor(o.status)}`}>
                                    {statusText(o.status)}
                                  </span>
                                </td>
                              </tr>
                            ))}
                            {orders.length === 0 && (
                              <tr><td colSpan={4} className="px-6 py-12 text-center text-muted-foreground">কোনো অর্ডার নেই</td></tr>
                            )}
                          </tbody>
                        </table>
                      </div>
                    </SectionCard>
                  </div>
                )}

                {/* ═══ Products ═══ */}
                {activeTab === "products" && (
                  <div className="space-y-4">
                    <div className="flex items-center justify-between flex-wrap gap-3">
                      <p className="text-sm text-muted-foreground">মোট {filteredProducts.length}টি প্রোডাক্ট</p>
                      <div className="flex gap-2">
                        <div className="relative sm:hidden">
                          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                          <Input placeholder="খুঁজুন..." value={searchQuery} onChange={e => setSearchQuery(e.target.value)}
                            className="pl-9 w-40 h-9 rounded-xl" />
                        </div>
                        <Button size="sm" className="gap-1.5 rounded-xl" onClick={() => { setShowAddForm(!showAddForm); setEditingProduct(null); }}>
                          {showAddForm ? <XCircle className="h-4 w-4" /> : <Plus className="h-4 w-4" />}
                          {showAddForm ? "বাতিল" : "নতুন প্রোডাক্ট"}
                        </Button>
                      </div>
                    </div>

                    <AnimatePresence>
                      {showAddForm && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <SectionCard title="নতুন প্রোডাক্ট যোগ করুন" icon={Plus}>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                              <div className="space-y-1.5 md:col-span-2"><Label className="text-xs">নাম *</Label><Input value={newProduct.name} onChange={e => setNewProduct(p => ({ ...p, name: e.target.value }))} placeholder="প্রোডাক্টের নাম" className="rounded-xl" /></div>
                              <div className="space-y-1.5 md:col-span-2"><Label className="text-xs">ক্যাটাগরি * (একাধিক সিলেক্ট করা যাবে)</Label>
                                <div className="flex flex-wrap gap-2 p-3 rounded-xl border border-input bg-background">
                                  {categories.filter(c => c.is_active).map(c => {
                                    const catName = c.name.trim();
                                    const selected = newProduct.category.split(",").map(s => s.trim()).filter(Boolean).includes(catName);
                                    return (
                                      <button key={c.id} type="button" onClick={() => {
                                        const current = newProduct.category.split(",").map(s => s.trim()).filter(Boolean);
                                        const updated = selected ? current.filter(x => x !== catName) : [...current, catName];
                                        setNewProduct(p => ({ ...p, category: updated.join(", ") }));
                                      }}
                                        className={`px-3 py-1.5 rounded-lg text-xs font-medium transition-all ${selected ? "bg-primary text-primary-foreground shadow-sm" : "bg-muted text-muted-foreground hover:bg-muted/80"}`}>
                                        {selected && <span className="mr-1">✓</span>}{catName}
                                      </button>
                                    );
                                  })}
                                </div>
                              </div>
                              {/* Image Upload */}
                              <div className="md:col-span-2 space-y-1.5">
                                <Label className="text-xs">প্রোডাক্ট ইমেজ</Label>
                                <input type="file" accept="image/*" ref={productImageRef} onChange={handleNewProductImage} className="hidden" />
                                <div className="flex items-center gap-3">
                                  <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingProductImage} onClick={() => productImageRef.current?.click()}>
                                    {uploadingProductImage ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                                    {uploadingProductImage ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড"}
                                  </Button>
                                  <span className="text-xs text-muted-foreground">অথবা</span>
                                  <Input value={newProduct.image_url} onChange={e => setNewProduct(p => ({ ...p, image_url: e.target.value }))} placeholder="ইমেজ URL দিন" className="rounded-xl flex-1" />
                                </div>
                                {newProduct.image_url && <img src={newProduct.image_url} alt="preview" className="h-16 w-16 rounded-xl object-cover border border-border mt-2" />}
                              </div>
                              <div className="md:col-span-2 space-y-1.5"><Label className="text-xs">সংক্ষিপ্ত বিবরণ</Label><Textarea value={newProduct.description} onChange={e => setNewProduct(p => ({ ...p, description: e.target.value }))} placeholder="প্রোডাক্ট সম্পর্কে সংক্ষেপে লিখুন..." className="rounded-xl min-h-[80px] resize-y" /></div>
                              <div className="md:col-span-2 space-y-1.5"><Label className="text-xs">বিস্তারিত বিবরণ (অপশনাল)</Label><Textarea value={newProduct.long_description} onChange={e => setNewProduct(p => ({ ...p, long_description: e.target.value }))} placeholder="প্রোডাক্ট সম্পর্কে বিস্তারিত লিখুন..." className="rounded-xl min-h-[100px]" /></div>
                              {/* Duration/Version Options */}
                              <div className="md:col-span-2 space-y-2">
                                <Label className="text-xs">ভার্সন / Duration অপশন</Label>
                                <p className="text-[11px] text-muted-foreground">প্রতিটি ভার্সনের নাম ও মূল্য আলাদাভাবে দিন</p>
                                <div className="grid grid-cols-2 sm:grid-cols-[1fr_auto_auto_auto] gap-2 items-end">
                                  <div className="col-span-2 sm:col-span-1 space-y-1">
                                    <Label className="text-[10px] text-muted-foreground">ভার্সন নাম</Label>
                                    <Input value={newVersionLabel} onChange={e => setNewVersionLabel(e.target.value)} placeholder="যেমন: 1 মাস"
                                      className="rounded-xl"
                                      onKeyDown={e => {
                                        if (e.key === "Enter" && newVersionLabel.trim()) {
                                          e.preventDefault();
                                          const val = newVersionPrice.trim() ? `${newVersionLabel.trim()} - ${newVersionPrice.trim().startsWith("৳") ? newVersionPrice.trim() : "৳" + newVersionPrice.trim()}` : newVersionLabel.trim();
                                          if (newVersionResellerPrice.trim()) {
                                            const rp = newVersionResellerPrice.trim().startsWith("৳") ? newVersionResellerPrice.trim() : "৳" + newVersionResellerPrice.trim();
                                            setNewProductResellerPrices(prev => ({ ...prev, [val]: rp }));
                                          }
                                          setNewProduct(p => ({ ...p, duration_options: [...p.duration_options, val] }));
                                          setNewVersionLabel(""); setNewVersionPrice(""); setNewVersionResellerPrice("");
                                        }
                                      }} />
                                  </div>
                                  <div className="space-y-1">
                                    <Label className="text-[10px] text-muted-foreground">মূল্য</Label>
                                    <Input value={newVersionPrice} onChange={e => setNewVersionPrice(e.target.value)} placeholder="৳299" className="rounded-xl" />
                                  </div>
                                  <div className="space-y-1">
                                    <Label className="text-[10px] text-muted-foreground">🏪 রিসেলার</Label>
                                    <Input value={newVersionResellerPrice} onChange={e => setNewVersionResellerPrice(e.target.value)} placeholder="৳199" className="rounded-xl" />
                                  </div>
                                  <Button type="button" variant="outline" size="sm" className="rounded-xl gap-1 h-10 col-span-2 sm:col-span-1" onClick={() => {
                                    if (newVersionLabel.trim()) {
                                      const val = newVersionPrice.trim() ? `${newVersionLabel.trim()} - ${newVersionPrice.trim().startsWith("৳") ? newVersionPrice.trim() : "৳" + newVersionPrice.trim()}` : newVersionLabel.trim();
                                      if (newVersionResellerPrice.trim()) {
                                        const rp = newVersionResellerPrice.trim().startsWith("৳") ? newVersionResellerPrice.trim() : "৳" + newVersionResellerPrice.trim();
                                        setNewProductResellerPrices(prev => ({ ...prev, [val]: rp }));
                                      }
                                      setNewProduct(p => ({ ...p, duration_options: [...p.duration_options, val] }));
                                      setNewVersionLabel(""); setNewVersionPrice(""); setNewVersionResellerPrice("");
                                    }
                                  }}>
                                    <Plus className="h-3.5 w-3.5" />যোগ
                                  </Button>
                                </div>
                                {newProduct.duration_options.length > 0 && (
                                  <div className="space-y-1.5 mt-2 border border-border rounded-xl overflow-hidden">
                                    <div className="text-[10px] font-semibold text-muted-foreground bg-muted px-3 py-1.5 flex justify-between">
                                      <span>ভার্সন</span><span>মূল্য / 🏪</span>
                                    </div>
                                    {newProduct.duration_options.map((v, i) => {
                                      const parsed = parseDurationOption(v);
                                      const resellerP = newProductResellerPrices[v] || "";
                                      const isEditing = editingVersionIdx === i;
                                      return (
                                        <div key={i} className="px-3 py-2 border-t border-border text-xs">
                                          {isEditing ? (
                                            <div className="space-y-2">
                                              <div className="space-y-1">
                                                <label className="text-[10px] text-muted-foreground">ভার্সন নাম</label>
                                                <Input value={editingVersionLabelInline} onChange={e => setEditingVersionLabelInline(e.target.value)} className="h-8 text-xs rounded-lg" />
                                              </div>
                                              <div className="grid grid-cols-2 gap-2">
                                                <div className="space-y-1">
                                                  <label className="text-[10px] text-muted-foreground">💰 কাস্টমার মূল্য</label>
                                                  <Input value={editingVersionPriceInline} onChange={e => setEditingVersionPriceInline(e.target.value)} className="h-8 text-xs rounded-lg" placeholder="৳299" />
                                                </div>
                                                <div className="space-y-1">
                                                  <label className="text-[10px] text-muted-foreground">🏪 রিসেলার মূল্য</label>
                                                  <Input value={editingVersionResellerPriceInline} onChange={e => setEditingVersionResellerPriceInline(e.target.value)} className="h-8 text-xs rounded-lg" placeholder="৳199" />
                                                </div>
                                              </div>
                                              <div className="flex gap-2">
                                                <button type="button" className="text-primary text-xs font-semibold" onClick={() => {
                                                  const oldVal = newProduct.duration_options[i];
                                                  const val = editingVersionPriceInline.trim() ? `${editingVersionLabelInline.trim()} - ${editingVersionPriceInline.trim().startsWith("৳") ? editingVersionPriceInline.trim() : "৳" + editingVersionPriceInline.trim()}` : editingVersionLabelInline.trim();
                                                  setNewProduct(p => ({ ...p, duration_options: p.duration_options.map((d, idx) => idx === i ? val : d) }));
                                                  setNewProductResellerPrices(prev => {
                                                    const next = { ...prev };
                                                    delete next[oldVal];
                                                    if (editingVersionResellerPriceInline.trim()) {
                                                      const rp = editingVersionResellerPriceInline.trim().startsWith("৳") ? editingVersionResellerPriceInline.trim() : "৳" + editingVersionResellerPriceInline.trim();
                                                      next[val] = rp;
                                                    }
                                                    return next;
                                                  });
                                                  setEditingVersionIdx(null);
                                                }}>Save</button>
                                                <button type="button" className="text-muted-foreground text-xs" onClick={() => setEditingVersionIdx(null)}>Cancel</button>
                                              </div>
                                            </div>
                                          ) : (
                                            <div className="flex items-center justify-between gap-2">
                                              <span className="font-medium text-foreground truncate flex-1">{parsed.label}</span>
                                              <div className="text-right leading-tight shrink-0">
                                                <span className="text-primary font-bold text-xs">💰 {parsed.price || "—"}</span>
                                                <span className="block text-[10px] text-accent-foreground">🏪 {resellerP || "—"}</span>
                                              </div>
                                              <div className="flex gap-2 shrink-0">
                                                <button type="button" className="text-primary text-[10px] font-semibold" onClick={() => {
                                                  setEditingVersionIdx(i);
                                                  setEditingVersionLabelInline(parsed.label);
                                                  setEditingVersionPriceInline(parsed.price?.replace("৳", "") || "");
                                                  setEditingVersionResellerPriceInline(resellerP?.replace("৳", "") || "");
                                                }}>Edit</button>
                                                <button type="button" className="text-destructive text-[10px] font-semibold" onClick={() => {
                                                  setNewProduct(p => ({ ...p, duration_options: p.duration_options.filter((_, idx) => idx !== i) }));
                                                  setNewProductResellerPrices(prev => { const next = { ...prev }; delete next[v]; return next; });
                                                }}>Remove</button>
                                              </div>
                                            </div>
                                          )}
                                        </div>
                                      );
                                    })}
                                  </div>
                                )}
                              </div>
                              {/* Features */}
                              <div className="md:col-span-2 space-y-2">
                                <Label className="text-xs">⭐ ফিচারস (অপশনাল)</Label>
                                <div className="flex gap-2">
                                  <Input
                                    value={newFeatureText}
                                    onChange={e => setNewFeatureText(e.target.value)}
                                    placeholder="ফিচার লিখুন, যেমন: Ad-Free YouTube"
                                    className="rounded-xl flex-1"
                                    onKeyDown={e => {
                                      if (e.key === "Enter" && newFeatureText.trim()) {
                                        e.preventDefault();
                                        setNewProduct(p => ({ ...p, features: [...p.features, newFeatureText.trim()] }));
                                        setNewFeatureText("");
                                      }
                                    }}
                                  />
                                  <Button type="button" size="sm" variant="outline" className="rounded-xl" disabled={!newFeatureText.trim()} onClick={() => {
                                    setNewProduct(p => ({ ...p, features: [...p.features, newFeatureText.trim()] }));
                                    setNewFeatureText("");
                                  }}><Plus className="h-4 w-4" /></Button>
                                </div>
                                {newProduct.features.length > 0 && (
                                  <div className="flex flex-wrap gap-1.5 mt-2">
                                    {newProduct.features.map((f, i) => (
                                      <span key={i} className="inline-flex items-center gap-1 px-2.5 py-1 rounded-lg bg-muted text-xs text-foreground">
                                        {f}
                                        <button type="button" onClick={() => setNewProduct(p => ({ ...p, features: p.features.filter((_, idx) => idx !== i) }))} className="text-destructive hover:text-destructive/80 ml-0.5">
                                          <X className="h-3 w-3" />
                                        </button>
                                      </span>
                                    ))}
                                  </div>
                                )}
                              </div>
                              {/* Custom Fields */}
                              <div className="md:col-span-2 space-y-2">
                                <Label className="text-xs">📝 কাস্টম ফিল্ড (অপশনাল)</Label>
                                <p className="text-[11px] text-muted-foreground">কাস্টমার অর্ডার করার সময় এই ফিল্ডগুলো পূরণ করবে (যেমন: Canva Email)</p>
                                <div className="flex gap-2 items-end flex-wrap">
                                  <div className="flex-1 min-w-[120px] space-y-1">
                                    <Label className="text-[10px] text-muted-foreground">ফিল্ড নাম *</Label>
                                    <Input id="new-cf-label" placeholder="যেমন: Canva Email" className="rounded-xl" />
                                  </div>
                                  <div className="flex-1 min-w-[120px] space-y-1">
                                    <Label className="text-[10px] text-muted-foreground">Placeholder</Label>
                                    <Input id="new-cf-placeholder" placeholder="যেমন: youremail@gmail.com" className="rounded-xl" />
                                  </div>
                                  <div className="w-[160px] space-y-1">
                                    <Label className="text-[10px] text-muted-foreground">হেল্পার টেক্সট</Label>
                                    <Input id="new-cf-helper" placeholder="ঐচ্ছিক নোট" className="rounded-xl" />
                                  </div>
                                  <div className="flex items-center gap-2">
                                    <label className="flex items-center gap-1 text-[10px] text-muted-foreground cursor-pointer">
                                      <input type="checkbox" id="new-cf-required" defaultChecked className="rounded" />
                                      Required
                                    </label>
                                    <Button type="button" variant="outline" size="sm" className="rounded-xl gap-1 h-10" onClick={() => {
                                      const label = (document.getElementById("new-cf-label") as HTMLInputElement)?.value?.trim();
                                      if (!label) return;
                                      const placeholder = (document.getElementById("new-cf-placeholder") as HTMLInputElement)?.value?.trim() || "";
                                      const helper_text = (document.getElementById("new-cf-helper") as HTMLInputElement)?.value?.trim() || "";
                                      const required = (document.getElementById("new-cf-required") as HTMLInputElement)?.checked ?? true;
                                      setNewProduct(p => ({ ...p, custom_fields: [...p.custom_fields, { label, placeholder, required, helper_text }] }));
                                      (document.getElementById("new-cf-label") as HTMLInputElement).value = "";
                                      (document.getElementById("new-cf-placeholder") as HTMLInputElement).value = "";
                                      (document.getElementById("new-cf-helper") as HTMLInputElement).value = "";
                                    }}>
                                      <Plus className="h-3.5 w-3.5" />যোগ
                                    </Button>
                                  </div>
                                </div>
                                {newProduct.custom_fields.length > 0 && (
                                  <div className="space-y-1 mt-2 border border-border rounded-xl overflow-hidden">
                                    <div className="grid grid-cols-[1fr_1fr_auto_auto] gap-0 text-[10px] font-semibold text-muted-foreground bg-muted px-3 py-1.5">
                                      <span>ফিল্ড নাম</span><span>Placeholder</span><span className="w-16 text-center">Required</span><span className="w-12"></span>
                                    </div>
                                    {newProduct.custom_fields.map((cf, i) => (
                                      <div key={i} className="grid grid-cols-[1fr_1fr_auto_auto] gap-2 items-center px-3 py-2 border-t border-border text-xs">
                                        <span className="font-medium text-foreground">{cf.label}</span>
                                        <span className="text-muted-foreground">{cf.placeholder || "—"}</span>
                                        <span className="w-16 text-center">{cf.required ? "✅" : "—"}</span>
                                        <button type="button" className="text-destructive text-[10px] font-semibold w-12 text-right" onClick={() => setNewProduct(p => ({ ...p, custom_fields: p.custom_fields.filter((_, idx) => idx !== i) }))}>Remove</button>
                                      </div>
                                    ))}
                                  </div>
                                )}
                              </div>
                            </div>
                            <div className="mt-4">
                              <Button size="sm" onClick={handleAddProduct} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />সেভ করুন</Button>
                            </div>
                          </SectionCard>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    <AnimatePresence>
                      {editingProduct && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <div className="bg-card rounded-2xl border-2 border-primary/30 overflow-hidden">
                            <div className="px-6 py-4 border-b border-primary/20 bg-primary/5 flex items-center gap-3">
                              <Edit2 className="h-4 w-4 text-primary" />
                              <h3 className="font-heading font-semibold text-sm">প্রোডাক্ট এডিট করুন</h3>
                            </div>
                            <div className="p-6">
                              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div className="space-y-1.5 md:col-span-2"><Label className="text-xs">নাম</Label><Input value={editingProduct.name} onChange={e => setEditingProduct(p => p ? { ...p, name: e.target.value } : null)} className="rounded-xl" /></div>
                                
                                <div className="space-y-1.5 md:col-span-2"><Label className="text-xs">ক্যাটাগরি (একাধিক সিলেক্ট করা যাবে)</Label>
                                  <div className="flex flex-wrap gap-2 p-3 rounded-xl border border-input bg-background">
                                    {categories.filter(c => c.is_active).map(c => {
                                      const catName = c.name.trim();
                                      const selected = editingProduct.category.split(",").map(s => s.trim()).filter(Boolean).includes(catName);
                                      return (
                                        <button key={c.id} type="button" onClick={() => {
                                          const current = editingProduct.category.split(",").map(s => s.trim()).filter(Boolean);
                                          const updated = selected ? current.filter(x => x !== catName) : [...current, catName];
                                          setEditingProduct(p => p ? { ...p, category: updated.join(", ") } : null);
                                        }}
                                          className={`px-3 py-1.5 rounded-lg text-xs font-medium transition-all ${selected ? "bg-primary text-primary-foreground shadow-sm" : "bg-muted text-muted-foreground hover:bg-muted/80"}`}>
                                          {selected && <span className="mr-1">✓</span>}{catName}
                                        </button>
                                      );
                                    })}
                                  </div>
                                </div>
                                {/* Image Upload */}
                                <div className="md:col-span-2 space-y-1.5">
                                  <Label className="text-xs">প্রোডাক্ট ইমেজ</Label>
                                  <input type="file" accept="image/*" ref={editProductImageRef} onChange={handleEditProductImage} className="hidden" />
                                  <div className="flex items-center gap-3">
                                    <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingProductImage} onClick={() => editProductImageRef.current?.click()}>
                                      {uploadingProductImage ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                                      {uploadingProductImage ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড"}
                                    </Button>
                                    <span className="text-xs text-muted-foreground">অথবা</span>
                                    <Input value={editingProduct.image_url || ""} onChange={e => setEditingProduct(p => p ? { ...p, image_url: e.target.value } : null)} placeholder="ইমেজ URL দিন" className="rounded-xl flex-1" />
                                  </div>
                                  {editingProduct.image_url && <img src={editingProduct.image_url} alt="preview" className="h-16 w-16 rounded-xl object-cover border border-border mt-2" />}
                                </div>
                                <div className="md:col-span-2 space-y-1.5"><Label className="text-xs">সংক্ষিপ্ত বিবরণ</Label><Textarea value={editingProduct.description || ""} onChange={e => setEditingProduct(p => p ? { ...p, description: e.target.value } : null)} placeholder="প্রোডাক্ট সম্পর্কে সংক্ষেপে লিখুন..." className="rounded-xl min-h-[80px] resize-y" /></div>
                                <div className="md:col-span-2 space-y-1.5"><Label className="text-xs">বিস্তারিত বিবরণ (অপশনাল)</Label><Textarea value={(editingProduct as any).long_description || ""} onChange={e => setEditingProduct(p => p ? { ...p, long_description: e.target.value } as any : null)} placeholder="প্রোডাক্ট সম্পর্কে বিস্তারিত লিখুন..." className="rounded-xl min-h-[100px]" /></div>
                                {/* Duration/Version Options */}
                                <div className="md:col-span-2 space-y-2">
                                  <Label className="text-xs">ভার্সন / Duration অপশন</Label>
                                  <p className="text-[11px] text-muted-foreground">প্রতিটি ভার্সনের নাম ও মূল্য আলাদাভাবে দিন</p>
                                  <div className="grid grid-cols-2 sm:grid-cols-[1fr_auto_auto_auto] gap-2 items-end">
                                    <div className="col-span-2 sm:col-span-1 space-y-1">
                                      <Label className="text-[10px] text-muted-foreground">ভার্সন নাম</Label>
                                      <Input value={editVersionLabel} onChange={e => setEditVersionLabel(e.target.value)} placeholder="যেমন: 1 মাস"
                                        className="rounded-xl"
                                        onKeyDown={e => {
                                          if (e.key === "Enter" && editVersionLabel.trim()) {
                                            e.preventDefault();
                                            const val = editVersionPrice.trim() ? `${editVersionLabel.trim()} - ${editVersionPrice.trim().startsWith("৳") ? editVersionPrice.trim() : "৳" + editVersionPrice.trim()}` : editVersionLabel.trim();
                                            if (editVersionResellerPrice.trim()) {
                                              const rp = editVersionResellerPrice.trim().startsWith("৳") ? editVersionResellerPrice.trim() : "৳" + editVersionResellerPrice.trim();
                                              setEditProductResellerPrices(prev => ({ ...prev, [val]: rp }));
                                            }
                                            setEditingProduct(p => p ? { ...p, duration_options: [...(p.duration_options || []), val] } : null);
                                            setEditVersionLabel(""); setEditVersionPrice(""); setEditVersionResellerPrice("");
                                          }
                                        }} />
                                    </div>
                                    <div className="space-y-1">
                                      <Label className="text-[10px] text-muted-foreground">মূল্য</Label>
                                      <Input value={editVersionPrice} onChange={e => setEditVersionPrice(e.target.value)} placeholder="৳299" className="rounded-xl" />
                                    </div>
                                    <div className="space-y-1">
                                      <Label className="text-[10px] text-muted-foreground">🏪 রিসেলার</Label>
                                      <Input value={editVersionResellerPrice} onChange={e => setEditVersionResellerPrice(e.target.value)} placeholder="৳199" className="rounded-xl" />
                                    </div>
                                    <Button type="button" variant="outline" size="sm" className="rounded-xl gap-1 h-10 col-span-2 sm:col-span-1" onClick={() => {
                                      if (editVersionLabel.trim()) {
                                        const val = editVersionPrice.trim() ? `${editVersionLabel.trim()} - ${editVersionPrice.trim().startsWith("৳") ? editVersionPrice.trim() : "৳" + editVersionPrice.trim()}` : editVersionLabel.trim();
                                        if (editVersionResellerPrice.trim()) {
                                          const rp = editVersionResellerPrice.trim().startsWith("৳") ? editVersionResellerPrice.trim() : "৳" + editVersionResellerPrice.trim();
                                          setEditProductResellerPrices(prev => ({ ...prev, [val]: rp }));
                                        }
                                        setEditingProduct(p => p ? { ...p, duration_options: [...(p.duration_options || []), val] } : null);
                                        setEditVersionLabel(""); setEditVersionPrice(""); setEditVersionResellerPrice("");
                                      }
                                    }}>
                                      <Plus className="h-3.5 w-3.5" />যোগ
                                    </Button>
                                  </div>
                                  {editingProduct.duration_options && editingProduct.duration_options.length > 0 && (
                                    <div className="space-y-1.5 mt-2 border border-border rounded-xl overflow-hidden">
                                    <div className="text-[10px] font-semibold text-muted-foreground bg-muted px-3 py-1.5 flex justify-between">
                                      <span>ভার্সন</span><span>মূল্য / 🏪</span>
                                    </div>
                                    {editingProduct.duration_options.map((v, i) => {
                                      const parsed = parseDurationOption(v);
                                      const resellerP = editProductResellerPrices[v] || "";
                                      const isEditing = editingVersionIdx === i + 1000;
                                      return (
                                        <div key={i} className="px-3 py-2 border-t border-border text-xs">
                                          {isEditing ? (
                                            <div className="space-y-2">
                                              <div className="space-y-1">
                                                <label className="text-[10px] text-muted-foreground">ভার্সন নাম</label>
                                                <Input value={editingVersionLabelInline} onChange={e => setEditingVersionLabelInline(e.target.value)} className="h-8 text-xs rounded-lg" />
                                              </div>
                                              <div className="grid grid-cols-2 gap-2">
                                                <div className="space-y-1">
                                                  <label className="text-[10px] text-muted-foreground">💰 কাস্টমার মূল্য</label>
                                                  <Input value={editingVersionPriceInline} onChange={e => setEditingVersionPriceInline(e.target.value)} className="h-8 text-xs rounded-lg" placeholder="৳299" />
                                                </div>
                                                <div className="space-y-1">
                                                  <label className="text-[10px] text-muted-foreground">🏪 রিসেলার মূল্য</label>
                                                  <Input value={editingVersionResellerPriceInline} onChange={e => setEditingVersionResellerPriceInline(e.target.value)} className="h-8 text-xs rounded-lg" placeholder="৳199" />
                                                </div>
                                              </div>
                                              <div className="flex gap-2">
                                                <button type="button" className="text-primary text-xs font-semibold" onClick={() => {
                                                  const oldVal = editingProduct.duration_options[i];
                                                  const val = editingVersionPriceInline.trim() ? `${editingVersionLabelInline.trim()} - ${editingVersionPriceInline.trim().startsWith("৳") ? editingVersionPriceInline.trim() : "৳" + editingVersionPriceInline.trim()}` : editingVersionLabelInline.trim();
                                                  setEditingProduct(p => p ? { ...p, duration_options: p.duration_options.map((d, idx) => idx === i ? val : d) } : null);
                                                  setEditProductResellerPrices(prev => {
                                                    const next = { ...prev };
                                                    delete next[oldVal];
                                                    if (editingVersionResellerPriceInline.trim()) {
                                                      const rp = editingVersionResellerPriceInline.trim().startsWith("৳") ? editingVersionResellerPriceInline.trim() : "৳" + editingVersionResellerPriceInline.trim();
                                                      next[val] = rp;
                                                    }
                                                    return next;
                                                  });
                                                  setEditingVersionIdx(null);
                                                }}>Save</button>
                                                <button type="button" className="text-muted-foreground text-xs" onClick={() => setEditingVersionIdx(null)}>Cancel</button>
                                              </div>
                                            </div>
                                          ) : (
                                            <div className="flex items-center justify-between gap-2">
                                              <span className="font-medium text-foreground truncate flex-1">{parsed.label}</span>
                                              <div className="text-right leading-tight shrink-0">
                                                <span className="text-primary font-bold text-xs">💰 {parsed.price || "—"}</span>
                                                <span className="block text-[10px] text-accent-foreground">🏪 {resellerP || "—"}</span>
                                              </div>
                                              <div className="flex gap-2 shrink-0">
                                                <button type="button" className="text-primary text-[10px] font-semibold" onClick={() => {
                                                  setEditingVersionIdx(i + 1000);
                                                  setEditingVersionLabelInline(parsed.label);
                                                  setEditingVersionPriceInline(parsed.price?.replace("৳", "") || "");
                                                  setEditingVersionResellerPriceInline(resellerP?.replace("৳", "") || "");
                                                }}>Edit</button>
                                                <button type="button" className="text-destructive text-[10px] font-semibold" onClick={() => {
                                                  setEditingProduct(p => p ? { ...p, duration_options: p.duration_options.filter((_, idx) => idx !== i) } : null);
                                                  setEditProductResellerPrices(prev => { const next = { ...prev }; delete next[v]; return next; });
                                                }}>Remove</button>
                                              </div>
                                            </div>
                                          )}
                                        </div>
                                      );
                                    })}
                                    </div>
                                  )}
                                </div>
                                {/* Features */}
                                <div className="md:col-span-2 space-y-2">
                                  <Label className="text-xs">⭐ ফিচারস (অপশনাল)</Label>
                                  <div className="flex gap-2">
                                    <Input
                                      placeholder="ফিচার লিখুন..."
                                      className="rounded-xl flex-1"
                                      value={editFeatureText}
                                      onChange={e => setEditFeatureText(e.target.value)}
                                      onKeyDown={e => {
                                        if (e.key === "Enter") {
                                          e.preventDefault();
                                          if (editFeatureText.trim()) {
                                            setEditingProduct(p => p ? { ...p, features: [...(p.features || []), editFeatureText.trim()] } : null);
                                            setEditFeatureText("");
                                          }
                                        }
                                      }}
                                    />
                                    <Button type="button" size="sm" variant="outline" className="rounded-xl" disabled={!editFeatureText.trim()} onClick={() => {
                                      if (editFeatureText.trim()) {
                                        setEditingProduct(p => p ? { ...p, features: [...(p.features || []), editFeatureText.trim()] } : null);
                                        setEditFeatureText("");
                                      }
                                    }}><Plus className="h-4 w-4" /></Button>
                                  </div>
                                  {editingProduct.features && editingProduct.features.length > 0 && (
                                    <div className="flex flex-wrap gap-1.5 mt-2">
                                      {editingProduct.features.map((f, i) => (
                                        <span key={i} className="inline-flex items-center gap-1 px-2.5 py-1 rounded-lg bg-muted text-xs text-foreground">
                                          {f}
                                          <button type="button" onClick={() => setEditingProduct(p => p ? { ...p, features: p.features.filter((_, idx) => idx !== i) } : null)} className="text-destructive hover:text-destructive/80 ml-0.5">
                                            <X className="h-3 w-3" />
                                          </button>
                                        </span>
                                      ))}
                                    </div>
                                  )}
                                </div>
                                {/* Custom Fields */}
                                <div className="md:col-span-2 space-y-2">
                                  <Label className="text-xs">📝 কাস্টম ফিল্ড (অপশনাল)</Label>
                                  <p className="text-[11px] text-muted-foreground">কাস্টমার অর্ডার করার সময় এই ফিল্ডগুলো পূরণ করবে</p>
                                  <div className="flex gap-2 items-end flex-wrap">
                                    <div className="flex-1 min-w-[120px] space-y-1">
                                      <Label className="text-[10px] text-muted-foreground">ফিল্ড নাম *</Label>
                                      <Input id="edit-cf-label" placeholder="যেমন: Canva Email" className="rounded-xl" />
                                    </div>
                                    <div className="flex-1 min-w-[120px] space-y-1">
                                      <Label className="text-[10px] text-muted-foreground">Placeholder</Label>
                                      <Input id="edit-cf-placeholder" placeholder="যেমন: youremail@gmail.com" className="rounded-xl" />
                                    </div>
                                    <div className="w-[160px] space-y-1">
                                      <Label className="text-[10px] text-muted-foreground">হেল্পার টেক্সট</Label>
                                      <Input id="edit-cf-helper" placeholder="ঐচ্ছিক নোট" className="rounded-xl" />
                                    </div>
                                    <div className="flex items-center gap-2">
                                      <label className="flex items-center gap-1 text-[10px] text-muted-foreground cursor-pointer">
                                        <input type="checkbox" id="edit-cf-required" defaultChecked className="rounded" />
                                        Required
                                      </label>
                                      <Button type="button" variant="outline" size="sm" className="rounded-xl gap-1 h-10" onClick={() => {
                                        const label = (document.getElementById("edit-cf-label") as HTMLInputElement)?.value?.trim();
                                        if (!label) return;
                                        const placeholder = (document.getElementById("edit-cf-placeholder") as HTMLInputElement)?.value?.trim() || "";
                                        const helper_text = (document.getElementById("edit-cf-helper") as HTMLInputElement)?.value?.trim() || "";
                                        const required = (document.getElementById("edit-cf-required") as HTMLInputElement)?.checked ?? true;
                                        setEditingProduct(p => p ? { ...p, custom_fields: [...((p as any).custom_fields || []), { label, placeholder, required, helper_text }] } as any : null);
                                        (document.getElementById("edit-cf-label") as HTMLInputElement).value = "";
                                        (document.getElementById("edit-cf-placeholder") as HTMLInputElement).value = "";
                                        (document.getElementById("edit-cf-helper") as HTMLInputElement).value = "";
                                      }}>
                                        <Plus className="h-3.5 w-3.5" />যোগ
                                      </Button>
                                    </div>
                                  </div>
                                  {((editingProduct as any)?.custom_fields || []).length > 0 && (
                                    <div className="space-y-1 mt-2 border border-border rounded-xl overflow-hidden">
                                      <div className="grid grid-cols-[1fr_1fr_auto_auto] gap-0 text-[10px] font-semibold text-muted-foreground bg-muted px-3 py-1.5">
                                        <span>ফিল্ড নাম</span><span>Placeholder</span><span className="w-16 text-center">Required</span><span className="w-12"></span>
                                      </div>
                                      {((editingProduct as any).custom_fields || []).map((cf: any, i: number) => (
                                        <div key={i} className="grid grid-cols-[1fr_1fr_auto_auto] gap-2 items-center px-3 py-2 border-t border-border text-xs">
                                          <span className="font-medium text-foreground">{cf.label}</span>
                                          <span className="text-muted-foreground">{cf.placeholder || "—"}</span>
                                          <span className="w-16 text-center">{cf.required ? "✅" : "—"}</span>
                                          <button type="button" className="text-destructive text-[10px] font-semibold w-12 text-right" onClick={() => setEditingProduct(p => p ? { ...p, custom_fields: ((p as any).custom_fields || []).filter((_: any, idx: number) => idx !== i) } as any : null)}>Remove</button>
                                        </div>
                                      ))}
                                    </div>
                                  )}
                                </div>
                              </div>
                              <div className="flex gap-2 mt-4">
                                <Button size="sm" onClick={handleUpdateProduct} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />আপডেট</Button>
                                <Button size="sm" variant="outline" onClick={() => setEditingProduct(null)} className="rounded-xl">বাতিল</Button>
                              </div>
                            </div>
                          </div>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    <div className="bg-card rounded-2xl border border-border overflow-hidden">
                      <div className="overflow-x-auto">
                        <table className="w-full text-sm">
                          <thead>
                            <tr className="border-b border-border text-muted-foreground bg-muted/30">
                              <th className="text-left px-4 py-3 font-medium">প্রোডাক্ট</th>
                              <th className="text-left px-4 py-3 font-medium hidden md:table-cell">ক্যাটাগরি</th>
                              <th className="text-left px-4 py-3 font-medium">মূল্য</th>
                              <th className="text-right px-4 py-3 font-medium">অ্যাকশন</th>
                            </tr>
                          </thead>
                          <tbody>
                            {filteredProducts.map((p, i) => (
                              <motion.tr key={p.id} initial={{ opacity: 0 }} animate={{ opacity: 1 }} transition={{ delay: i * 0.03 }}
                                className="border-b border-border last:border-0 hover:bg-muted/30 transition-colors">
                                <td className="px-4 py-3">
                                  <div className="flex items-center gap-3">
                                    {p.image_url ? (
                                      <img src={p.image_url} alt={p.name} className="h-10 w-10 rounded-xl object-cover bg-muted flex-shrink-0" />
                                    ) : (
                                      <div className="h-10 w-10 rounded-xl bg-muted flex items-center justify-center flex-shrink-0">
                                        <Package className="h-4 w-4 text-muted-foreground" />
                                      </div>
                                    )}
                                    <div className="min-w-0">
                                      <span className="font-medium truncate max-w-[200px] block">{p.name}</span>
                                      {p.duration_options && p.duration_options.length > 0 && (
                                        <span className="text-[10px] text-muted-foreground">{p.duration_options.length}টি ভার্সন</span>
                                      )}
                                    </div>
                                  </div>
                                </td>
                                <td className="px-4 py-3 hidden md:table-cell">
                                  <span className="px-2.5 py-1 rounded-lg text-xs bg-muted text-muted-foreground font-medium">{p.category}</span>
                                </td>
                                <td className="px-4 py-3 font-semibold text-primary">{p.duration_options && p.duration_options.length > 0 ? (getPriceRangeFromDurations(p.duration_options) || p.price) : p.price}</td>
                                <td className="px-4 py-3">
                                  <div className="flex items-center gap-1 justify-end">
                                    <Button variant="ghost" size="icon" className="h-8 w-8 rounded-lg" onClick={() => { setEditingProduct({ ...p, features: (p.features || []).map((f: string) => f?.trim()).filter(Boolean) }); setEditProductResellerPrices((p as any).reseller_prices || {}); setEditFeatureText(""); setShowAddForm(false); }}>
                                      <Edit2 className="h-3.5 w-3.5" />
                                    </Button>
                                    <Button variant="ghost" size="icon" className="h-8 w-8 rounded-lg text-destructive hover:text-destructive"
                                      onClick={() => setConfirmDelete({ type: "product", id: p.id, name: p.name })}>
                                      <Trash2 className="h-3.5 w-3.5" />
                                    </Button>
                                  </div>
                                </td>
                              </motion.tr>
                            ))}
                            {filteredProducts.length === 0 && (
                              <tr><td colSpan={4}>
                                <EmptyState icon={Package} title="কোনো প্রোডাক্ট নেই" description="নতুন প্রোডাক্ট যোগ করতে উপরের বাটন ক্লিক করুন।"
                                  action={<Button size="sm" className="rounded-xl gap-1.5" onClick={() => setShowAddForm(true)}><Plus className="h-4 w-4" />প্রোডাক্ট যোগ করুন</Button>} />
                              </td></tr>
                            )}
                          </tbody>
                        </table>
                      </div>
                    </div>
                  </div>
                )}

                {/* ═══ Categories ═══ */}
                {activeTab === "categories" && (
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <p className="text-sm text-muted-foreground">মোট {categories.length}টি ক্যাটাগরি</p>
                      <Button size="sm" className="gap-1.5 rounded-xl" onClick={() => { setShowAddCategory(!showAddCategory); setEditingCategory(null); }}>
                        {showAddCategory ? <XCircle className="h-4 w-4" /> : <Plus className="h-4 w-4" />}
                        {showAddCategory ? "বাতিল" : "নতুন ক্যাটাগরি"}
                      </Button>
                    </div>

                    <AnimatePresence>
                      {showAddCategory && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <SectionCard title="নতুন ক্যাটাগরি যোগ করুন" icon={Plus}>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                              <div className="space-y-1.5"><Label className="text-xs">নাম *</Label><Input value={newCategory.name} onChange={e => setNewCategory(c => ({ ...c, name: e.target.value }))} placeholder="ক্যাটাগরির নাম" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">সর্ট অর্ডার</Label><Input type="number" value={newCategory.sort_order} onChange={e => setNewCategory(c => ({ ...c, sort_order: parseInt(e.target.value) || 0 }))} placeholder="0" className="rounded-xl" /></div>
                            </div>
                            <div className="mt-4"><Button size="sm" onClick={handleAddCategory} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />সেভ করুন</Button></div>
                          </SectionCard>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    <AnimatePresence>
                      {editingCategory && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <div className="bg-card rounded-2xl border-2 border-primary/30 overflow-hidden">
                            <div className="px-6 py-4 border-b border-primary/20 bg-primary/5 flex items-center gap-3">
                              <Edit2 className="h-4 w-4 text-primary" />
                              <h3 className="font-heading font-semibold text-sm">ক্যাটাগরি এডিট করুন</h3>
                            </div>
                            <div className="p-6">
                              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <div className="space-y-1.5"><Label className="text-xs">নাম</Label><Input value={editingCategory.name} onChange={e => setEditingCategory(c => c ? { ...c, name: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="space-y-1.5"><Label className="text-xs">সর্ট অর্ডার</Label><Input type="number" value={editingCategory.sort_order} onChange={e => setEditingCategory(c => c ? { ...c, sort_order: parseInt(e.target.value) || 0 } : null)} className="rounded-xl" /></div>
                                <div className="flex items-end">
                                  <div className="flex items-center gap-3 h-10">
                                    <Switch checked={editingCategory.is_active} onCheckedChange={v => setEditingCategory(c => c ? { ...c, is_active: v } : null)} />
                                    <Label className="text-sm">{editingCategory.is_active ? "সক্রিয়" : "নিষ্ক্রিয়"}</Label>
                                  </div>
                                </div>
                              </div>
                              <div className="flex gap-2 mt-4">
                                <Button size="sm" onClick={handleUpdateCategory} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />আপডেট</Button>
                                <Button size="sm" variant="outline" onClick={() => setEditingCategory(null)} className="rounded-xl">বাতিল</Button>
                              </div>
                            </div>
                          </div>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3">
                      {categories.map((c, i) => (
                        <motion.div key={c.id} initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: i * 0.05 }}
                          className={`bg-card rounded-2xl border p-4 transition-all hover:shadow-md ${c.is_active ? "border-border" : "border-border opacity-50"}`}>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                              <div className={`h-10 w-10 rounded-xl flex items-center justify-center ${c.is_active ? "bg-primary/10 text-primary" : "bg-muted text-muted-foreground"}`}>
                                <Tag className="h-5 w-5" />
                              </div>
                              <div>
                                <h4 className="font-semibold text-sm">{c.name}</h4>
                                <p className="text-xs text-muted-foreground">সর্ট: {c.sort_order}</p>
                              </div>
                            </div>
                            <div className="flex items-center gap-1">
                              <span className={`px-2 py-0.5 rounded-lg text-[10px] font-semibold ${c.is_active ? "bg-primary/10 text-primary" : "bg-muted text-muted-foreground"}`}>
                                {c.is_active ? "সক্রিয়" : "নিষ্ক্রিয়"}
                              </span>
                            </div>
                          </div>
                          <div className="flex gap-1 mt-3 pt-3 border-t border-border">
                            <Button variant="ghost" size="sm" className="h-8 text-xs rounded-lg flex-1 gap-1" onClick={() => { setEditingCategory(c); setShowAddCategory(false); }}>
                              <Edit2 className="h-3 w-3" />এডিট
                            </Button>
                            <Button variant="ghost" size="sm" className="h-8 text-xs rounded-lg text-destructive hover:text-destructive flex-1 gap-1"
                              onClick={() => setConfirmDelete({ type: "category", id: c.id, name: c.name })}>
                              <Trash2 className="h-3 w-3" />ডিলিট
                            </Button>
                          </div>
                        </motion.div>
                      ))}
                    </div>
                    {categories.length === 0 && (
                      <EmptyState icon={Tag} title="কোনো ক্যাটাগরি নেই" description="প্রোডাক্ট ক্যাটাগরি যোগ করতে উপরের বাটন ক্লিক করুন।"
                        action={<Button size="sm" className="rounded-xl gap-1.5" onClick={() => setShowAddCategory(true)}><Plus className="h-4 w-4" />ক্যাটাগরি যোগ করুন</Button>} />
                    )}
                  </div>
                )}

                {/* ═══ Orders ═══ */}
                {activeTab === "orders" && (
                  <div className="space-y-4">
                    {/* Top bar: filters + add order */}
                    <div className="flex items-center justify-between flex-wrap gap-3">
                      <div className="flex items-center gap-2 flex-wrap">
                        {(["all", "pending", "hold", "completed", "cancelled", "trashed"] as const).map(f => (
                          <button key={f} onClick={() => { setOrderFilter(f); setSelectedOrders(new Set()); }}
                            className={`px-3 py-1.5 rounded-xl text-xs font-medium transition-all ${orderFilter === f ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground hover:text-foreground"}`}>
                            {f === "all" ? `সব (${activeOrders.length})` : f === "pending" ? `পেন্ডিং (${activeOrders.filter(o=>o.status==="pending").length})` : f === "hold" ? `হোল্ড (${activeOrders.filter(o=>o.status==="hold").length})` : f === "completed" ? `সম্পন্ন (${activeOrders.filter(o=>o.status==="completed").length})` : f === "cancelled" ? `বাতিল (${activeOrders.filter(o=>o.status==="cancelled").length})` : `ট্র্যাশ (${trashedCount})`}
                          </button>
                        ))}
                      </div>
                      <div className="flex items-center gap-2">
                        <div className="relative sm:hidden">
                          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                          <Input placeholder="খুঁজুন..." value={searchQuery} onChange={e => setSearchQuery(e.target.value)}
                            className="pl-9 w-40 h-9 rounded-xl" />
                        </div>
                        <Button size="sm" className="gap-1.5 rounded-xl" onClick={() => setShowAddOrder(!showAddOrder)}>
                          {showAddOrder ? <XCircle className="h-4 w-4" /> : <PlusCircle className="h-4 w-4" />}
                          {showAddOrder ? "বাতিল" : "অর্ডার যোগ করুন"}
                        </Button>
                      </div>
                    </div>

                    {/* Add Order Form */}
                    <AnimatePresence>
                      {showAddOrder && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <SectionCard title="ম্যানুয়াল অর্ডার যোগ করুন" icon={PlusCircle}>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                              <div className="space-y-1.5"><Label className="text-xs">কাস্টমার নাম *</Label><Input value={newOrder.customer_name} onChange={e => setNewOrder(o => ({ ...o, customer_name: e.target.value }))} placeholder="নাম লিখুন" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">ফোন নম্বর</Label><Input value={newOrder.customer_phone} onChange={e => setNewOrder(o => ({ ...o, customer_phone: e.target.value }))} placeholder="01XXXXXXXXX" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">প্রোডাক্ট *</Label>
                                <select value={newOrder.product_name} onChange={e => setNewOrder(o => ({ ...o, product_name: e.target.value }))} className="w-full rounded-xl border border-input bg-background px-3 py-2 text-sm">
                                  <option value="">প্রোডাক্ট সিলেক্ট করুন</option>
                                  {products.map(p => <option key={p.id} value={p.name}>{p.name}</option>)}
                                </select>
                              </div>
                              <div className="space-y-1.5"><Label className="text-xs">মূল্য (৳) *</Label><Input value={newOrder.amount} onChange={e => setNewOrder(o => ({ ...o, amount: e.target.value }))} placeholder="৳500" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">পেমেন্ট মেথড</Label>
                                <select value={newOrder.payment_method} onChange={e => setNewOrder(o => ({ ...o, payment_method: e.target.value }))} className="w-full rounded-xl border border-input bg-background px-3 py-2 text-sm">
                                  <option value="manual">ম্যানুয়াল</option>
                                  <option value="bkash">বিকাশ</option>
                                  <option value="nagad">নগদ</option>
                                  <option value="bank">ব্যাংক</option>
                                  <option value="card">কার্ড</option>
                                </select>
                              </div>
                              <div className="space-y-1.5"><Label className="text-xs">পেমেন্ট স্ট্যাটাস</Label>
                                <select value={newOrder.payment_status} onChange={e => setNewOrder(o => ({ ...o, payment_status: e.target.value }))} className="w-full rounded-xl border border-input bg-background px-3 py-2 text-sm">
                                  <option value="unpaid">আনপেইড</option>
                                  <option value="paid">পেইড</option>
                                </select>
                              </div>
                              <div className="space-y-1.5"><Label className="text-xs">অর্ডার স্ট্যাটাস</Label>
                                <select value={newOrder.status} onChange={e => setNewOrder(o => ({ ...o, status: e.target.value }))} className="w-full rounded-xl border border-input bg-background px-3 py-2 text-sm">
                                  <option value="processing">প্রসেসিং</option>
                                  <option value="pending">পেন্ডিং</option>
                                  <option value="completed">সম্পন্ন</option>
                                  <option value="hold">হোল্ড</option>
                                </select>
                              </div>
                            </div>
                            <div className="mt-4"><Button size="sm" onClick={handleAddOrder} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />অর্ডার সেভ করুন</Button></div>
                          </SectionCard>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    {/* Bulk action bar */}
                    {selectedOrders.size > 0 && (
                      <div className="flex items-center gap-2 p-3 bg-primary/5 border border-primary/20 rounded-xl flex-wrap">
                        <span className="text-sm font-medium">{selectedOrders.size}টি সিলেক্ট করা হয়েছে</span>
                        <div className="flex gap-1.5 flex-wrap ml-auto">
                          {orderFilter !== "trashed" ? (
                            <>
                              <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1" onClick={() => handleBulkAction("completed")}>
                                <CheckCircle2 className="h-3 w-3" />সম্পন্ন
                              </Button>
                              <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1 text-orange-600 border-orange-200" onClick={() => handleBulkAction("cancelled")}>
                                <XCircle className="h-3 w-3" />বাতিল
                              </Button>
                              <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1 text-destructive" onClick={() => handleBulkAction("trash")}>
                                <Trash2 className="h-3 w-3" />ট্র্যাশ
                              </Button>
                            </>
                          ) : (
                            <>
                              <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1" onClick={() => handleBulkAction("restore")}>
                                <RotateCcw className="h-3 w-3" />রিস্টোর
                              </Button>
                              <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1 text-destructive" onClick={() => handleBulkAction("delete")}>
                                <Trash2 className="h-3 w-3" />স্থায়ী ডিলিট
                              </Button>
                            </>
                          )}
                        </div>
                      </div>
                    )}

                    <div className="bg-card rounded-2xl border border-border overflow-hidden">
                      <div className="overflow-x-auto">
                        <table className="w-full text-sm">
                          <thead>
                            <tr className="border-b border-border text-muted-foreground bg-muted/30">
                              <th className="px-3 py-3 w-10">
                                <input type="checkbox" checked={filteredOrders.length > 0 && selectedOrders.size === filteredOrders.length}
                                  onChange={toggleSelectAll} className="rounded border-border" />
                              </th>
                              <th className="text-left px-4 py-3 font-medium">কাস্টমার</th>
                              <th className="text-left px-4 py-3 font-medium hidden md:table-cell">ফোন</th>
                              <th className="text-left px-4 py-3 font-medium hidden lg:table-cell">প্রোডাক্ট</th>
                              <th className="text-left px-4 py-3 font-medium">মূল্য</th>
                              <th className="text-left px-4 py-3 font-medium">পেমেন্ট</th>
                              <th className="text-left px-4 py-3 font-medium">স্ট্যাটাস</th>
                              <th className="text-right px-4 py-3 font-medium">অ্যাকশন</th>
                            </tr>
                          </thead>
                          <tbody>
                            {filteredOrders.map((o, i) => (
                              <motion.tr key={o.id} initial={{ opacity: 0 }} animate={{ opacity: 1 }} transition={{ delay: i * 0.03 }}
                                className={`border-b border-border last:border-0 hover:bg-muted/30 transition-colors ${selectedOrders.has(o.id) ? "bg-primary/5" : ""}`}>
                                <td className="px-3 py-3.5">
                                  <input type="checkbox" checked={selectedOrders.has(o.id)} onChange={() => toggleSelectOrder(o.id)} className="rounded border-border" />
                                </td>
                                <td className="px-4 py-3.5">
                                  <div>
                                    <span className="font-medium">{o.customer_name}</span>
                                    <p className="text-xs text-muted-foreground md:hidden">{o.product_name}</p>
                                    <p className="text-[10px] text-muted-foreground">{new Date(o.created_at).toLocaleDateString("bn-BD", { day: "numeric", month: "short", year: "numeric", hour: "2-digit", minute: "2-digit" })}</p>
                                  </div>
                                </td>
                                <td className="px-4 py-3.5 hidden md:table-cell text-muted-foreground">{o.customer_phone || "—"}</td>
                                <td className="px-4 py-3.5 hidden lg:table-cell text-muted-foreground max-w-[200px] truncate">{o.product_name}</td>
                                <td className="px-4 py-3.5 font-semibold">{o.amount}</td>
                                <td className="px-4 py-3.5">
                                  <span className={`inline-flex items-center px-2.5 py-1 rounded-lg text-xs font-semibold ${o.payment_status === "paid" ? "bg-primary/10 text-primary" : "bg-amber-500/10 text-amber-600"}`}>
                                    {o.payment_status === "paid" ? "পেইড" : "আনপেইড"}
                                  </span>
                                </td>
                                <td className="px-4 py-3.5">
                                  <span className={`inline-flex items-center px-2.5 py-1 rounded-lg text-xs font-semibold ${statusColor(o.status)}`}>
                                    {statusText(o.status)}
                                  </span>
                                </td>
                                <td className="px-4 py-3.5">
                                  <div className="flex gap-1 justify-end flex-wrap">
                                    <Button size="sm" variant="ghost" className="h-7 w-7 p-0 rounded-lg" onClick={async () => {
                                      setViewingOrder(o);
                                      setViewingOrderEmail(null);
                                      if (o.user_id) {
                                        const { data } = await supabase.from("profiles").select("email, name, phone").eq("user_id", o.user_id).maybeSingle();
                                        if (data?.email) setViewingOrderEmail(data.email);
                                      }
                                    }}>
                                      <Eye className="h-3.5 w-3.5" />
                                    </Button>
                                    {orderFilter === "trashed" ? (
                                      <>
                                        <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1" onClick={() => handleRestoreOrder(o.id)}>
                                          <RotateCcw className="h-3 w-3" />রিস্টোর
                                        </Button>
                                        <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1 text-destructive" onClick={() => handlePermanentDelete(o.id)}>
                                          <Trash2 className="h-3 w-3" />ডিলিট
                                        </Button>
                                      </>
                                    ) : (o.status === "pending" || o.status === "hold" || o.status === "processing") ? (
                                      <>
                                        <Button size="sm" className="h-7 text-xs rounded-lg gap-1" onClick={() => handleOrderStatus(o.id, "completed")}>
                                          <CheckCircle2 className="h-3 w-3" />সম্পন্ন
                                        </Button>
                                        {o.status !== "hold" && (
                                          <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg text-orange-600 border-orange-200 hover:bg-orange-50" onClick={() => handleOrderStatus(o.id, "hold")}>
                                            <Clock className="h-3 w-3" />হোল্ড
                                          </Button>
                                        )}
                                        <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg text-destructive" onClick={() => handleOrderStatus(o.id, "cancelled")}>
                                          <XCircle className="h-3 w-3" />বাতিল
                                        </Button>
                                        <Button size="sm" variant="ghost" className="h-7 text-xs rounded-lg text-muted-foreground" onClick={() => handleTrashOrder(o.id)}>
                                          <Archive className="h-3 w-3" />
                                        </Button>
                                      </>
                                    ) : (
                                      <Button size="sm" variant="ghost" className="h-7 text-xs rounded-lg text-muted-foreground" onClick={() => handleTrashOrder(o.id)}>
                                        <Archive className="h-3 w-3" />ট্র্যাশ
                                      </Button>
                                    )}
                                  </div>
                                </td>
                              </motion.tr>
                            ))}
                            {filteredOrders.length === 0 && (
                              <tr><td colSpan={8}>
                                <EmptyState icon={ShoppingCart} title={orderFilter === "trashed" ? "ট্র্যাশ খালি" : "কোনো অর্ডার নেই"} description={orderFilter === "trashed" ? "ট্র্যাশে কোনো অর্ডার নেই।" : "এখনো কোনো অর্ডার আসেনি।"} />
                              </td></tr>
                            )}
                          </tbody>
                        </table>
                      </div>
                    </div>

                    {/* Order Detail Panel */}
                    <AnimatePresence>
                      {viewingOrder && (
                        <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
                          className="fixed inset-0 z-50 flex items-end sm:items-center justify-center bg-black/40 p-0 sm:p-4" onClick={() => setViewingOrder(null)}>
                          <motion.div initial={{ y: 100, opacity: 0 }} animate={{ y: 0, opacity: 1 }} exit={{ y: 100, opacity: 0 }} transition={{ type: "spring", damping: 25, stiffness: 300 }}
                            className="bg-card rounded-t-2xl sm:rounded-2xl border border-border shadow-2xl w-full sm:max-w-lg max-h-[92vh] sm:max-h-[85vh] overflow-y-auto" onClick={e => e.stopPropagation()}>
                            {/* Drag handle for mobile */}
                            <div className="flex justify-center pt-2 sm:hidden"><div className="w-10 h-1 rounded-full bg-muted-foreground/30" /></div>
                            <div className="px-4 sm:px-6 py-3 sm:py-4 border-b border-border bg-muted/30 sm:rounded-t-2xl flex items-center justify-between">
                              <div className="flex items-center gap-2 sm:gap-3">
                                <Eye className="h-4 w-4 sm:h-5 sm:w-5 text-primary" />
                                <h3 className="font-heading font-semibold text-sm sm:text-base">অর্ডার বিস্তারিত</h3>
                              </div>
                              <Button variant="ghost" size="icon" className="h-8 w-8 rounded-lg" onClick={() => setViewingOrder(null)}>
                                <X className="h-4 w-4" />
                              </Button>
                            </div>
                            <div className="p-4 sm:p-6 space-y-3 sm:space-y-4">
                              {/* Order ID & Date */}
                              <div className="grid grid-cols-2 gap-3">
                                <div className="space-y-1">
                                  <p className="text-[10px] text-muted-foreground uppercase tracking-wider font-semibold">অর্ডার ID</p>
                                  <p className="text-xs font-mono bg-muted px-2 py-1.5 rounded-lg break-all">{viewingOrder.id}</p>
                                </div>
                                <div className="space-y-1">
                                  <p className="text-[10px] text-muted-foreground uppercase tracking-wider font-semibold">তারিখ</p>
                                  <p className="text-sm">{new Date(viewingOrder.created_at).toLocaleDateString("bn-BD", { day: "numeric", month: "long", year: "numeric", hour: "2-digit", minute: "2-digit" })}</p>
                                </div>
                              </div>

                              {/* Customer Info */}
                              <div className="border border-border rounded-xl p-3 sm:p-4 space-y-2 sm:space-y-2.5 bg-muted/20">
                                <p className="text-xs font-semibold text-muted-foreground flex items-center gap-1.5"><Users className="h-3.5 w-3.5" />কাস্টমার তথ্য</p>
                                <div className="grid grid-cols-1 gap-2 sm:grid-cols-2 sm:gap-2.5 text-sm">
                                  <div className="flex items-center gap-2">
                                    <Users className="h-3.5 w-3.5 text-muted-foreground flex-shrink-0" />
                                    <div className="min-w-0">
                                      <p className="text-[10px] text-muted-foreground">নাম</p>
                                      <p className="font-medium truncate">{viewingOrder.customer_name}</p>
                                    </div>
                                  </div>
                                  <div className="flex items-center gap-2">
                                    <Phone className="h-3.5 w-3.5 text-muted-foreground flex-shrink-0" />
                                    <div className="min-w-0">
                                      <p className="text-[10px] text-muted-foreground">ফোন</p>
                                      <p className="font-medium">{viewingOrder.customer_phone || "—"}</p>
                                    </div>
                                  </div>
                                  <div className="flex items-center gap-2 sm:col-span-2">
                                    <Mail className="h-3.5 w-3.5 text-muted-foreground flex-shrink-0" />
                                    <div className="min-w-0">
                                      <p className="text-[10px] text-muted-foreground">ইমেইল</p>
                                      <p className="font-medium truncate">{viewingOrderEmail || (viewingOrder.user_id ? "লোড হচ্ছে..." : "গেস্ট অর্ডার")}</p>
                                    </div>
                                  </div>
                                  {viewingOrder.user_id && (
                                    <div className="flex items-center gap-2 sm:col-span-2">
                                      <Shield className="h-3.5 w-3.5 text-muted-foreground flex-shrink-0" />
                                      <div className="min-w-0">
                                        <p className="text-[10px] text-muted-foreground">User ID</p>
                                        <p className="font-mono text-[11px] break-all">{viewingOrder.user_id}</p>
                                      </div>
                                    </div>
                                  )}
                                </div>
                              </div>

                              <div className="border border-border rounded-xl p-3 sm:p-4 space-y-2 sm:space-y-2.5 bg-muted/20">
                                <p className="text-xs font-semibold text-muted-foreground flex items-center gap-1.5"><Package className="h-3.5 w-3.5" />প্রোডাক্ট ও পেমেন্ট</p>
                                <div className="grid grid-cols-2 gap-2 sm:gap-2.5 text-sm">
                                  <div className="col-span-2">
                                    <p className="text-[10px] text-muted-foreground">প্রোডাক্ট</p>
                                    <p className="font-medium text-sm">{viewingOrder.product_name}</p>
                                  </div>
                                  <div>
                                    <p className="text-[10px] text-muted-foreground">মূল্য</p>
                                    <p className="font-semibold text-primary text-base sm:text-lg">{viewingOrder.amount}</p>
                                  </div>
                                  <div>
                                    <p className="text-[10px] text-muted-foreground">পেমেন্ট মেথড</p>
                                    <p className="font-medium capitalize text-sm">{viewingOrder.payment_method || "—"}</p>
                                  </div>
                                  {viewingOrder.transaction_id && (
                                    <div className="col-span-2">
                                      <p className="text-[10px] text-muted-foreground">Transaction ID</p>
                                      <p className="font-mono text-[11px] bg-muted px-2 py-1.5 rounded-lg break-all">{viewingOrder.transaction_id}</p>
                                    </div>
                                  )}
                                </div>
                              </div>

                              {/* Quick Actions: Status & Payment Update */}
                              <div className="border border-border rounded-xl p-3 sm:p-4 space-y-2.5 sm:space-y-3 bg-muted/20">
                                <p className="text-xs font-semibold text-muted-foreground flex items-center gap-1.5"><Settings className="h-3.5 w-3.5" />দ্রুত আপডেট</p>
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                                  {/* Order Status */}
                                  <div className="space-y-1.5">
                                    <Label className="text-xs text-muted-foreground">অর্ডার স্ট্যাটাস</Label>
                                    <select
                                      className="w-full h-9 rounded-lg border border-border bg-background px-3 text-sm focus:outline-none focus:ring-2 focus:ring-primary/30"
                                      value={viewingOrder.status}
                                      onChange={async (e) => {
                                        const newStatus = e.target.value;
                                        const { error } = await supabase.from("orders").update({ status: newStatus }).eq("id", viewingOrder.id);
                                        if (!error) {
                                          setViewingOrder({ ...viewingOrder, status: newStatus });
                                          setOrders(prev => prev.map(o => o.id === viewingOrder.id ? { ...o, status: newStatus } : o));
                                          toast({ title: "স্ট্যাটাস আপডেট হয়েছে" });
                                        } else {
                                          toast({ title: "আপডেট ব্যর্থ", variant: "destructive" });
                                        }
                                      }}
                                    >
                                      <option value="processing">⚙️ প্রসেসিং</option>
                                      <option value="pending">⏳ পেন্ডিং</option>
                                      <option value="hold">⏸️ হোল্ড</option>
                                      <option value="completed">✅ সম্পন্ন</option>
                                      <option value="cancelled">❌ বাতিল</option>
                                    </select>
                                  </div>
                                  {/* Payment Status */}
                                  <div className="space-y-1.5">
                                    <Label className="text-xs text-muted-foreground">পেমেন্ট স্ট্যাটাস</Label>
                                    <select
                                      className="w-full h-9 rounded-lg border border-border bg-background px-3 text-sm focus:outline-none focus:ring-2 focus:ring-primary/30"
                                      value={viewingOrder.payment_status || "unpaid"}
                                      onChange={async (e) => {
                                        const newPayStatus = e.target.value;
                                        const { error } = await supabase.from("orders").update({ payment_status: newPayStatus }).eq("id", viewingOrder.id);
                                        if (!error) {
                                          setViewingOrder({ ...viewingOrder, payment_status: newPayStatus });
                                          setOrders(prev => prev.map(o => o.id === viewingOrder.id ? { ...o, payment_status: newPayStatus } : o));
                                          toast({ title: "পেমেন্ট স্ট্যাটাস আপডেট হয়েছে" });
                                        } else {
                                          toast({ title: "আপডেট ব্যর্থ", variant: "destructive" });
                                        }
                                      }}
                                    >
                                      <option value="unpaid">⏳ আনপেইড</option>
                                      <option value="paid">✅ পেইড</option>
                                    </select>
                                  </div>
                                </div>
                                <div className="flex items-center gap-2 pt-1 flex-wrap">
                                  <span className={`inline-flex items-center px-2 sm:px-2.5 py-1 rounded-lg text-[11px] sm:text-xs font-semibold ${statusColor(viewingOrder.status)}`}>
                                    {statusText(viewingOrder.status)}
                                  </span>
                                  <span className={`inline-flex items-center px-2 sm:px-2.5 py-1 rounded-lg text-[11px] sm:text-xs font-semibold ${viewingOrder.payment_status === "paid" ? "bg-primary/10 text-primary" : "bg-amber-500/10 text-amber-600"}`}>
                                    {viewingOrder.payment_status === "paid" ? "✅ পেইড" : "⏳ আনপেইড"}
                                  </span>
                                </div>
                              </div>
                            </div>
                          </motion.div>
                        </motion.div>
                      )}
                    </AnimatePresence>
                  </div>
                )}

                {/* ═══ Banner ═══ */}
                {activeTab === "banner" && (
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <p className="text-sm text-muted-foreground">মোট {banners.length}টি ব্যানার</p>
                      <Button size="sm" className="gap-1.5 rounded-xl" onClick={() => { setShowAddBanner(!showAddBanner); setEditingBanner(null); }}>
                        {showAddBanner ? <XCircle className="h-4 w-4" /> : <Plus className="h-4 w-4" />}
                        {showAddBanner ? "বাতিল" : "নতুন ব্যানার"}
                      </Button>
                    </div>

                    <AnimatePresence>
                      {showAddBanner && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <SectionCard title="নতুন ব্যানার যোগ করুন" icon={Plus}>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                              <div className="space-y-1.5"><Label className="text-xs">টাইটেল *</Label><Input value={newBanner.title} onChange={e => setNewBanner(b => ({ ...b, title: e.target.value }))} placeholder="Premium Digital Products" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">সাবটাইটেল</Label><Input value={newBanner.subtitle} onChange={e => setNewBanner(b => ({ ...b, subtitle: e.target.value }))} placeholder="#1 Digital Marketplace" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">CTA বাটন টেক্সট</Label><Input value={newBanner.cta_text} onChange={e => setNewBanner(b => ({ ...b, cta_text: e.target.value }))} placeholder="Shop Now" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">CTA লিংক</Label><Input value={newBanner.cta_link} onChange={e => setNewBanner(b => ({ ...b, cta_link: e.target.value }))} placeholder="#products" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">সেকেন্ডারি বাটন</Label><Input value={newBanner.secondary_cta_text} onChange={e => setNewBanner(b => ({ ...b, secondary_cta_text: e.target.value }))} placeholder="Best Selling" className="rounded-xl" /></div>
                              <div className="space-y-1.5"><Label className="text-xs">সেকেন্ডারি লিংক</Label><Input value={newBanner.secondary_cta_link} onChange={e => setNewBanner(b => ({ ...b, secondary_cta_link: e.target.value }))} placeholder="#best-selling" className="rounded-xl" /></div>
                              <div className="md:col-span-2 space-y-1.5">
                                <Label className="text-xs">ব্যানার ইমেজ</Label>
                                <input type="file" accept="image/*" ref={fileInputRef} onChange={handleNewBannerImage} className="hidden" />
                                <div className="flex items-center gap-3">
                                  <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingImage} onClick={() => fileInputRef.current?.click()}>
                                    {uploadingImage ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                                    {uploadingImage ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড"}
                                  </Button>
                                  {newBanner.image_url && <img src={newBanner.image_url} alt="preview" className="h-10 w-16 rounded-xl object-cover border border-border" />}
                                </div>
                              </div>
                              <div className="md:col-span-2 space-y-1.5"><Label className="text-xs">বিবরণ</Label><Input value={newBanner.description} onChange={e => setNewBanner(b => ({ ...b, description: e.target.value }))} placeholder="ব্যানারের বিবরণ" className="rounded-xl" /></div>
                            </div>
                            <div className="mt-4"><Button size="sm" onClick={handleAddBanner} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />সেভ করুন</Button></div>
                          </SectionCard>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    <AnimatePresence>
                      {editingBanner && (
                        <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                          <div className="bg-card rounded-2xl border-2 border-primary/30 overflow-hidden">
                            <div className="px-6 py-4 border-b border-primary/20 bg-primary/5 flex items-center gap-3">
                              <Edit2 className="h-4 w-4 text-primary" />
                              <h3 className="font-heading font-semibold text-sm">ব্যানার এডিট করুন</h3>
                            </div>
                            <div className="p-6">
                              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div className="space-y-1.5"><Label className="text-xs">টাইটেল</Label><Input value={editingBanner.title} onChange={e => setEditingBanner(b => b ? { ...b, title: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="space-y-1.5"><Label className="text-xs">সাবটাইটেল</Label><Input value={editingBanner.subtitle || ""} onChange={e => setEditingBanner(b => b ? { ...b, subtitle: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="space-y-1.5"><Label className="text-xs">CTA বাটন</Label><Input value={editingBanner.cta_text || ""} onChange={e => setEditingBanner(b => b ? { ...b, cta_text: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="space-y-1.5"><Label className="text-xs">CTA লিংক</Label><Input value={editingBanner.cta_link || ""} onChange={e => setEditingBanner(b => b ? { ...b, cta_link: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="space-y-1.5"><Label className="text-xs">সেকেন্ডারি বাটন</Label><Input value={editingBanner.secondary_cta_text || ""} onChange={e => setEditingBanner(b => b ? { ...b, secondary_cta_text: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="space-y-1.5"><Label className="text-xs">সেকেন্ডারি লিংক</Label><Input value={editingBanner.secondary_cta_link || ""} onChange={e => setEditingBanner(b => b ? { ...b, secondary_cta_link: e.target.value } : null)} className="rounded-xl" /></div>
                                <div className="md:col-span-2 space-y-1.5">
                                  <Label className="text-xs">ব্যানার ইমেজ</Label>
                                  <input type="file" accept="image/*" ref={editFileInputRef} onChange={handleEditBannerImage} className="hidden" />
                                  <div className="flex items-center gap-3">
                                    <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingImage} onClick={() => editFileInputRef.current?.click()}>
                                      {uploadingImage ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                                      {uploadingImage ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড"}
                                    </Button>
                                    {editingBanner?.image_url && <img src={editingBanner.image_url} alt="preview" className="h-10 w-16 rounded-xl object-cover border border-border" />}
                                  </div>
                                </div>
                                <div className="md:col-span-2 space-y-1.5"><Label className="text-xs">বিবরণ</Label><Input value={editingBanner.description || ""} onChange={e => setEditingBanner(b => b ? { ...b, description: e.target.value } : null)} className="rounded-xl" /></div>
                              </div>
                              <div className="flex items-center gap-3 mt-4">
                                <Switch checked={editingBanner.is_active} onCheckedChange={c => setEditingBanner(b => b ? { ...b, is_active: c } : null)} />
                                <Label className="text-sm">{editingBanner.is_active ? "অ্যাক্টিভ" : "ইনঅ্যাক্টিভ"}</Label>
                              </div>
                              <div className="flex gap-2 mt-4">
                                <Button size="sm" onClick={handleUpdateBanner} className="gap-1.5 rounded-xl"><Save className="h-4 w-4" />আপডেট</Button>
                                <Button size="sm" variant="outline" onClick={() => setEditingBanner(null)} className="rounded-xl">বাতিল</Button>
                              </div>
                            </div>
                          </div>
                        </motion.div>
                      )}
                    </AnimatePresence>

                    {/* Banner Cards */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                      {banners.map((b, i) => (
                        <motion.div key={b.id} initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: i * 0.05 }}
                          className={`bg-card rounded-2xl border overflow-hidden transition-all hover:shadow-lg ${b.is_active ? "border-border" : "border-border opacity-50"}`}>
                          {b.image_url && (
                            <div className="h-32 overflow-hidden">
                              <img src={b.image_url} alt={b.title} className="w-full h-full object-cover" />
                            </div>
                          )}
                          <div className="p-4">
                            <div className="flex items-start justify-between gap-2 mb-2">
                              <div className="min-w-0">
                                <div className="flex items-center gap-2 mb-1">
                                  <h4 className="font-heading font-semibold text-sm truncate">{b.title}</h4>
                                  <span className={`px-2 py-0.5 rounded-lg text-[10px] font-semibold flex-shrink-0 ${b.is_active ? "bg-primary/10 text-primary" : "bg-muted text-muted-foreground"}`}>
                                    {b.is_active ? "অ্যাক্টিভ" : "ইনঅ্যাক্টিভ"}
                                  </span>
                                </div>
                                {b.subtitle && <p className="text-xs text-muted-foreground">{b.subtitle}</p>}
                              </div>
                              <Switch checked={b.is_active} onCheckedChange={() => handleToggleBannerActive(b)} />
                            </div>
                            {b.description && <p className="text-xs text-muted-foreground line-clamp-2 mb-3">{b.description}</p>}
                            <div className="flex flex-wrap gap-1.5 mb-3">
                              {b.cta_text && <span className="text-[10px] bg-primary/10 text-primary px-2 py-0.5 rounded-lg font-medium">{b.cta_text}</span>}
                              {b.secondary_cta_text && <span className="text-[10px] bg-muted text-muted-foreground px-2 py-0.5 rounded-lg">{b.secondary_cta_text}</span>}
                            </div>
                            <div className="flex gap-1 pt-3 border-t border-border">
                              <Button variant="ghost" size="sm" className="h-8 text-xs rounded-lg flex-1 gap-1" onClick={() => { setEditingBanner(b); setShowAddBanner(false); }}>
                                <Edit2 className="h-3 w-3" />এডিট
                              </Button>
                              <Button variant="ghost" size="sm" className="h-8 text-xs rounded-lg text-destructive hover:text-destructive flex-1 gap-1"
                                onClick={() => setConfirmDelete({ type: "banner", id: b.id, name: b.title })}>
                                <Trash2 className="h-3 w-3" />ডিলিট
                              </Button>
                            </div>
                          </div>
                        </motion.div>
                      ))}
                    </div>
                    {banners.length === 0 && (
                      <EmptyState icon={Image} title="কোনো ব্যানার নেই" description="হোম পেজে ব্যানার যোগ করতে উপরের বাটন ক্লিক করুন।"
                        action={<Button size="sm" className="rounded-xl gap-1.5" onClick={() => setShowAddBanner(true)}><Plus className="h-4 w-4" />ব্যানার যোগ করুন</Button>} />
                    )}
                  </div>
                )}

                {/* ═══ Product Keys ═══ */}
                {activeTab === "product_keys" && <AdminProductKeys />}

                {/* ═══ Partners ═══ */}
                {activeTab === "partners" && <AdminPartnersPanel fetchData={fetchData} />}

                {/* ═══ SMS ═══ */}
                {activeTab === "sms" && <AdminSmsPanel sendSms={sendSms} orders={orders} />}

                {/* ═══ WhatsApp ═══ */}
                {activeTab === "whatsapp" && <AdminWhatsAppPanel orders={orders} />}

                {/* ═══ Email / SMTP ═══ */}
                {activeTab === "email" && <AdminEmailPanel />}

                {/* ═══ Blog ═══ */}
                {activeTab === "blog" && <AdminBlogPanel fetchData={fetchData} />}

                {/* ═══ Product Requests ═══ */}
                {activeTab === "product_requests" && <AdminProductRequestsPanel />}

                {/* ═══ Promo Codes ═══ */}
                {activeTab === "promo_codes" && <AdminPromoPanel />}

                {/* ═══ Users ═══ */}
                {activeTab === "users" && <AdminUsersPanel />}
                {activeTab === "subscribers" && <AdminSubscribersPanel />}
                {activeTab === "voice_call" && <AdminVoiceCallPanel />}
                {activeTab === "auth_settings" && <AdminAuthSettingsPanel />}
                {activeTab === "pixel" && <AdminPixelPanel />}
                {activeTab === "reviews" && <AdminReviewsPanel />}
                {activeTab === "faq" && <AdminFaqPanel />}
                {activeTab === "popup" && <AdminPopupPanel />}
                {activeTab === "incomplete_orders" && <AdminIncompleteOrdersPanel />}

                {activeTab === "settings" && (
                  <div className="space-y-6 max-w-3xl">
                    {/* Logo */}
                    <SectionCard title="সাইট লোগো" description="হেডারে দেখানো লোগো আপলোড বা পরিবর্তন করুন" icon={Image}>
                      <div className="flex items-center gap-4 mb-4">
                        {logoUrl ? (
                          <img src={logoUrl} alt="Site Logo" className="h-16 max-w-[200px] object-contain rounded-xl border border-border p-2" />
                        ) : (
                          <div className="h-16 w-40 rounded-xl border-2 border-dashed border-border flex items-center justify-center text-muted-foreground text-sm">কোনো লোগো নেই</div>
                        )}
                      </div>
                      <input type="file" accept="image/*" ref={logoFileRef} onChange={handleLogoUpload} className="hidden" />
                      <div className="flex gap-2">
                        <Button variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingLogo} onClick={() => logoFileRef.current?.click()}>
                          {uploadingLogo ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                          {uploadingLogo ? "আপলোড হচ্ছে..." : logoUrl ? "লোগো পরিবর্তন" : "লোগো আপলোড"}
                        </Button>
                        {logoUrl && (
                          <Button variant="outline" size="sm" className="gap-1.5 rounded-xl text-destructive hover:text-destructive" onClick={handleRemoveLogo}>
                            <Trash2 className="h-4 w-4" />রিমুভ
                          </Button>
                        )}
                      </div>
                    </SectionCard>

                    {/* Favicon */}
                    <SectionCard title="Favicon" description="ব্রাউজার ট্যাবে দেখানো ছোট আইকন" icon={Globe}>
                      <div className="flex items-center gap-4 mb-4">
                        {siteFields.favicon_url ? (
                          <img src={siteFields.favicon_url} alt="Favicon" className="h-10 w-10 object-contain rounded-xl border border-border p-1" />
                        ) : (
                          <div className="h-10 w-10 rounded-xl border-2 border-dashed border-border flex items-center justify-center text-muted-foreground text-xs">N/A</div>
                        )}
                      </div>
                      <input type="file" accept="image/png,image/ico,image/svg+xml,image/x-icon" ref={faviconFileRef} onChange={handleFaviconUpload} className="hidden" />
                      <Button variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingFavicon} onClick={() => faviconFileRef.current?.click()}>
                        {uploadingFavicon ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                        {uploadingFavicon ? "আপলোড হচ্ছে..." : siteFields.favicon_url ? "পরিবর্তন" : "আপলোড"}
                      </Button>
                    </SectionCard>

                    {/* Site Info */}
                    <SectionCard title="সাইট টাইটেল ও বিবরণ" description="SEO ও ব্রাউজার ট্যাব তথ্য" icon={Globe}>
                      <div className="space-y-4">
                        <div className="space-y-1.5"><Label className="text-xs">সাইট টাইটেল</Label><Input value={siteFields.site_title} onChange={e => setSiteFields(f => ({ ...f, site_title: e.target.value }))} className="rounded-xl" /></div>
                        <div className="space-y-1.5"><Label className="text-xs">সাইট বিবরণ (Meta Description)</Label><Input value={siteFields.site_description} onChange={e => setSiteFields(f => ({ ...f, site_description: e.target.value }))} className="rounded-xl" /></div>
                      </div>
                    </SectionCard>

                    {/* Contact */}
                    <SectionCard title="যোগাযোগ তথ্য" description="সাইটে দেখানো যোগাযোগ তথ্য" icon={Phone}>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5"><Phone className="h-3 w-3" />ফোন নম্বর</Label>
                          <Input value={siteFields.phone} onChange={e => setSiteFields(f => ({ ...f, phone: e.target.value }))} className="rounded-xl" />
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5"><Mail className="h-3 w-3" />ইমেইল</Label>
                          <Input value={siteFields.email} onChange={e => setSiteFields(f => ({ ...f, email: e.target.value }))} className="rounded-xl" />
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5"><Facebook className="h-3 w-3" />Facebook URL</Label>
                          <Input value={siteFields.facebook_url} onChange={e => setSiteFields(f => ({ ...f, facebook_url: e.target.value }))} className="rounded-xl" />
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5"><MessageCircle className="h-3 w-3" />WhatsApp নম্বর</Label>
                          <Input value={siteFields.whatsapp_number} onChange={e => setSiteFields(f => ({ ...f, whatsapp_number: e.target.value }))} className="rounded-xl" />
                        </div>
                        <div className="md:col-span-2 space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5"><MapPin className="h-3 w-3" />ঠিকানা</Label>
                          <Input value={siteFields.address} onChange={e => setSiteFields(f => ({ ...f, address: e.target.value }))} className="rounded-xl" />
                        </div>
                      </div>
                    </SectionCard>

                    {/* Custom Sidebar Links */}
                    <SectionCard title="সাইডবার কাস্টম লিংক" description="অ্যাডমিন সাইডবারে কাস্টম লিংক যোগ করুন" icon={ExternalLink}>
                      <div className="space-y-4">
                        {customLinks.map((link, i) => (
                          <div key={i} className="flex items-center gap-3 p-3 rounded-xl bg-muted/50 border border-border">
                            <span className="text-lg">{link.icon}</span>
                            <div className="flex-1 min-w-0">
                              <p className="text-sm font-medium truncate">{link.label}</p>
                              <p className="text-[11px] text-muted-foreground truncate">{link.url}</p>
                            </div>
                            <Button variant="ghost" size="icon" className="h-8 w-8 rounded-lg text-destructive hover:text-destructive flex-shrink-0"
                              onClick={async () => {
                                const updated = customLinks.filter((_, idx) => idx !== i);
                                setCustomLinks(updated);
                                await upsertSetting("sidebar_custom_links", JSON.stringify(updated));
                                toast({ title: "✅ লিংক মুছে ফেলা হয়েছে!" });
                              }}>
                              <Trash2 className="h-3.5 w-3.5" />
                            </Button>
                          </div>
                        ))}

                        {customLinks.length === 0 && !showAddCustomLink && (
                          <p className="text-sm text-muted-foreground text-center py-4">কোনো কাস্টম লিংক নেই</p>
                        )}

                        <AnimatePresence>
                          {showAddCustomLink && (
                            <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
                              <div className="p-4 rounded-xl bg-muted/30 border border-border space-y-3">
                                <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
                                  <div className="space-y-1">
                                    <Label className="text-xs">আইকন (Emoji)</Label>
                                    <Input value={newCustomLink.icon} onChange={e => setNewCustomLink(l => ({ ...l, icon: e.target.value }))} placeholder="🔗" className="rounded-xl" maxLength={4} />
                                  </div>
                                  <div className="space-y-1">
                                    <Label className="text-xs">লেবেল *</Label>
                                    <Input value={newCustomLink.label} onChange={e => setNewCustomLink(l => ({ ...l, label: e.target.value }))} placeholder="Facebook Page" className="rounded-xl" />
                                  </div>
                                  <div className="space-y-1">
                                    <Label className="text-xs">URL *</Label>
                                    <Input value={newCustomLink.url} onChange={e => setNewCustomLink(l => ({ ...l, url: e.target.value }))} placeholder="https://facebook.com/..." className="rounded-xl" />
                                  </div>
                                </div>
                                <div className="flex gap-2">
                                  <Button size="sm" className="gap-1.5 rounded-xl" onClick={async () => {
                                    if (!newCustomLink.label || !newCustomLink.url) { toast({ title: "লেবেল ও URL দিন", variant: "destructive" }); return; }
                                    const updated = [...customLinks, { ...newCustomLink }];
                                    setCustomLinks(updated);
                                    await upsertSetting("sidebar_custom_links", JSON.stringify(updated));
                                    setNewCustomLink({ label: "", url: "", icon: "🔗" });
                                    setShowAddCustomLink(false);
                                    toast({ title: "✅ কাস্টম লিংক যোগ হয়েছে!" });
                                  }}>
                                    <Save className="h-4 w-4" />যোগ করুন
                                  </Button>
                                  <Button size="sm" variant="outline" className="rounded-xl" onClick={() => setShowAddCustomLink(false)}>বাতিল</Button>
                                </div>
                              </div>
                            </motion.div>
                          )}
                        </AnimatePresence>

                        {!showAddCustomLink && (
                          <Button size="sm" variant="outline" className="gap-1.5 rounded-xl" onClick={() => setShowAddCustomLink(true)}>
                            <Plus className="h-4 w-4" />নতুন লিংক যোগ করুন
                          </Button>
                        )}
                      </div>
                    </SectionCard>

                    {/* Floating Support Button */}
                    <SectionCard title="ফ্লোটিং সাপোর্ট বাটন" description="ওয়েবসাইটে WhatsApp, Messenger, Telegram সাপোর্ট বাটন দেখান" icon={MessageCircle}>
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5">💬 WhatsApp লিংক</Label>
                          <Input value={siteFields.support_whatsapp} onChange={e => setSiteFields(f => ({ ...f, support_whatsapp: e.target.value }))} placeholder="https://wa.me/8801XXXXXXXXX" className="rounded-xl" />
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5">💙 Messenger লিংক</Label>
                          <Input value={siteFields.support_messenger} onChange={e => setSiteFields(f => ({ ...f, support_messenger: e.target.value }))} placeholder="https://m.me/pagename" className="rounded-xl" />
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs flex items-center gap-1.5">✈️ Telegram লিংক</Label>
                          <Input value={siteFields.support_telegram} onChange={e => setSiteFields(f => ({ ...f, support_telegram: e.target.value }))} placeholder="https://t.me/username" className="rounded-xl" />
                        </div>
                      </div>
                      <p className="text-[11px] text-muted-foreground mt-3">খালি রাখলে সেই চ্যানেল দেখাবে না। কমপক্ষে একটি লিংক দিলে বাটন দেখাবে।</p>
                    </SectionCard>

                    {/* App Install Banner */}
                    <SectionCard title="অ্যাপ ইনস্টল ব্যানার" description="ওয়েবসাইটের উপরে অ্যাপ ইনস্টল ব্যানার দেখান" icon={Globe}>
                      <div className="space-y-4">
                        <div className="flex items-center gap-3">
                          <Switch checked={siteFields.app_install_enabled === "true"} onCheckedChange={v => setSiteFields(f => ({ ...f, app_install_enabled: v ? "true" : "false" }))} />
                          <Label className="text-sm font-medium">{siteFields.app_install_enabled === "true" ? "✅ চালু" : "বন্ধ"}</Label>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div className="space-y-1.5">
                            <Label className="text-xs">টাইটেল</Label>
                            <Input value={siteFields.app_install_title} onChange={e => setSiteFields(f => ({ ...f, app_install_title: e.target.value }))} placeholder="Install App" className="rounded-xl" />
                          </div>
                          <div className="space-y-1.5">
                            <Label className="text-xs">সাবটাইটেল</Label>
                            <Input value={siteFields.app_install_subtitle} onChange={e => setSiteFields(f => ({ ...f, app_install_subtitle: e.target.value }))} placeholder="For better experience" className="rounded-xl" />
                          </div>
                          <div className="space-y-1.5">
                            <Label className="text-xs">ইনস্টল লিংক (URL)</Label>
                            <Input value={siteFields.app_install_link} onChange={e => setSiteFields(f => ({ ...f, app_install_link: e.target.value }))} placeholder="https://play.google.com/..." className="rounded-xl" />
                          </div>
                          <div className="space-y-1.5">
                            <Label className="text-xs">অ্যাপ আইকন</Label>
                            <input type="file" accept="image/*" ref={appIconRef} className="hidden" onChange={async (e) => {
                              const file = e.target.files?.[0];
                              if (!file) return;
                              setUploadingAppIcon(true);
                              try {
                                const fileName = `app-icon-${Date.now()}.${file.name.split('.').pop()}`;
                                const { error } = await supabase.storage.from("logos").upload(fileName, file, { upsert: true });
                                if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return; }
                                const { data: urlData } = supabase.storage.from("logos").getPublicUrl(fileName);
                                setSiteFields(f => ({ ...f, app_install_image: urlData.publicUrl }));
                                toast({ title: "✅ আইকন আপলোড হয়েছে!" });
                              } finally { setUploadingAppIcon(false); }
                            }} />
                            <div className="flex items-center gap-3">
                              <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingAppIcon} onClick={() => appIconRef.current?.click()}>
                                {uploadingAppIcon ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                                {uploadingAppIcon ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড"}
                              </Button>
                              <span className="text-xs text-muted-foreground">অথবা</span>
                              <Input value={siteFields.app_install_image} onChange={e => setSiteFields(f => ({ ...f, app_install_image: e.target.value }))} placeholder="ইমেজ URL পেস্ট করুন" className="rounded-xl flex-1" />
                            </div>
                          </div>
                        </div>
                        {siteFields.app_install_image && (
                          <div className="flex items-center gap-3 p-3 bg-muted/30 rounded-xl border border-border">
                            <img src={siteFields.app_install_image} alt="App Icon Preview" className="h-10 w-10 rounded-xl object-cover border border-border" />
                            <span className="text-xs text-muted-foreground">আইকন প্রিভিউ</span>
                            <Button variant="ghost" size="sm" className="ml-auto text-xs text-destructive" onClick={() => setSiteFields(f => ({ ...f, app_install_image: "" }))}>রিমুভ</Button>
                          </div>
                        )}
                        <p className="text-[11px] text-muted-foreground">সেটিংস সেভ করলে ওয়েবসাইটের একদম উপরে এই ব্যানার দেখাবে। ইউজার ✕ দিয়ে বন্ধ করতে পারবে।</p>
                      </div>
                    </SectionCard>

                    {/* PWA Icon */}
                    <SectionCard title="PWA আইকন (হোম স্ক্রীন)" description="অ্যাপ ইনস্টল করলে হোম স্ক্রীনে যে আইকন দেখাবে" icon={Globe}>
                      <div className="space-y-4">
                        <input type="file" accept="image/png,image/webp" ref={pwaIconRef} className="hidden" onChange={async (e) => {
                          const file = e.target.files?.[0];
                          if (!file) return;
                          setUploadingPwaIcon(true);
                          try {
                            const fileName = `pwa-icon-${Date.now()}.${file.name.split('.').pop()}`;
                            const { error } = await supabase.storage.from("logos").upload(fileName, file, { upsert: true });
                            if (error) { toast({ title: "আপলোড এরর", description: error.message, variant: "destructive" }); return; }
                            const { data: urlData } = supabase.storage.from("logos").getPublicUrl(fileName);
                            setSiteFields(f => ({ ...f, pwa_icon_url: urlData.publicUrl }));
                            toast({ title: "✅ PWA আইকন আপলোড হয়েছে!" });
                          } finally { setUploadingPwaIcon(false); }
                        }} />
                        <div className="flex items-center gap-3">
                          <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingPwaIcon} onClick={() => pwaIconRef.current?.click()}>
                            {uploadingPwaIcon ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                            {uploadingPwaIcon ? "আপলোড হচ্ছে..." : "আইকন আপলোড"}
                          </Button>
                          <span className="text-xs text-muted-foreground">অথবা</span>
                          <Input value={siteFields.pwa_icon_url} onChange={e => setSiteFields(f => ({ ...f, pwa_icon_url: e.target.value }))} placeholder="আইকন URL পেস্ট করুন" className="rounded-xl flex-1" />
                        </div>
                        {siteFields.pwa_icon_url && (
                          <div className="flex items-center gap-3 p-3 bg-muted/30 rounded-xl border border-border">
                            <img src={siteFields.pwa_icon_url} alt="PWA Icon Preview" className="h-14 w-14 rounded-2xl object-cover border border-border" />
                            <div>
                              <span className="text-xs text-muted-foreground block">PWA আইকন প্রিভিউ</span>
                              <span className="text-[10px] text-muted-foreground">PNG বা WebP, কমপক্ষে 512×512px রিকমেন্ডেড</span>
                            </div>
                            <Button variant="ghost" size="sm" className="ml-auto text-xs text-destructive" onClick={() => setSiteFields(f => ({ ...f, pwa_icon_url: "" }))}>রিমুভ</Button>
                          </div>
                        )}
                        <p className="text-[11px] text-muted-foreground">এই আইকনটি ইউজাররা অ্যাপ ইনস্টল করলে হোম স্ক্রীনে দেখাবে। সেভ করলে /install পেজেও আপডেট হবে।</p>
                      </div>
                    </SectionCard>

                    <SectionCard title="পেমেন্ট ব্যানার" description="ফুটার ও প্রোডাক্ট পেজে দেখানো পেমেন্ট মেথড ইমেজ" icon={CreditCard}>
                      <div className="space-y-4">
                        <div className="space-y-1.5">
                          <Label className="text-xs">পেমেন্ট ব্যানার ইমেজ</Label>
                          <input type="file" accept="image/*" ref={paymentBannerRef} onChange={handlePaymentBannerUpload} className="hidden" />
                          <div className="flex items-center gap-3">
                            <Button type="button" variant="outline" size="sm" className="gap-1.5 rounded-xl" disabled={uploadingPaymentBanner} onClick={() => paymentBannerRef.current?.click()}>
                              {uploadingPaymentBanner ? <Loader2 className="h-4 w-4 animate-spin" /> : <Upload className="h-4 w-4" />}
                              {uploadingPaymentBanner ? "আপলোড হচ্ছে..." : "ইমেজ আপলোড"}
                            </Button>
                            <span className="text-xs text-muted-foreground">অথবা</span>
                            <Input value={paymentBannerUrl || ""} onChange={e => setPaymentBannerUrl(e.target.value)} placeholder="ইমেজ URL পেস্ট করুন" className="rounded-xl flex-1" />
                          </div>
                          {paymentBannerUrl && (
                            <div className="mt-2 p-2 bg-muted/30 rounded-xl border border-border">
                              <img src={paymentBannerUrl} alt="Payment Banner Preview" className="w-full h-auto max-h-20 object-contain rounded" />
                              <Button variant="ghost" size="sm" className="mt-1 text-xs text-destructive" onClick={async () => {
                                await supabase.from("site_settings").delete().eq("key", "payment_banner_url");
                                setPaymentBannerUrl(null);
                                toast({ title: "🗑️ পেমেন্ট ব্যানার রিমুভ হয়েছে!" });
                              }}>রিমুভ করুন</Button>
                            </div>
                          )}
                          <p className="text-[11px] text-muted-foreground">এই ইমেজটি ফুটার এবং প্রোডাক্ট পেজে পেমেন্ট মেথড হিসেবে দেখাবে</p>
                        </div>

                        {/* Trust Badge */}
                        <div className="space-y-2">
                          <Label className="text-xs font-semibold">🛡️ Trust Badge (প্রোডাক্ট পেজ)</Label>
                          <Input value={trustBadgeUrl || ""} onChange={e => setTrustBadgeUrl(e.target.value)} placeholder="Trust badge ইমেজ URL পেস্ট করুন" className="rounded-xl" />
                          {trustBadgeUrl && (
                            <div className="mt-2 p-2 bg-muted/30 rounded-xl border border-border">
                              <img src={trustBadgeUrl} alt="Trust Badge Preview" className="w-full h-auto max-h-20 object-contain rounded" />
                              <Button variant="ghost" size="sm" className="mt-1 text-xs text-destructive" onClick={async () => {
                                await supabase.from("site_settings").delete().eq("key", "trust_badge_url");
                                setTrustBadgeUrl(null);
                                toast({ title: "🗑️ Trust Badge রিমুভ হয়েছে!" });
                              }}>রিমুভ করুন</Button>
                            </div>
                          )}
                          <p className="text-[11px] text-muted-foreground">এই ব্যাজটি প্রোডাক্ট ডিটেইল পেজে Buy Now বাটনের নিচে দেখাবে</p>
                        </div>
                      </div>
                    </SectionCard>

                    {/* Payment */}
                    <SectionCard title="পেমেন্ট সেটিংস" description="TZSMMPay পেমেন্ট গেটওয়ে কনফিগারেশন" icon={CreditCard}>
                      <div className="space-y-4">
                        <div className="space-y-1.5">
                          <Label className="text-xs">TZSMMPay API Key</Label>
                          <Input type="password" value={siteFields.tzsmmpay_api_key} onChange={e => setSiteFields(f => ({ ...f, tzsmmpay_api_key: e.target.value }))} placeholder="আপনার TZSMMPay API key দিন" className="rounded-xl" />
                          <p className="text-[11px] text-muted-foreground">API key পরিবর্তন করলে সেভ করতে ভুলবেন না।</p>
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs">Payment Gateway URL</Label>
                          <Input value={siteFields.payment_gateway_url} onChange={e => setSiteFields(f => ({ ...f, payment_gateway_url: e.target.value }))} placeholder="pay.tzssbd.com" className="rounded-xl" />
                          <p className="text-[11px] text-muted-foreground">https:// ছাড়া শুধু ডোমেইন দিন</p>
                        </div>
                      </div>
                    </SectionCard>

                    {/* Telegram */}
                    <SectionCard title="টেলিগ্রাম নোটিফিকেশন" description="অর্ডার আসলে টেলিগ্রামে নোটিফিকেশন পাঠান" icon={MessageCircle}>
                      <div className="space-y-4">
                        <div className="space-y-1.5">
                          <Label className="text-xs">Telegram Bot Token</Label>
                          <Input type="password" value={siteFields.telegram_bot_token} onChange={e => setSiteFields(f => ({ ...f, telegram_bot_token: e.target.value }))} placeholder="123456:ABC-DEF..." className="rounded-xl" />
                          <p className="text-[11px] text-muted-foreground">BotFather থেকে পাওয়া Bot Token দিন</p>
                        </div>
                        <div className="space-y-1.5">
                          <Label className="text-xs">Chat ID</Label>
                          <Input value={siteFields.telegram_chat_id} onChange={e => setSiteFields(f => ({ ...f, telegram_chat_id: e.target.value }))} placeholder="-1001234567890" className="rounded-xl" />
                          <p className="text-[11px] text-muted-foreground">যে চ্যাট বা গ্রুপে নোটিফিকেশন পাবেন সেটার ID দিন</p>
                        </div>
                      </div>
                    </SectionCard>

                    {/* Developer Credit - Password Protected */}
                    <SectionCard title="ডেভেলপার ক্রেডিট 🔒" description="ফুটারে দেখানো ডেভেলপারের তথ্য (পাসওয়ার্ড প্রয়োজন)" icon={ExternalLink}>
                      {!devCreditUnlocked ? (
                        <div className="space-y-3">
                          <p className="text-sm text-muted-foreground">এই সেকশন এডিট করতে পাসওয়ার্ড দিন:</p>
                          <div className="flex gap-2 items-end">
                            <div className="flex-1 space-y-1.5">
                              <Label className="text-xs">পাসওয়ার্ড</Label>
                              <Input type="password" value={devCreditPassword} onChange={e => setDevCreditPassword(e.target.value)} placeholder="পাসওয়ার্ড দিন" className="rounded-xl"
                                onKeyDown={e => { if (e.key === "Enter") { if (storedDevPassword && devCreditPassword === storedDevPassword) { setDevCreditUnlocked(true); setDevCreditPassword(""); } else { toast({ title: "❌ পাসওয়ার্ড ভুল!", variant: "destructive" }); setDevCreditPassword(""); } } }} />
                            </div>
                            <Button size="sm" className="rounded-xl gap-1.5" onClick={() => {
                              if (storedDevPassword && devCreditPassword === storedDevPassword) { setDevCreditUnlocked(true); setDevCreditPassword(""); }
                              else { toast({ title: "❌ পাসওয়ার্ড ভুল!", variant: "destructive" }); setDevCreditPassword(""); }
                            }}>
                              <Shield className="h-4 w-4" />আনলক
                            </Button>
                          </div>
                        </div>
                      ) : (
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div className="space-y-1.5">
                            <Label className="text-xs flex items-center gap-1.5"><Globe className="h-3 w-3" />ডেভেলপার নাম</Label>
                            <Input value={siteFields.developer_name} onChange={e => setSiteFields(f => ({ ...f, developer_name: e.target.value }))} placeholder="TEAM RONY" className="rounded-xl" />
                          </div>
                          <div className="space-y-1.5">
                            <Label className="text-xs flex items-center gap-1.5"><ExternalLink className="h-3 w-3" />ডেভেলপার URL</Label>
                            <Input value={siteFields.developer_url} onChange={e => setSiteFields(f => ({ ...f, developer_url: e.target.value }))} placeholder="https://hanifahmedrony.site" className="rounded-xl" />
                          </div>
                        </div>
                      )}
                    </SectionCard>

                    <div className="sticky bottom-4 z-10">
                      <Button onClick={handleSaveSettings} disabled={savingSettings} className="gap-1.5 rounded-xl shadow-lg shadow-primary/20 w-full sm:w-auto">
                        {savingSettings ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
                        {savingSettings ? "সেভ হচ্ছে..." : "সেটিংস সেভ করুন"}
                      </Button>
                    </div>
                  </div>
                )}

              </motion.div>
            </AnimatePresence>
          )}
        </div>
      </main>
    </div>
  );
};

export default Admin;
