import { useState, useEffect } from "react";
import { useNavigate, Link, useLocation } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { lovable } from "@/integrations/lovable/index";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { toast } from "@/hooks/use-toast";
import { Mail, Lock, User, ArrowLeft, Eye, EyeOff, ShieldCheck, Sparkles, Loader2 } from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";

const Auth = () => {
  const [isLogin, setIsLogin] = useState(true);
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [name, setName] = useState("");
  const [phone, setPhone] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [loading, setLoading] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const navigate = useNavigate();
  const location = useLocation();
  const redirectTo = (location.state as any)?.from || "/";

  // Auth settings from admin
  const [authSettings, setAuthSettings] = useState({
    emailLoginEnabled: true,
    googleLoginEnabled: true,
    registrationEnabled: true,
    phoneRequired: true,
    minPasswordLength: 6,
    sessionNote: "",
    maintenanceMode: false,
    maintenanceMessage: "",
  });
  const [settingsLoaded, setSettingsLoaded] = useState(false);

  useEffect(() => {
    const load = async () => {
      const { data } = await supabase.from("site_settings").select("key, value");
      if (data) {
        const map: Record<string, string> = {};
        data.forEach(r => { if (r.value) map[r.key] = r.value; });
        setAuthSettings({
          emailLoginEnabled: map.auth_email_login !== "false",
          googleLoginEnabled: map.auth_google_login !== "false",
          registrationEnabled: map.auth_registration_enabled !== "false",
          phoneRequired: map.auth_phone_required !== "false",
          minPasswordLength: parseInt(map.auth_min_password_length || "6"),
          sessionNote: map.auth_session_note || "",
          maintenanceMode: map.auth_maintenance_mode === "true",
          maintenanceMessage: map.auth_maintenance_message || "সাইটটি রক্ষণাবেক্ষণের জন্য সাময়িকভাবে বন্ধ আছে।",
        });
      }
      setSettingsLoaded(true);
    };
    load();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      if (!isLogin && !authSettings.registrationEnabled) {
        toast({ title: "রেজিস্ট্রেশন বন্ধ আছে", description: "এই মুহূর্তে নতুন অ্যাকাউন্ট তৈরি করা যাচ্ছে না।", variant: "destructive" });
        setLoading(false);
        return;
      }
      if (!isLogin && password !== confirmPassword) {
        toast({ title: "পাসওয়ার্ড মিলছে না!", description: "পাসওয়ার্ড এবং কনফার্ম পাসওয়ার্ড একই হতে হবে।", variant: "destructive" });
        setLoading(false);
        return;
      }
      if (isLogin) {
        const { error } = await supabase.auth.signInWithPassword({ email, password });
        if (error) throw error;
        toast({ title: "সফলভাবে লগইন হয়েছে!" });
        navigate(redirectTo);
      } else {
        const { error } = await supabase.auth.signUp({
          email,
          password,
          options: {
            emailRedirectTo: window.location.origin,
            data: { name, phone },
          },
        });
        if (error) throw error;
        toast({
          title: "অ্যাকাউন্ট তৈরি হয়েছে!",
          description: "আপনার ইমেইলে একটি ভেরিফিকেশন লিংক পাঠানো হয়েছে। প্লিজ ইমেইল ভেরিফাই করুন।",
        });
      }
    } catch (error: any) {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    } finally {
      setLoading(false);
    }
  };

  const inputClass = "pl-10 pr-10 h-12 rounded-lg border-border bg-background focus:border-primary transition-colors";
  const iconClass = "absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-foreground/50 z-10";

  return (
    <div className="min-h-screen relative flex items-center justify-center px-4 py-12 overflow-hidden">
      {/* Gradient background */}
      <div className="absolute inset-0 bg-gradient-to-br from-primary/10 via-background to-primary/5" />
      <div className="absolute top-0 right-0 w-96 h-96 bg-primary/8 rounded-full blur-3xl -translate-y-1/2 translate-x-1/3" />
      <div className="absolute bottom-0 left-0 w-80 h-80 bg-primary/6 rounded-full blur-3xl translate-y-1/3 -translate-x-1/4" />

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
        className="w-full max-w-md relative z-10"
      >
        <button
          onClick={() => navigate("/")}
          className="flex items-center gap-1.5 text-sm text-muted-foreground hover:text-foreground mb-6 transition-colors group"
        >
          <ArrowLeft className="h-4 w-4 group-hover:-translate-x-0.5 transition-transform" /> Back to Home
        </button>

        <motion.div
          layout
          className="bg-card/90 backdrop-blur-md border border-border/50 rounded-2xl p-8 relative overflow-hidden"
        >
          {/* Decorative top accent */}
          <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-primary/60 via-primary to-primary/60" />

          <div className="text-center mb-8">
            <motion.div
              key={isLogin ? "login-icon" : "signup-icon"}
              initial={{ scale: 0.5, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              transition={{ type: "spring", stiffness: 300, damping: 20 }}
              className="w-14 h-14 rounded-full bg-primary/10 flex items-center justify-center mx-auto mb-4"
            >
              {isLogin ? (
                <ShieldCheck className="h-7 w-7 text-primary" />
              ) : (
                <Sparkles className="h-7 w-7 text-primary" />
              )}
            </motion.div>
            <AnimatePresence mode="wait">
              <motion.div
                key={isLogin ? "login" : "signup"}
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                exit={{ opacity: 0, y: -10 }}
                transition={{ duration: 0.2 }}
              >
                <h1 className="font-heading text-2xl font-bold text-foreground">
                  {isLogin ? "Welcome Back" : "Create Account"}
                </h1>
                <p className="text-sm text-muted-foreground mt-1">
                  {isLogin
                    ? "আপনার অ্যাকাউন্টে লগইন করুন"
                    : "নতুন অ্যাকাউন্ট তৈরি করুন"}
                </p>
              </motion.div>
            </AnimatePresence>
          </div>

          <form onSubmit={handleSubmit} className="space-y-4">
            <AnimatePresence mode="wait">
              {!isLogin && (
                <motion.div
                  initial={{ height: 0, opacity: 0 }}
                  animate={{ height: "auto", opacity: 1 }}
                  exit={{ height: 0, opacity: 0 }}
                  transition={{ duration: 0.3 }}
                  className="space-y-4 overflow-hidden"
                >
                  <div className="relative">
                    <User className={iconClass} />
                    <Input placeholder="Your Name" value={name} onChange={(e) => setName(e.target.value)} className={inputClass} required />
                  </div>
                  <div className="relative">
                    <span className="absolute left-3 top-1/2 -translate-y-1/2 text-foreground/50 text-sm z-10">📞</span>
                    <Input placeholder="Phone Number" value={phone} onChange={(e) => setPhone(e.target.value)} className={inputClass} required={authSettings.phoneRequired} />
                  </div>
                </motion.div>
              )}
            </AnimatePresence>

            <div className="relative">
              <Mail className={iconClass} />
              <Input type="email" placeholder="Email Address" value={email} onChange={(e) => setEmail(e.target.value)} className={inputClass} required />
            </div>

            <div className="relative">
              <Lock className={iconClass} />
              <Input
                type={showPassword ? "text" : "password"}
                placeholder="Password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                className={inputClass}
                required
                minLength={authSettings.minPasswordLength}
              />
              <button type="button" onClick={() => setShowPassword(!showPassword)} className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground transition-colors">
                {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
              </button>
            </div>

            {isLogin && (
              <div className="text-right">
                <Link to="/reset-password-request" className="text-sm text-primary hover:underline font-medium">
                  Forgot Password?
                </Link>
              </div>
            )}

            <AnimatePresence>
              {!isLogin && (
                <motion.div
                  initial={{ height: 0, opacity: 0 }}
                  animate={{ height: "auto", opacity: 1 }}
                  exit={{ height: 0, opacity: 0 }}
                  transition={{ duration: 0.3 }}
                  className="relative overflow-hidden"
                >
                  <Lock className={iconClass} />
                  <Input
                    type={showConfirmPassword ? "text" : "password"}
                    placeholder="Confirm Password"
                    value={confirmPassword}
                    onChange={(e) => setConfirmPassword(e.target.value)}
                    className={inputClass}
                    required
                    minLength={authSettings.minPasswordLength}
                  />
                  <button type="button" onClick={() => setShowConfirmPassword(!showConfirmPassword)} className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground transition-colors">
                    {showConfirmPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                  </button>
                </motion.div>
              )}
            </AnimatePresence>

            <motion.div whileTap={{ scale: 0.98 }}>
              <Button type="submit" className="w-full h-12 rounded-lg font-bold text-base" disabled={loading}>
                {loading
                  ? "Please wait..."
                  : isLogin
                  ? "Login"
                  : "Sign Up"}
              </Button>
            </motion.div>
          </form>

          {/* Session Note */}
          {authSettings.sessionNote && (
            <div className="mt-4 p-3 rounded-xl bg-primary/5 border border-primary/20 text-center">
              <p className="text-xs text-primary font-medium">{authSettings.sessionNote}</p>
            </div>
          )}

          {/* Google Login */}
          {authSettings.googleLoginEnabled && (
            <>
              <div className="flex items-center gap-3 my-6">
                <div className="flex-1 h-px bg-border" />
                <span className="text-xs text-muted-foreground">or</span>
                <div className="flex-1 h-px bg-border" />
              </div>
              <motion.div whileTap={{ scale: 0.98 }}>
                <Button
                  type="button"
                  variant="outline"
                  className="w-full h-12 rounded-lg font-medium text-base gap-3"
                  onClick={async () => {
                    const { error } = await lovable.auth.signInWithOAuth("google", {
                      redirect_uri: window.location.origin,
                    });
                    if (error) {
                      toast({ title: "Google লগইন ব্যর্থ", description: String(error), variant: "destructive" });
                    }
                  }}
                >
                  <svg className="h-5 w-5" viewBox="0 0 24 24">
                    <path d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92a5.06 5.06 0 0 1-2.2 3.32v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.1z" fill="#4285F4"/>
                    <path d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z" fill="#34A853"/>
                    <path d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z" fill="#FBBC05"/>
                    <path d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z" fill="#EA4335"/>
                  </svg>
                  Sign in with Google
                </Button>
              </motion.div>
            </>
          )}

          <div className="mt-6 text-center">
            {(isLogin && authSettings.registrationEnabled) || !isLogin ? (
              <p className="text-sm text-muted-foreground">
                {isLogin ? "Don't have an account?" : "Already have an account?"}{" "}
                <button
                  onClick={() => setIsLogin(!isLogin)}
                  className="text-primary font-semibold hover:underline"
                >
                  {isLogin ? "Sign Up" : "Login"}
                </button>
              </p>
            ) : null}
          </div>
        </motion.div>
      </motion.div>
    </div>
  );
};

export default Auth;
