import { useState, useEffect } from "react";
import { useParams, useNavigate } from "react-router-dom";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import { supabase } from "@/integrations/supabase/client";
import { Calendar, ArrowLeft } from "lucide-react";
import { Button } from "@/components/ui/button";

const BlogPost = () => {
  const { slug } = useParams();
  const navigate = useNavigate();
  const [post, setPost] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchPost = async () => {
      const { data } = await supabase
        .from("blog_posts")
        .select("*")
        .eq("slug", slug)
        .eq("is_published", true)
        .single();
      setPost(data);
      setLoading(false);
    };
    if (slug) fetchPost();
  }, [slug]);

  if (loading) {
    return (
      <div className="min-h-screen bg-background">
        <Header />
        <div className="container mx-auto px-4 py-12 max-w-3xl animate-pulse space-y-4">
          <div className="h-8 bg-muted rounded w-3/4" />
          <div className="h-64 bg-muted rounded-xl" />
          <div className="space-y-2">
            <div className="h-4 bg-muted rounded w-full" />
            <div className="h-4 bg-muted rounded w-5/6" />
            <div className="h-4 bg-muted rounded w-4/6" />
          </div>
        </div>
      </div>
    );
  }

  if (!post) {
    return (
      <div className="min-h-screen bg-background">
        <Header />
        <div className="container mx-auto px-4 py-16 text-center">
          <h1 className="text-2xl font-bold text-foreground mb-4">পোস্ট পাওয়া যায়নি</h1>
          <Button onClick={() => navigate("/blog")}>ব্লগে ফিরে যান</Button>
        </div>
        <ShopFooter />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      <Header />
      <article className="container mx-auto px-4 py-12 max-w-3xl">
        <Button variant="ghost" className="gap-1.5 mb-6" onClick={() => navigate("/blog")}>
          <ArrowLeft className="h-4 w-4" /> ব্লগে ফিরে যান
        </Button>
        <h1 className="text-3xl font-bold text-foreground mb-4">{post.title}</h1>
        <div className="flex items-center gap-2 text-sm text-muted-foreground mb-6">
          <Calendar className="h-4 w-4" />
          <span>{new Date(post.created_at).toLocaleDateString("bn-BD")}</span>
        </div>
        {post.image_url && (
          <img src={post.image_url} alt={post.title} className="w-full rounded-xl mb-8 max-h-96 object-cover" />
        )}
        <div className="prose prose-sm max-w-none text-foreground whitespace-pre-wrap leading-relaxed">
          {post.content}
        </div>
      </article>
      <ShopFooter />
    </div>
  );
};

export default BlogPost;
