import { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { toast } from "@/hooks/use-toast";
import { Mail, ArrowLeft, KeyRound, CheckCircle2, Phone, Lock, Eye, EyeOff, ShieldCheck } from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";

const ForgotPassword = () => {
  const [method, setMethod] = useState<"email" | "phone">("email");
  const [email, setEmail] = useState("");
  const [phone, setPhone] = useState("");
  const [otp, setOtp] = useState("");
  const [newPassword, setNewPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [loading, setLoading] = useState(false);
  const [emailSent, setEmailSent] = useState(false);
  const [otpSent, setOtpSent] = useState(false);
  const [otpVerified, setOtpVerified] = useState(false);
  const [cooldown, setCooldown] = useState(0);
  const cooldownRef = useRef<NodeJS.Timeout | null>(null);
  const navigate = useNavigate();

  useEffect(() => {
    if (cooldown > 0) {
      cooldownRef.current = setTimeout(() => setCooldown(c => c - 1), 1000);
    }
    return () => { if (cooldownRef.current) clearTimeout(cooldownRef.current); };
  }, [cooldown]);

  const inputClass = "pl-10 pr-10 h-12 rounded-lg border-border bg-background focus:border-primary transition-colors";
  const iconClass = "absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-foreground/50 z-10";

  const handleEmailReset = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      const { data, error } = await supabase.functions.invoke("send-reset-email", {
        body: { 
          email, 
          reset_url: `${window.location.origin}/reset-password` 
        },
      });
      if (error || !data?.success) {
        throw new Error(data?.error || error?.message || "ইমেইল পাঠাতে সমস্যা হয়েছে");
      }
      setEmailSent(true);
      toast({ title: "ইমেইল পাঠানো হয়েছে!", description: "আপনার ইমেইলে পাসওয়ার্ড রিসেট লিংক পাঠানো হয়েছে।" });
    } catch (error: any) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } finally {
      setLoading(false);
    }
  };

  const handleSendOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!phone) {
      toast({ title: "ফোন নম্বর দিন", variant: "destructive" });
      return;
    }
    if (cooldown > 0) {
      toast({ title: `${cooldown} সেকেন্ড পর আবার চেষ্টা করুন`, variant: "destructive" });
      return;
    }
    setLoading(true);
    try {
      const response = await supabase.functions.invoke("send-otp", {
        body: { phone },
      });
      const data = response.data;
      const error = response.error;
      if (error || !data?.success) {
        throw new Error(data?.error || error?.message || "OTP পাঠাতে সমস্যা হয়েছে");
      }
      setOtpSent(true);
      setCooldown(60);
      toast({ title: "✅ OTP পাঠানো হয়েছে!", description: "আপনার ফোনে ৬ সংখ্যার OTP পাঠানো হয়েছে।" });
    } catch (error: any) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } finally {
      setLoading(false);
    }
  };

  const handleVerifyAndReset = async (e: React.FormEvent) => {
    e.preventDefault();
    if (newPassword !== confirmPassword) {
      toast({ title: "পাসওয়ার্ড মিলছে না!", variant: "destructive" });
      return;
    }
    setLoading(true);
    try {
      const { data, error } = await supabase.functions.invoke("verify-otp-reset", {
        body: { phone, otp, new_password: newPassword },
      });
      if (error || !data?.success) {
        throw new Error(data?.error || error?.message || "ভেরিফাই করতে সমস্যা হয়েছে");
      }
      setOtpVerified(true);
      toast({ title: "✅ পাসওয়ার্ড সফলভাবে পরিবর্তন হয়েছে!" });
    } catch (error: any) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } finally {
      setLoading(false);
    }
  };

  const getIcon = () => {
    if (emailSent || otpVerified) return <CheckCircle2 className="h-7 w-7 text-primary" />;
    if (otpSent) return <ShieldCheck className="h-7 w-7 text-primary" />;
    return <KeyRound className="h-7 w-7 text-primary" />;
  };

  const getTitle = () => {
    if (otpVerified) return "পাসওয়ার্ড পরিবর্তন হয়েছে!";
    if (otpSent) return "OTP ভেরিফাই করুন";
    if (emailSent) return "ইমেইল পাঠানো হয়েছে";
    return "পাসওয়ার্ড রিসেট";
  };

  const getSubtitle = () => {
    if (otpVerified) return "আপনি এখন নতুন পাসওয়ার্ড দিয়ে লগইন করতে পারবেন";
    if (otpSent) return "আপনার ফোনে পাঠানো OTP ও নতুন পাসওয়ার্ড দিন";
    if (emailSent) return "আপনার ইমেইলে রিসেট লিংক পাঠানো হয়েছে";
    return method === "email" ? "আপনার ইমেইল দিন, রিসেট লিংক পাঠানো হবে" : "আপনার ফোন নম্বর দিন, OTP পাঠানো হবে";
  };

  return (
    <div className="min-h-screen relative flex items-center justify-center px-4 py-12 overflow-hidden">
      <div className="absolute inset-0 bg-gradient-to-br from-primary/10 via-background to-primary/5" />
      <div className="absolute top-0 right-0 w-96 h-96 bg-primary/8 rounded-full blur-3xl -translate-y-1/2 translate-x-1/3" />
      <div className="absolute bottom-0 left-0 w-80 h-80 bg-primary/6 rounded-full blur-3xl translate-y-1/3 -translate-x-1/4" />

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
        className="w-full max-w-md relative z-10"
      >
        <button
          onClick={() => navigate("/auth")}
          className="flex items-center gap-1.5 text-sm text-muted-foreground hover:text-foreground mb-6 transition-colors group"
        >
          <ArrowLeft className="h-4 w-4 group-hover:-translate-x-0.5 transition-transform" /> লগইনে ফিরে যান
        </button>

        <div className="bg-card/90 backdrop-blur-md border border-border/50 rounded-2xl p-8 relative overflow-hidden">
          <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-primary/60 via-primary to-primary/60" />

          <div className="text-center mb-8">
            <motion.div
              key={otpVerified ? "done" : otpSent ? "otp" : "start"}
              initial={{ scale: 0.5, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              transition={{ type: "spring", stiffness: 300, damping: 20 }}
              className="w-14 h-14 rounded-full bg-primary/10 flex items-center justify-center mx-auto mb-4"
            >
              {getIcon()}
            </motion.div>
            <AnimatePresence mode="wait">
              <motion.div
                key={getTitle()}
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                exit={{ opacity: 0, y: -10 }}
              >
                <h1 className="font-heading text-2xl font-bold text-foreground">{getTitle()}</h1>
                <p className="text-sm text-muted-foreground mt-1">{getSubtitle()}</p>
              </motion.div>
            </AnimatePresence>
          </div>

          {/* Method selector - only show initially */}
          {!emailSent && !otpSent && !otpVerified && (
            <div className="flex gap-2 mb-6">
              <button onClick={() => setMethod("email")}
                className={`flex-1 flex items-center justify-center gap-2 px-4 py-2.5 rounded-xl text-sm font-medium transition-all ${method === "email" ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground hover:text-foreground"}`}>
                <Mail className="h-4 w-4" /> ইমেইল
              </button>
              <button onClick={() => setMethod("phone")}
                className={`flex-1 flex items-center justify-center gap-2 px-4 py-2.5 rounded-xl text-sm font-medium transition-all ${method === "phone" ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground hover:text-foreground"}`}>
                <Phone className="h-4 w-4" /> ফোন (OTP)
              </button>
            </div>
          )}

          <AnimatePresence mode="wait">
            {/* Success state */}
            {(emailSent || otpVerified) && (
              <motion.div
                key="success"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                className="text-center space-y-4"
              >
                <p className="text-sm text-muted-foreground">
                  {emailSent
                    ? "আপনার ইমেইলে একটি পাসওয়ার্ড রিসেট লিংক পাঠানো হয়েছে। প্লিজ আপনার ইমেইল চেক করুন।"
                    : "আপনার পাসওয়ার্ড সফলভাবে পরিবর্তন হয়েছে। এখন লগইন করুন।"}
                </p>
                <motion.div whileTap={{ scale: 0.98 }}>
                  <Button variant="outline" onClick={() => navigate("/auth")} className="w-full h-12 rounded-lg font-bold">
                    লগইনে ফিরে যান
                  </Button>
                </motion.div>
              </motion.div>
            )}

            {/* Email form */}
            {!emailSent && !otpSent && !otpVerified && method === "email" && (
              <motion.form key="email-form" onSubmit={handleEmailReset} className="space-y-4"
                initial={{ opacity: 0, x: -20 }} animate={{ opacity: 1, x: 0 }} exit={{ opacity: 0, x: 20 }}>
                <div className="relative">
                  <Mail className={iconClass} />
                  <Input type="email" placeholder="ইমেইল" value={email} onChange={(e) => setEmail(e.target.value)} className={inputClass} required />
                </div>
                <motion.div whileTap={{ scale: 0.98 }}>
                  <Button type="submit" className="w-full h-12 rounded-lg font-bold text-base" disabled={loading}>
                    {loading ? "অপেক্ষা করুন..." : "রিসেট লিংক পাঠান"}
                  </Button>
                </motion.div>
              </motion.form>
            )}

            {/* Phone OTP request form */}
            {!emailSent && !otpSent && !otpVerified && method === "phone" && (
              <motion.form key="phone-form" onSubmit={handleSendOtp} className="space-y-4"
                initial={{ opacity: 0, x: 20 }} animate={{ opacity: 1, x: 0 }} exit={{ opacity: 0, x: -20 }}>
                <div className="relative">
                  <Phone className={iconClass} />
                  <Input type="tel" placeholder="01XXXXXXXXX" value={phone} onChange={(e) => setPhone(e.target.value)} className={inputClass} required />
                </div>
                <motion.div whileTap={{ scale: 0.98 }}>
                  <Button type="submit" className="w-full h-12 rounded-lg font-bold text-base" disabled={loading}>
                    {loading ? "অপেক্ষা করুন..." : "OTP পাঠান"}
                  </Button>
                </motion.div>
              </motion.form>
            )}

            {/* OTP verify + new password form */}
            {otpSent && !otpVerified && (
              <motion.form key="otp-form" onSubmit={handleVerifyAndReset} className="space-y-4"
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                <div className="relative">
                  <ShieldCheck className={iconClass} />
                  <Input
                    type="text"
                    placeholder="৬ সংখ্যার OTP"
                    value={otp}
                    onChange={(e) => setOtp(e.target.value.replace(/\D/g, '').slice(0, 6))}
                    className={`${inputClass} text-center text-lg tracking-[0.5em] font-bold`}
                    required
                    maxLength={6}
                  />
                </div>
                <div className="relative">
                  <Lock className={iconClass} />
                  <Input
                    type={showPassword ? "text" : "password"}
                    placeholder="নতুন পাসওয়ার্ড"
                    value={newPassword}
                    onChange={(e) => setNewPassword(e.target.value)}
                    className={inputClass}
                    required
                    minLength={6}
                  />
                  <button type="button" onClick={() => setShowPassword(!showPassword)} className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground transition-colors">
                    {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                  </button>
                </div>
                <div className="relative">
                  <Lock className={iconClass} />
                  <Input
                    type="password"
                    placeholder="কনফার্ম পাসওয়ার্ড"
                    value={confirmPassword}
                    onChange={(e) => setConfirmPassword(e.target.value)}
                    className={inputClass}
                    required
                    minLength={6}
                  />
                </div>
                <motion.div whileTap={{ scale: 0.98 }}>
                  <Button type="submit" className="w-full h-12 rounded-lg font-bold text-base" disabled={loading}>
                    {loading ? "ভেরিফাই হচ্ছে..." : "পাসওয়ার্ড পরিবর্তন করুন"}
                  </Button>
                </motion.div>

                <div className="text-center">
                  <button type="button" 
                    onClick={() => { if (cooldown <= 0) { setOtpSent(false); setOtp(""); } }}
                    disabled={cooldown > 0}
                    className={`text-sm font-medium ${cooldown > 0 ? 'text-muted-foreground cursor-not-allowed' : 'text-primary hover:underline'}`}>
                    {cooldown > 0 ? `আবার পাঠাতে ${cooldown}s অপেক্ষা করুন` : 'আবার OTP পাঠান'}
                  </button>
                </div>
              </motion.form>
            )}
          </AnimatePresence>
        </div>
      </motion.div>
    </div>
  );
};

export default ForgotPassword;
