import { useEffect, useState } from "react";
import { trackPixelEvent } from "@/hooks/usePixel";
import { useSearchParams, useNavigate, useParams } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import TopBar from "@/components/TopBar";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import {
  CheckCircle2, XCircle, Loader2, ArrowLeft, Package, Copy, Check,
  User, CalendarDays, Phone, Mail, CreditCard, Hash, ShoppingBag,
  Clock, Receipt, KeyRound, HelpCircle, MessageCircle
} from "lucide-react";
import { Button } from "@/components/ui/button";
import BottomBar from "@/components/BottomBar";
import { motion } from "framer-motion";

interface OrderDetail {
  id: string;
  product_name: string;
  amount: string;
  status: string;
  payment_status: string | null;
  payment_method: string | null;
  transaction_id: string | null;
  created_at: string;
  customer_name: string;
  customer_phone: string | null;
}

interface DeliveredKey {
  product_name: string;
  key_value: string;
}

interface PaymentCallbackProps {
  forceStatus?: "success" | "cancel";
}

const PaymentCallback = ({ forceStatus }: PaymentCallbackProps) => {
  const [searchParams] = useSearchParams();
  const { orderIds } = useParams<{ orderIds: string }>();
  const navigate = useNavigate();
  const [status, setStatus] = useState<"loading" | "success" | "failed">("loading");
  const [message, setMessage] = useState("");
  const [orders, setOrders] = useState<OrderDetail[]>([]);
  const [deliveredKeys, setDeliveredKeys] = useState<DeliveredKey[]>([]);
  const [copiedKey, setCopiedKey] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState<string | null>(null);

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
    setCopiedKey(text);
    setTimeout(() => setCopiedKey(null), 2000);
  };

  const fetchOrderDetails = async (ids: string[]) => {
    try {
      const { data } = await supabase
        .from("orders")
        .select("id, product_name, amount, status, payment_status, payment_method, transaction_id, created_at, customer_name, customer_phone")
        .in("id", ids);
      if (data) setOrders(data);

      // Get user email from profile
      const { data: { user } } = await supabase.auth.getUser();
      if (user?.email) setUserEmail(user.email);

      const { data: keys } = await supabase
        .from("product_keys")
        .select("key_value, order_id")
        .in("order_id", ids)
        .eq("is_sold", true);

      if (keys && data) {
        const mapped = keys.map(k => {
          const order = data.find(o => o.id === k.order_id);
          return { product_name: order?.product_name || "", key_value: k.key_value };
        });
        setDeliveredKeys(mapped);
      }
    } catch {}
  };

  useEffect(() => {
    const verifyPayment = async () => {
      const transactionId = searchParams.get("transaction_id")
        || searchParams.get("trx_id")
        || searchParams.get("track_id");
      const urlStatus = forceStatus || searchParams.get("status");
      const idsArray = orderIds ? orderIds.split(",").map(id => id.trim()) : [];

      if (urlStatus === "success") {
        supabase.functions.invoke("tzsmmpay-verify", {
          body: {
            transaction_id: transactionId || undefined,
            order_ids: idsArray.length > 0 ? idsArray : undefined,
            force_paid: true,
          },
        }).catch(() => {});

        setStatus("success");
        setMessage("আপনার পেমেন্ট সফল হয়েছে!");
        trackPixelEvent("Purchase", { transaction_id: transactionId || undefined, currency: "BDT" });

        if (idsArray.length > 0) {
          setTimeout(() => fetchOrderDetails(idsArray), 2000);
        }
        return;
      }

      if (urlStatus === "cancel") {
        if (idsArray.length > 0) fetchOrderDetails(idsArray);
        setStatus("failed");
        setMessage("পেমেন্ট বাতিল করা হয়েছে।");
        return;
      }

      if (!transactionId) {
        setStatus("failed");
        setMessage("Transaction ID পাওয়া যায়নি।");
        return;
      }

      try {
        const { data, error } = await supabase.functions.invoke("tzsmmpay-verify", {
          body: { transaction_id: transactionId },
        });

        if (error) {
          setStatus("failed");
          setMessage("পেমেন্ট ভেরিফাই করতে সমস্যা হয়েছে।");
          return;
        }

        if (data.payment_status === "paid") {
          setStatus("success");
          setMessage("আপনার পেমেন্ট সফল হয়েছে!");
          if (idsArray.length > 0) fetchOrderDetails(idsArray);
        } else {
          setStatus("failed");
          setMessage("পেমেন্ট সম্পন্ন হয়নি। অনুগ্রহ করে আবার চেষ্টা করুন।");
        }
      } catch {
        setStatus("failed");
        setMessage("পেমেন্ট ভেরিফাই করতে সমস্যা হয়েছে।");
      }
    };

    verifyPayment();
  }, [searchParams, forceStatus, orderIds]);

  const totalAmount = orders.reduce((sum, o) => {
    const match = o.amount.match(/[\d,]+\.?\d*/);
    return sum + (match ? parseFloat(match[0].replace(/,/g, "")) : 0);
  }, 0);

  const getPaymentMethodLabel = (method: string | null) => {
    if (!method) return "অনলাইন পেমেন্ট";
    const map: Record<string, string> = {
      tzsmmpay: "অনলাইন পেমেন্ট (bKash/Nagad)",
      bkash: "bKash",
      nagad: "Nagad",
      manual: "ম্যানুয়াল",
    };
    return map[method] || method;
  };

  const getStatusBadge = (s: string) => {
    const map: Record<string, { label: string; cls: string }> = {
      paid: { label: "পেইড ✅", cls: "bg-primary/10 text-primary" },
      unpaid: { label: "আনপেইড", cls: "bg-destructive/10 text-destructive" },
      pending: { label: "পেন্ডিং", cls: "bg-yellow-500/10 text-yellow-600" },
      processing: { label: "প্রসেসিং", cls: "bg-blue-500/10 text-blue-600" },
      completed: { label: "সম্পন্ন ✅", cls: "bg-primary/10 text-primary" },
    };
    return map[s] || { label: s, cls: "bg-muted text-muted-foreground" };
  };

  const SectionCard = ({ children, delay = 0 }: { children: React.ReactNode; delay?: number }) => (
    <motion.div
      initial={{ opacity: 0, y: 15 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay, duration: 0.4 }}
      className="bg-card rounded-2xl border border-border overflow-hidden mb-4 text-left"
    >
      {children}
    </motion.div>
  );

  const SectionHeader = ({ icon: Icon, title }: { icon: any; title: string }) => (
    <div className="bg-muted/50 px-5 py-3 border-b border-border flex items-center gap-2">
      <Icon className="h-4 w-4 text-primary" />
      <h2 className="font-heading font-semibold text-sm text-foreground">{title}</h2>
    </div>
  );

  const InfoRow = ({ icon: Icon, label, value }: { icon: any; label: string; value: React.ReactNode }) => (
    <div className="flex items-start gap-3 py-2">
      <Icon className="h-4 w-4 text-muted-foreground mt-0.5 shrink-0" />
      <div className="min-w-0 flex-1">
        <p className="text-xs text-muted-foreground">{label}</p>
        <p className="text-sm font-medium text-foreground break-all">{value}</p>
      </div>
    </div>
  );

  return (
    <div className="min-h-screen bg-background">
      <TopBar />
      <Header />
      <div className="container mx-auto px-3 sm:px-4 py-8 sm:py-16">
        <div className="max-w-lg mx-auto">

          {/* Loading State */}
          {status === "loading" && (
            <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} className="text-center py-16">
              <div className="relative mx-auto w-20 h-20 mb-6">
                <div className="absolute inset-0 rounded-full bg-primary/10 animate-ping" />
                <div className="relative flex items-center justify-center w-20 h-20 rounded-full bg-primary/10">
                  <Loader2 className="h-10 w-10 text-primary animate-spin" />
                </div>
              </div>
              <h1 className="font-heading text-xl sm:text-2xl font-bold mb-2 text-foreground">পেমেন্ট ভেরিফাই হচ্ছে...</h1>
              <p className="text-muted-foreground text-sm">অনুগ্রহ করে অপেক্ষা করুন।</p>
            </motion.div>
          )}

          {/* Success State */}
          {status === "success" && (
            <motion.div initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }} transition={{ duration: 0.5 }} className="text-center">
              {/* Success Icon */}
              <motion.div initial={{ scale: 0 }} animate={{ scale: 1 }} transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.1 }} className="mx-auto mb-5">
                <div className="relative mx-auto w-24 h-24">
                  <div className="absolute inset-0 rounded-full bg-primary/20 animate-pulse" />
                  <div className="absolute inset-2 rounded-full bg-primary/10" />
                  <div className="relative flex items-center justify-center w-24 h-24">
                    <CheckCircle2 className="h-14 w-14 text-primary" strokeWidth={1.5} />
                  </div>
                </div>
              </motion.div>

              <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.3 }}>
                <h1 className="font-heading text-2xl sm:text-3xl font-bold mb-1 text-foreground">পেমেন্ট সফল! 🎉</h1>
                <p className="text-muted-foreground text-sm mb-6">{message}</p>
              </motion.div>

              {orders.length > 0 && (
                <>
                  {/* Order Number & Status */}
                  <SectionCard delay={0.35}>
                    <SectionHeader icon={Receipt} title="অর্ডার ইনফরমেশন" />
                    <div className="p-5 space-y-1">
                      {orders.map((order) => (
                        <div key={order.id} className="space-y-2">
                          <InfoRow icon={Hash} label="অর্ডার নম্বর" value={
                            <span className="font-mono text-xs">{order.id}</span>
                          } />
                          <InfoRow icon={Clock} label="অর্ডার স্ট্যাটাস" value={
                            <span className={`inline-block px-2.5 py-0.5 rounded-full text-xs font-semibold ${getStatusBadge(order.status).cls}`}>
                              {getStatusBadge(order.status).label}
                            </span>
                          } />
                          <InfoRow icon={CreditCard} label="পেমেন্ট স্ট্যাটাস" value={
                            <span className={`inline-block px-2.5 py-0.5 rounded-full text-xs font-semibold ${getStatusBadge(order.payment_status || "unpaid").cls}`}>
                              {getStatusBadge(order.payment_status || "unpaid").label}
                            </span>
                          } />
                          {order.transaction_id && (
                            <InfoRow icon={Hash} label="ট্রানজ্যাকশন আইডি" value={
                              <span className="font-mono text-xs">{order.transaction_id}</span>
                            } />
                          )}
                          <InfoRow icon={CreditCard} label="পেমেন্ট মেথড" value={getPaymentMethodLabel(order.payment_method)} />
                          <InfoRow icon={CalendarDays} label="অর্ডারের তারিখ" value={new Date(order.created_at).toLocaleString("bn-BD", {
                            year: "numeric", month: "long", day: "numeric",
                            hour: "2-digit", minute: "2-digit", second: "2-digit"
                          })} />
                        </div>
                      ))}
                    </div>
                  </SectionCard>

                  {/* Product Details */}
                  <SectionCard delay={0.4}>
                    <SectionHeader icon={ShoppingBag} title="প্রোডাক্ট ডিটেইলস" />
                    <div className="p-5 space-y-3">
                      {orders.map((order, idx) => (
                        <div key={order.id} className={`flex justify-between items-start gap-3 ${idx > 0 ? "border-t border-border pt-3" : ""}`}>
                          <div className="min-w-0 flex-1">
                            <p className="font-medium text-sm text-foreground">{order.product_name}</p>
                            <p className="text-xs text-muted-foreground mt-0.5">× ১</p>
                          </div>
                          <span className="text-primary font-bold text-sm whitespace-nowrap">৳{order.amount.replace(/[^\d.,]/g, "")}</span>
                        </div>
                      ))}
                      {orders.length > 0 && (
                        <div className="border-t border-border pt-3 flex justify-between items-center">
                          <span className="text-sm font-semibold text-foreground">মোট</span>
                          <span className="text-primary font-bold text-base">৳{totalAmount.toLocaleString("bn-BD")}</span>
                        </div>
                      )}
                    </div>
                  </SectionCard>

                  {/* Customer / Contact Details */}
                  <SectionCard delay={0.45}>
                    <SectionHeader icon={User} title="কাস্টমার তথ্য" />
                    <div className="p-5 space-y-1">
                      <InfoRow icon={User} label="নাম" value={orders[0]?.customer_name} />
                      {orders[0]?.customer_phone && (
                        <InfoRow icon={Phone} label="ফোন নম্বর" value={orders[0].customer_phone} />
                      )}
                      {userEmail && (
                        <InfoRow icon={Mail} label="ইমেইল" value={userEmail} />
                      )}
                    </div>
                  </SectionCard>

                  {/* Delivered Keys */}
                  {deliveredKeys.length > 0 && (
                    <SectionCard delay={0.5}>
                      <div className="bg-primary/5 px-5 py-3 border-b border-border flex items-center gap-2">
                        <KeyRound className="h-4 w-4 text-primary" />
                        <h2 className="font-heading font-semibold text-sm text-foreground">ডেলিভারি হওয়া লাইসেন্স কী</h2>
                      </div>
                      <div className="p-5 space-y-3">
                        {deliveredKeys.map((dk, idx) => (
                          <div key={idx} className="bg-muted/40 rounded-xl p-3.5 space-y-1.5">
                            <p className="text-xs font-medium text-muted-foreground">{dk.product_name}</p>
                            <div className="flex items-center gap-2">
                              <code className="text-sm font-mono bg-background px-3 py-1.5 rounded-lg border border-border flex-1 break-all text-foreground">
                                {dk.key_value}
                              </code>
                              <Button
                                size="icon"
                                variant="ghost"
                                className="h-9 w-9 shrink-0 rounded-lg hover:bg-primary/10"
                                onClick={() => copyToClipboard(dk.key_value)}
                              >
                                {copiedKey === dk.key_value ? (
                                  <Check className="h-4 w-4 text-primary" />
                                ) : (
                                  <Copy className="h-4 w-4 text-muted-foreground" />
                                )}
                              </Button>
                            </div>
                          </div>
                        ))}
                        <p className="text-xs text-muted-foreground flex items-center gap-1.5">
                          💡 প্রোডাক্ট কী প্রোফাইল পেজেও দেখতে পারবেন
                        </p>
                      </div>
                    </SectionCard>
                  )}

                  {/* Support Info */}
                  <SectionCard delay={0.55}>
                    <SectionHeader icon={HelpCircle} title="সাহায্য প্রয়োজন?" />
                    <div className="p-5 space-y-3">
                      <p className="text-sm text-muted-foreground">
                        অর্ডার সংক্রান্ত কোনো সমস্যা হলে আমাদের সাথে যোগাযোগ করুন।
                      </p>
                      <Button
                        variant="outline"
                        size="sm"
                        className="gap-2 rounded-xl"
                        onClick={() => window.open("https://wa.me/8801700000000", "_blank")}
                      >
                        <MessageCircle className="h-4 w-4" />
                        WhatsApp এ মেসেজ করুন
                      </Button>
                    </div>
                  </SectionCard>
                </>
              )}

              {/* Action Buttons */}
              <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.6 }} className="flex flex-col gap-3 mt-6">
                <Button onClick={() => navigate("/orders")} variant="outline" className="w-full h-12 rounded-xl text-sm font-medium gap-2">
                  <Package className="h-4 w-4" /> আমার অর্ডারগুলো দেখুন
                </Button>
                <Button onClick={() => navigate("/profile")} variant="outline" className="w-full h-12 rounded-xl text-sm font-medium">
                  প্রোফাইলে যান
                </Button>
                <Button onClick={() => navigate("/")} className="w-full h-12 rounded-xl text-sm font-semibold gap-2">
                  <ArrowLeft className="h-4 w-4" /> হোম পেজে যান
                </Button>
              </motion.div>
            </motion.div>
          )}

          {/* Failed State */}
          {status === "failed" && (
            <motion.div initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }} transition={{ duration: 0.5 }} className="text-center">
              <motion.div initial={{ scale: 0 }} animate={{ scale: 1 }} transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.1 }} className="mx-auto mb-5">
                <div className="relative mx-auto w-24 h-24">
                  <div className="absolute inset-0 rounded-full bg-destructive/15 animate-pulse" />
                  <div className="absolute inset-2 rounded-full bg-destructive/5" />
                  <div className="relative flex items-center justify-center w-24 h-24">
                    <XCircle className="h-14 w-14 text-destructive" strokeWidth={1.5} />
                  </div>
                </div>
              </motion.div>
              <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.3 }}>
                <h1 className="font-heading text-2xl sm:text-3xl font-bold mb-1 text-foreground">পেমেন্ট ব্যর্থ</h1>
                <p className="text-muted-foreground text-sm mb-8">{message}</p>
              </motion.div>
              <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.4 }} className="flex flex-col gap-3">
                <Button onClick={() => navigate("/checkout")} variant="outline" className="w-full h-12 rounded-xl text-sm font-medium">
                  আবার চেষ্টা করুন
                </Button>
                <Button onClick={() => navigate("/")} className="w-full h-12 rounded-xl text-sm font-semibold gap-2">
                  <ArrowLeft className="h-4 w-4" /> হোম পেজে যান
                </Button>
              </motion.div>
            </motion.div>
          )}
        </div>
      </div>
      <ShopFooter />
      <BottomBar />
    </div>
  );
};

export default PaymentCallback;
