import { useState } from "react";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { toast } from "sonner";
import { useAuth } from "@/contexts/AuthContext";
import { supabase } from "@/integrations/supabase/client";

const ProductRequest = () => {
  const { user } = useAuth();
  const [name, setName] = useState("");
  const [details, setDetails] = useState("");
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!name.trim()) { toast.error("প্রোডাক্টের নাম লিখুন"); return; }
    setLoading(true);
    try {
      const { error } = await supabase.from("product_requests").insert({
        product_name: name.trim(),
        details: details.trim() || null,
        user_id: user?.id || null,
        user_email: user?.email || null,
        user_phone: user?.user_metadata?.phone || null,
      });
      if (error) throw error;
      toast.success("আপনার রিকোয়েস্ট সফলভাবে জমা হয়েছে! আমরা শীঘ্রই আপনার সাথে যোগাযোগ করবো।");
      setName("");
      setDetails("");
    } catch {
      toast.error("রিকোয়েস্ট জমা করতে সমস্যা হয়েছে। আবার চেষ্টা করুন।");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header />
      <div className="container mx-auto px-4 py-12 max-w-lg">
        <h1 className="text-3xl font-bold text-foreground text-center mb-2">প্রোডাক্ট রিকোয়েস্ট</h1>
        <p className="text-muted-foreground text-center mb-8">
          আপনার পছন্দের ডিজিটাল প্রোডাক্ট আমাদের শপে নেই? রিকোয়েস্ট করুন, আমরা ব্যবস্থা করবো!
        </p>
        <form onSubmit={handleSubmit} className="space-y-4 bg-card border border-border rounded-xl p-6">
          <div>
            <label className="text-sm font-medium text-foreground mb-1 block">প্রোডাক্টের নাম *</label>
            <Input value={name} onChange={(e) => setName(e.target.value)} placeholder="যেমন: Adobe Photoshop" />
          </div>
          <div>
            <label className="text-sm font-medium text-foreground mb-1 block">বিস্তারিত (ঐচ্ছিক)</label>
            <Textarea value={details} onChange={(e) => setDetails(e.target.value)} placeholder="কোন ভার্সন, কত দিনের জন্য, অন্যান্য তথ্য..." rows={4} />
          </div>
          <Button type="submit" className="w-full" disabled={loading}>
            {loading ? "জমা হচ্ছে..." : "রিকোয়েস্ট জমা দিন"}
          </Button>
        </form>
      </div>
      <ShopFooter />
    </div>
  );
};

export default ProductRequest;
