const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

interface EventData {
  event_name: string;
  event_time?: number;
  event_source_url?: string;
  user_data?: {
    client_ip_address?: string;
    client_user_agent?: string;
    fbp?: string;
    fbc?: string;
    em?: string;
    ph?: string;
    external_id?: string;
  };
  custom_data?: Record<string, unknown>;
  action_source?: string;
  event_id?: string;
}

Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    // Load pixel config from site_settings
    const { data: settings } = await supabase
      .from("site_settings")
      .select("key, value")
      .in("key", [
        "pixel_enabled",
        "pixel_id",
        "pixel_access_token",
        "pixel_test_event_code",
      ]);

    const cfg: Record<string, string> = {};
    settings?.forEach((r: { key: string; value: string | null }) => {
      if (r.value) cfg[r.key] = r.value;
    });

    if (cfg.pixel_enabled !== "true" || !cfg.pixel_id || !cfg.pixel_access_token) {
      return new Response(
        JSON.stringify({ success: false, error: "Pixel or CAPI not configured" }),
        { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const body = await req.json();
    const events: EventData[] = Array.isArray(body.data) ? body.data : [body.data];

    // Enrich events with server-side defaults
    const now = Math.floor(Date.now() / 1000);
    const clientIp = req.headers.get("x-forwarded-for")?.split(",")[0]?.trim() ||
      req.headers.get("cf-connecting-ip") || "";
    const clientUa = req.headers.get("user-agent") || "";

    const enrichedEvents = events.map((ev) => ({
      event_name: ev.event_name,
      event_time: ev.event_time || now,
      event_source_url: ev.event_source_url || "",
      action_source: ev.action_source || "website",
      event_id: ev.event_id || crypto.randomUUID(),
      user_data: {
        client_ip_address: clientIp,
        client_user_agent: clientUa,
        ...ev.user_data,
      },
      custom_data: ev.custom_data || {},
    }));

    // Build CAPI request
    const params = new URLSearchParams({
      access_token: cfg.pixel_access_token,
    });
    if (cfg.pixel_test_event_code) {
      params.set("test_event_code", cfg.pixel_test_event_code);
    }

    const capiUrl = `https://graph.facebook.com/v21.0/${cfg.pixel_id}/events?${params.toString()}`;

    const capiRes = await fetch(capiUrl, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ data: enrichedEvents }),
    });

    const capiResult = await capiRes.json();

    if (!capiRes.ok) {
      console.error("CAPI error:", JSON.stringify(capiResult));
      return new Response(
        JSON.stringify({ success: false, error: capiResult }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    return new Response(
      JSON.stringify({ success: true, result: capiResult }),
      { status: 200, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error: unknown) {
    console.error("CAPI edge function error:", error);
    const msg = error instanceof Error ? error.message : "Unknown error";
    return new Response(
      JSON.stringify({ success: false, error: msg }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
