const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

async function sendEmailViaSMTP(host: string, port: number, user: string, pass: string, to: string, subject: string, html: string, text: string, fromName: string = "TZSS") {
  let conn: Deno.TlsConn;

  if (port === 465) {
    // Implicit TLS (port 465)
    conn = await Deno.connectTls({ hostname: host, port });
  } else {
    // STARTTLS (port 587) - connect plain first, then upgrade
    const plainConn = await Deno.connect({ hostname: host, port });
    const plainEncoder = new TextEncoder();
    const plainDecoder = new TextDecoder();

    async function plainRead(): Promise<string> {
      const buf = new Uint8Array(4096);
      const n = await plainConn.read(buf);
      if (n === null) throw new Error("Connection closed");
      return plainDecoder.decode(buf.subarray(0, n));
    }
    async function plainSend(cmd: string): Promise<string> {
      await plainConn.write(plainEncoder.encode(cmd + "\r\n"));
      return await plainRead();
    }

    // Read greeting
    await plainRead();
    // EHLO
    await plainSend("EHLO localhost");
    // STARTTLS
    const starttlsRes = await plainSend("STARTTLS");
    if (!starttlsRes.startsWith("220")) {
      plainConn.close();
      throw new Error("STARTTLS failed: " + starttlsRes);
    }
    // Upgrade to TLS
    conn = await Deno.startTls(plainConn, { hostname: host });
  }

  const encoder = new TextEncoder();
  const decoder = new TextDecoder();

  async function readResponse(): Promise<string> {
    const buf = new Uint8Array(4096);
    const n = await conn.read(buf);
    if (n === null) throw new Error("Connection closed");
    return decoder.decode(buf.subarray(0, n));
  }

  async function sendCommand(cmd: string): Promise<string> {
    await conn.write(encoder.encode(cmd + "\r\n"));
    return await readResponse();
  }

  // Read greeting (only for implicit TLS on port 465)
  if (port === 465) {
    await readResponse();
  }

  // EHLO
  const ehloRes = await sendCommand(`EHLO localhost`);
  console.log("EHLO response:", ehloRes.substring(0, 100));

  // AUTH LOGIN
  await sendCommand("AUTH LOGIN");
  await sendCommand(btoa(user));
  const authRes = await sendCommand(btoa(pass));
  if (!authRes.startsWith("235")) {
    conn.close();
    throw new Error("SMTP Auth failed: " + authRes);
  }

  // MAIL FROM
  await sendCommand(`MAIL FROM:<${user}>`);
  // RCPT TO
  await sendCommand(`RCPT TO:<${to}>`);
  // DATA
  await sendCommand("DATA");

  // Build MIME message
  const boundary = "----=_Part_" + Date.now();
  const message = [
    `From: ${fromName} <${user}>`,
    `To: ${to}`,
    `Subject: ${subject}`,
    `MIME-Version: 1.0`,
    `Content-Type: multipart/alternative; boundary="${boundary}"`,
    ``,
    `--${boundary}`,
    `Content-Type: text/plain; charset=UTF-8`,
    ``,
    text || subject,
    ``,
    `--${boundary}`,
    `Content-Type: text/html; charset=UTF-8`,
    ``,
    html,
    ``,
    `--${boundary}--`,
    `.`,
  ].join("\r\n");

  await conn.write(encoder.encode(message + "\r\n"));
  await readResponse();

  // QUIT
  await sendCommand("QUIT");
  conn.close();
}

import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    // Try loading SMTP config from site_settings (admin panel)
    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    const { data: settings } = await supabase.from("site_settings").select("key, value").in("key", ["smtp_host", "smtp_port", "smtp_user", "smtp_password", "smtp_encryption", "site_title", "smtp_from_name"]);
    const cfg: Record<string, string> = {};
    settings?.forEach((r: any) => { if (r.value) cfg[r.key] = r.value; });
    const siteTitle = "TZSS";

    const SMTP_HOST = cfg.smtp_host || Deno.env.get("SMTP_HOST");
    const SMTP_USER = cfg.smtp_user || Deno.env.get("SMTP_USER");
    const SMTP_PASSWORD = cfg.smtp_password || Deno.env.get("SMTP_PASSWORD");
    const smtpPort = parseInt(cfg.smtp_port || "465", 10);
    const encryption = cfg.smtp_encryption || "ssl";

    if (!SMTP_HOST || !SMTP_USER || !SMTP_PASSWORD) {
      throw new Error("SMTP credentials not configured");
    }

    const { to, subject, html, text } = await req.json();

    if (!to || !subject || (!html && !text)) {
      return new Response(
        JSON.stringify({ success: false, error: "to, subject, and html/text are required" }),
        { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    // Use configured port, fallback to other port on failure
    const fallbackPort = smtpPort === 465 ? 587 : 465;
    try {
      await sendEmailViaSMTP(SMTP_HOST, smtpPort, SMTP_USER, SMTP_PASSWORD, to, subject, html || "", text || "", siteTitle);
    } catch (e) {
      console.log(`Port ${smtpPort} failed, trying ${fallbackPort}:`, e);
      await sendEmailViaSMTP(SMTP_HOST, fallbackPort, SMTP_USER, SMTP_PASSWORD, to, subject, html || "", text || "", siteTitle);
    }

    return new Response(
      JSON.stringify({ success: true }),
      { status: 200, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error: unknown) {
    console.error("Email send error:", error);
    const errorMessage = error instanceof Error ? error.message : "Unknown error";
    return new Response(
      JSON.stringify({ success: false, error: errorMessage }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
