import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

async function sendEmailViaSMTP(host: string, port: number, user: string, pass: string, to: string, subject: string, html: string, text: string, fromName: string = "TZSS") {
  let conn: Deno.TlsConn;

  if (port === 465) {
    conn = await Deno.connectTls({ hostname: host, port });
  } else {
    const plainConn = await Deno.connect({ hostname: host, port });
    const plainEncoder = new TextEncoder();
    const plainDecoder = new TextDecoder();

    async function plainRead(): Promise<string> {
      const buf = new Uint8Array(4096);
      const n = await plainConn.read(buf);
      if (n === null) throw new Error("Connection closed");
      return plainDecoder.decode(buf.subarray(0, n));
    }
    async function plainSend(cmd: string): Promise<string> {
      await plainConn.write(plainEncoder.encode(cmd + "\r\n"));
      return await plainRead();
    }

    await plainRead();
    await plainSend("EHLO localhost");
    const starttlsRes = await plainSend("STARTTLS");
    if (!starttlsRes.startsWith("220")) {
      plainConn.close();
      throw new Error("STARTTLS failed: " + starttlsRes);
    }
    conn = await Deno.startTls(plainConn, { hostname: host });
  }
  const encoder = new TextEncoder();
  const decoder = new TextDecoder();

  async function readResponse(): Promise<string> {
    const buf = new Uint8Array(4096);
    const n = await conn.read(buf);
    if (n === null) throw new Error("Connection closed");
    return decoder.decode(buf.subarray(0, n));
  }

  async function sendCommand(cmd: string): Promise<string> {
    await conn.write(encoder.encode(cmd + "\r\n"));
    return await readResponse();
  }

  await readResponse();
  await sendCommand(`EHLO localhost`);
  await sendCommand("AUTH LOGIN");
  await sendCommand(btoa(user));
  const authRes = await sendCommand(btoa(pass));
  if (!authRes.startsWith("235")) {
    conn.close();
    throw new Error("SMTP Auth failed: " + authRes);
  }

  await sendCommand(`MAIL FROM:<${user}>`);
  await sendCommand(`RCPT TO:<${to}>`);
  await sendCommand("DATA");

  const boundary = "----=_Part_" + Date.now();
  const message = [
    `From: ${fromName} <${user}>`,
    `To: ${to}`,
    `Subject: ${subject}`,
    `MIME-Version: 1.0`,
    `Content-Type: multipart/alternative; boundary="${boundary}"`,
    ``,
    `--${boundary}`,
    `Content-Type: text/plain; charset=UTF-8`,
    ``,
    text || subject,
    ``,
    `--${boundary}`,
    `Content-Type: text/html; charset=UTF-8`,
    ``,
    html,
    ``,
    `--${boundary}--`,
    `.`,
  ].join("\r\n");

  await conn.write(encoder.encode(message + "\r\n"));
  await readResponse();
  await sendCommand("QUIT");
  conn.close();
}

Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { email, reset_url } = await req.json();

    if (!email) {
      return new Response(
        JSON.stringify({ success: false, error: "Email is required" }),
        { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    // Generate password reset link
    const { data, error } = await supabase.auth.admin.generateLink({
      type: "recovery",
      email,
      options: {
        redirectTo: reset_url || `${supabaseUrl}/reset-password`,
      },
    });

    if (error || !data?.properties?.action_link) {
      console.error("Generate link error:", error);
      return new Response(
        JSON.stringify({ success: false, error: "রিসেট লিংক তৈরি করতে সমস্যা হয়েছে" }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const resetLink = data.properties.action_link;

    // Load SMTP config + email templates from site_settings
    const { data: allSettings } = await supabase.from("site_settings").select("key, value").in("key", ["smtp_host", "smtp_port", "smtp_user", "smtp_password", "smtp_encryption", "email_templates", "site_title", "smtp_from_name"]);
    const cfg: Record<string, string> = {};
    allSettings?.forEach((r: any) => { if (r.value) cfg[r.key] = r.value; });
    const siteTitle = "TZSS";

    const SMTP_HOST = cfg.smtp_host || Deno.env.get("SMTP_HOST");
    const SMTP_USER = cfg.smtp_user || Deno.env.get("SMTP_USER");
    const SMTP_PASSWORD = cfg.smtp_password || Deno.env.get("SMTP_PASSWORD");
    const smtpPort = parseInt(cfg.smtp_port || "465", 10);

    if (!SMTP_HOST || !SMTP_USER || !SMTP_PASSWORD) {
      return new Response(
        JSON.stringify({ success: false, error: "Email not configured" }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    // Default email template
    let emailSubject = `🔐 পাসওয়ার্ড রিসেট - ${siteTitle}`;
    let emailHtml = `
      <div style="font-family: 'Segoe UI', sans-serif; max-width: 600px; margin: 0 auto; background: #ffffff; border-radius: 12px; overflow: hidden; border: 1px solid #e5e7eb;">
        <div style="background: linear-gradient(135deg, #6366f1, #8b5cf6); padding: 30px; text-align: center;">
          <h1 style="color: white; margin: 0; font-size: 24px;">${siteTitle}</h1>
          <p style="color: rgba(255,255,255,0.9); margin: 8px 0 0;">পাসওয়ার্ড রিসেট</p>
        </div>
        <div style="padding: 30px;">
          <p style="font-size: 16px;">আপনি পাসওয়ার্ড রিসেটের অনুরোধ করেছেন।</p>
          <p>নিচের বাটনে ক্লিক করে নতুন পাসওয়ার্ড সেট করুন:</p>
          <div style="text-align: center; margin: 30px 0;">
            <a href="{reset_link}" style="background: linear-gradient(135deg, #6366f1, #8b5cf6); color: white; padding: 14px 32px; border-radius: 8px; text-decoration: none; font-weight: bold; font-size: 16px; display: inline-block;">পাসওয়ার্ড রিসেট করুন</a>
          </div>
          <p style="color: #6b7280; font-size: 13px;">যদি আপনি এই অনুরোধ না করে থাকেন, এই ইমেইল উপেক্ষা করুন।</p>
          <p style="color: #6b7280; font-size: 13px;">এই লিংক ১ ঘণ্টার মধ্যে মেয়াদোত্তীর্ণ হবে।</p>
          <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 24px 0;">
          <p style="color: #9ca3af; font-size: 12px; text-align: center;">${siteTitle}</p>
        </div>
      </div>`;

    // Try to load custom template
    try {
      if (cfg.email_templates) {
        const templates = JSON.parse(cfg.email_templates);
        const tpl = templates.find((t: any) => t.key === "password_reset");
        if (tpl?.enabled !== false) {
          if (tpl?.subject) emailSubject = tpl.subject;
          if (tpl?.body) emailHtml = tpl.body;
        }
      }
    } catch {}

    // Replace tags
    emailSubject = emailSubject.replace(/\{reset_link\}/g, resetLink);
    emailHtml = emailHtml.replace(/\{reset_link\}/g, resetLink);

    const fallbackPort = smtpPort === 465 ? 587 : 465;
    try {
      await sendEmailViaSMTP(
        SMTP_HOST, smtpPort, SMTP_USER, SMTP_PASSWORD, email,
        emailSubject, emailHtml,
        `পাসওয়ার্ড রিসেট করতে এই লিংকে যান: ${resetLink}`, siteTitle
      );
    } catch (e) {
      console.log(`Port ${smtpPort} failed, trying ${fallbackPort}:`, e);
      await sendEmailViaSMTP(
        SMTP_HOST, fallbackPort, SMTP_USER, SMTP_PASSWORD, email,
        emailSubject, emailHtml,
        `পাসওয়ার্ড রিসেট করতে এই লিংকে যান: ${resetLink}`, siteTitle
      );
    }

    return new Response(
      JSON.stringify({ success: true }),
      { status: 200, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error: unknown) {
    console.error("Password reset email error:", error);
    const errorMessage = error instanceof Error ? error.message : "Unknown error";
    return new Response(
      JSON.stringify({ success: false, error: errorMessage }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
