import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { transaction_id, order_ids, force_paid } = await req.json();

    if (!transaction_id && !order_ids) {
      return new Response(JSON.stringify({ error: "Missing transaction_id or order_ids" }), {
        status: 400,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    let apiKey = Deno.env.get("TZSMMPAY_API_KEY");
    const { data: apiKeySetting } = await supabase
      .from("site_settings")
      .select("value")
      .eq("key", "tzsmmpay_api_key")
      .maybeSingle();
    if (apiKeySetting?.value) apiKey = apiKeySetting.value;

    if (!apiKey) {
      return new Response(JSON.stringify({ error: "Payment API key not configured" }), {
        status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    let isPaid = false;
    let paidOrders: any[] = [];

    if (force_paid && order_ids && order_ids.length > 0) {
      isPaid = true;
      for (const id of order_ids) {
        await supabase.from("orders").update({ payment_status: "paid", status: "processing" }).eq("id", id);
      }
      const { data } = await supabase
        .from("orders")
        .select("customer_name, customer_phone, product_name, amount, user_id, transaction_id")
        .in("id", order_ids);
      paidOrders = data || [];
    } else if (transaction_id) {
      let gatewayHost = "pay.tzssbd.com";
      const { data: gw } = await supabase.from("site_settings").select("value").eq("key", "payment_gateway_url").maybeSingle();
      if (gw?.value) gatewayHost = gw.value;

      const verifyUrl = new URL(`https://${gatewayHost}/api/payment/verify`);
      verifyUrl.searchParams.set("api_key", apiKey);
      verifyUrl.searchParams.set("transaction_id", transaction_id);

      const verifyRes = await fetch(verifyUrl.toString());
      const verifyData = await verifyRes.json();
      isPaid = verifyData.status === "completed" || verifyData.status === "Completed" || verifyData.status === "COMPLETED";

      if (isPaid) {
        await supabase.from("orders").update({ payment_status: "paid", status: "processing" }).eq("transaction_id", transaction_id);
        const { data } = await supabase
          .from("orders")
          .select("customer_name, customer_phone, product_name, amount, user_id, transaction_id")
          .eq("transaction_id", transaction_id);
        paidOrders = data || [];
      }
    }

    if (isPaid && paidOrders.length > 0) {
      const deliveredKeys: { product_name: string; key_value: string }[] = [];

      for (const order of paidOrders) {
        const nameMatch = order.product_name.match(/^(.+?)(?:\s*\(([^)]+)\))?\s*(?:x\d+)?$/i);
        const cleanName = nameMatch ? nameMatch[1].trim() : order.product_name.replace(/\s*x\d+$/i, '').trim();
        const durationOption = nameMatch ? (nameMatch[2] || null) : null;

        const { data: product } = await supabase.from("products").select("id").eq("name", cleanName).maybeSingle();

        if (product) {
          let keyQuery = supabase.from("product_keys").select("id, key_value").eq("product_id", product.id).eq("is_sold", false);
          if (durationOption) keyQuery = keyQuery.eq("duration_option", durationOption);
          const { data: availableKey } = await keyQuery.limit(1).maybeSingle();

          if (availableKey) {
            const matchOrderId = order_ids?.length > 0 ? order_ids[paidOrders.indexOf(order)] : null;
            await supabase.from("product_keys").update({
              is_sold: true, order_id: matchOrderId || null, sold_at: new Date().toISOString(),
            }).eq("id", availableKey.id);
            deliveredKeys.push({ product_name: order.product_name, key_value: availableKey.key_value });
          }
        }
      }

      // ══════════════════════════════════════════════
      // Email Notifications
      // ══════════════════════════════════════════════
      const firstOrder = paidOrders[0];
      
      let customerEmail = "";
      if (firstOrder.user_id) {
        const { data: profile } = await supabase.from("profiles").select("email").eq("user_id", firstOrder.user_id).maybeSingle();
        if (profile?.email) customerEmail = profile.email;
      }

      const { data: allSettings } = await supabase
        .from("site_settings").select("key, value")
        .in("key", ["email_templates", "site_title", "logo_url", "email", "phone", "address", "shop_url", "smtp_from_name"]);
      const settingsMap: Record<string, string> = {};
      allSettings?.forEach((r: any) => { if (r.value) settingsMap[r.key] = r.value; });

      const shopName = "TZSS";
      const shopEmail = settingsMap.email || "info@tzssbd.com";
      const shopLogo = settingsMap.logo_url || "";
      const shopPhone = settingsMap.phone || "";
      const shopAddress = settingsMap.address || "";
      const shopUrl = settingsMap.shop_url || "https://tzssbd.com";

      let emailTemplates: any[] = [];
      try { if (settingsMap.email_templates) emailTemplates = JSON.parse(settingsMap.email_templates); } catch {}

      const allProducts = paidOrders.map(o => o.product_name).join(", ");
      const totalAmount = paidOrders.reduce((sum, o) => {
        const m = o.amount.match(/[\d,]+\.?\d*/);
        return sum + (m ? parseFloat(m[0].replace(/,/g, "")) : 0);
      }, 0);

      const productRows = paidOrders.map(o => {
        const m = o.amount.match(/[\d,]+\.?\d*/);
        const amt = m ? m[0] : o.amount;
        return `<tr><td style="padding:14px 16px;border-bottom:1px solid #f1f5f9;color:#1e293b;font-size:14px">${o.product_name}</td><td style="padding:14px 16px;border-bottom:1px solid #f1f5f9;text-align:right;color:#0f172a;font-weight:600;font-size:14px">&#2547;${amt}</td></tr>`;
      }).join('');

      const orderDate = new Date().toLocaleDateString('en-US', { timeZone: 'Asia/Dhaka', year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit', hour12: true });
      const orderNumber = (firstOrder.transaction_id || "N/A").slice(0, 8).toUpperCase();
      const statusText = deliveredKeys.length > 0 ? "Completed" : "Processing";
      const statusBg = deliveredKeys.length > 0 ? "#dcfce7" : "#fef3c7";
      const statusColor = deliveredKeys.length > 0 ? "#166534" : "#92400e";

      const replaceTags = (str: string) => {
        return str
          .replace(/\{name\}/g, firstOrder.customer_name || "")
          .replace(/\{product\}/g, allProducts)
          .replace(/\{total\}/g, String(totalAmount))
          .replace(/\{phone\}/g, firstOrder.customer_phone || "")
          .replace(/\{email\}/g, customerEmail)
          .replace(/\{billing_email\}/g, customerEmail)
          .replace(/\{billing_phone\}/g, firstOrder.customer_phone || "")
          .replace(/\{order_id\}/g, orderNumber)
          .replace(/\{transaction_id\}/g, firstOrder.transaction_id || "")
          .replace(/\{shop_name\}/g, shopName)
          .replace(/\{shop_email\}/g, shopEmail)
          .replace(/\{support_email\}/g, shopEmail)
          .replace(/\{shop_url\}/g, shopUrl)
          .replace(/\{shop_phone\}/g, shopPhone)
          .replace(/\{admin_url\}/g, `${shopUrl}/admin`)
          .replace(/\{profile_url\}/g, `${shopUrl}/orders`)
          .replace(/\{order_status\}/g, statusText.toLowerCase());
      };

      // ═══ Email Builder (table-based, mobile responsive) ═══
      const buildWooEmail = (headerBg: string, headerTitle: string, headerSub: string, contentHtml: string) => {
        const logoHtml = shopLogo 
          ? `<img src="${shopLogo}" alt="${shopName}" style="height:36px;max-width:160px;margin-bottom:10px" />`
          : `<p style="font-size:17px;font-weight:700;color:#ffffff;letter-spacing:0.5px;margin:0 0 10px">${shopName}</p>`;
        
        return `<!DOCTYPE html><html lang="bn"><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1.0"><meta http-equiv="X-UA-Compatible" content="IE=edge"><style type="text/css">@media only screen and (max-width:620px){.email-wrap{width:100%!important;padding:12px 8px!important}.hdr-pad{padding:28px 20px 24px!important}.content-pad{padding:20px 16px!important}.info-pad{padding:0 16px!important}.tbl-pad{padding:0 16px!important}.billing-pad{padding:0 16px!important;margin:16px 0!important}.cta-pad{padding:16px!important}.ftr-pad{padding:20px 16px!important}.ftr-main{padding:20px 16px!important}}</style></head><body style="margin:0;padding:0;background-color:#f1f5f9;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,Helvetica,Arial,sans-serif;-webkit-font-smoothing:antialiased"><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" style="background-color:#f1f5f9"><tr><td align="center" style="padding:24px 12px"><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="600" class="email-wrap" style="max-width:600px;width:100%"><tr><td class="hdr-pad" style="background:${headerBg};padding:32px 36px 28px;text-align:center;border-radius:14px 14px 0 0">${logoHtml}<h1 style="color:#ffffff;margin:10px 0 6px;font-size:20px;font-weight:700;line-height:1.3">${headerTitle}</h1><p style="color:rgba(255,255,255,0.78);margin:0;font-size:13px;font-weight:400;line-height:1.5">${headerSub}</p></td></tr><tr><td style="background-color:#ffffff">${contentHtml}</td></tr><tr><td style="background-color:#f8fafc;padding:12px 28px;text-align:center;border-left:1px solid #e2e8f0;border-right:1px solid #e2e8f0"><p style="color:#94a3b8;font-size:11px;margin:0;font-style:italic">This is an auto-generated email. Please do not reply.</p></td></tr><tr><td class="ftr-main" style="background-color:#0f172a;padding:24px 32px;text-align:center;border-radius:0 0 14px 14px"><p style="color:#f1f5f9;font-size:14px;font-weight:700;margin:0 0 4px">${shopName}</p>${shopAddress ? `<p style="color:#64748b;font-size:11px;margin:0 0 12px">${shopAddress}</p>` : ''}<table role="presentation" cellpadding="0" cellspacing="0" border="0" align="center" style="margin:12px auto 8px"><tr><td style="padding:0 4px"><a href="${shopUrl}" style="display:inline-block;background-color:#1e293b;color:#a78bfa;font-size:11px;text-decoration:none;padding:5px 12px;border-radius:6px;font-weight:500">Website</a></td><td style="padding:0 4px"><a href="mailto:${shopEmail}" style="display:inline-block;background-color:#1e293b;color:#a78bfa;font-size:11px;text-decoration:none;padding:5px 12px;border-radius:6px;font-weight:500">Support</a></td>${shopPhone ? `<td style="padding:0 4px"><a href="tel:${shopPhone}" style="display:inline-block;background-color:#1e293b;color:#a78bfa;font-size:11px;text-decoration:none;padding:5px 12px;border-radius:6px;font-weight:500">Call Us</a></td>` : ''}</tr></table><p style="color:#475569;font-size:10px;margin:8px 0 0">&copy; ${new Date().getFullYear()} ${shopName}. All rights reserved.</p></td></tr></table></td></tr></table></body></html>`;
      };

      // ═══ Order Info Summary ═══
      const orderInfoHtml = `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="info-pad" style="padding:0 28px;margin-bottom:16px"><tr><td><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" style="background-color:#f8fafc;border-radius:8px;border:1px solid #e2e8f0;font-size:13px"><tr><td style="padding:11px 14px;color:#64748b;font-weight:500;border-bottom:1px solid #e2e8f0;width:38%">Order Number</td><td style="padding:11px 14px;color:#0f172a;font-weight:700;border-bottom:1px solid #e2e8f0;font-family:monospace;font-size:14px">#${orderNumber}</td></tr><tr><td style="padding:11px 14px;color:#64748b;font-weight:500;border-bottom:1px solid #e2e8f0">Date</td><td style="padding:11px 14px;color:#0f172a;font-weight:500;border-bottom:1px solid #e2e8f0">${orderDate}</td></tr><tr><td style="padding:11px 14px;color:#64748b;font-weight:500;border-bottom:1px solid #e2e8f0">Transaction ID</td><td style="padding:11px 14px;color:#7c3aed;font-weight:500;border-bottom:1px solid #e2e8f0;font-family:monospace;font-size:11px;word-break:break-all">${firstOrder.transaction_id || 'N/A'}</td></tr><tr><td style="padding:11px 14px;color:#64748b;font-weight:500;border-bottom:1px solid #e2e8f0">Customer</td><td style="padding:11px 14px;color:#0f172a;font-weight:600;border-bottom:1px solid #e2e8f0">${firstOrder.customer_name}</td></tr><tr><td style="padding:11px 14px;color:#64748b;font-weight:500;border-bottom:1px solid #e2e8f0">Phone</td><td style="padding:11px 14px;border-bottom:1px solid #e2e8f0"><a href="tel:${firstOrder.customer_phone || ''}" style="color:#7c3aed;text-decoration:none">${firstOrder.customer_phone || 'N/A'}</a></td></tr><tr><td style="padding:11px 14px;color:#64748b;font-weight:500;border-bottom:1px solid #e2e8f0">Email</td><td style="padding:11px 14px;color:#0f172a;border-bottom:1px solid #e2e8f0;word-break:break-all;font-size:12px">${customerEmail || 'N/A'}</td></tr><tr><td style="padding:11px 14px;color:#64748b;font-weight:500">Status</td><td style="padding:11px 14px"><span style="display:inline-block;background-color:${statusBg};color:${statusColor};padding:3px 10px;border-radius:12px;font-size:11px;font-weight:600">${statusText}</span></td></tr></table></td></tr></table>`;

      // ═══ Order Table ═══
      const orderTableHtml = `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="tbl-pad" style="padding:0 28px"><tr><td><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" style="font-size:14px"><tr><td style="text-align:left;padding:10px 14px;background-color:#f8fafc;border-bottom:2px solid #e2e8f0;color:#64748b;font-size:11px;text-transform:uppercase;letter-spacing:0.6px;font-weight:600">Product</td><td style="text-align:right;padding:10px 14px;background-color:#f8fafc;border-bottom:2px solid #e2e8f0;color:#64748b;font-size:11px;text-transform:uppercase;letter-spacing:0.6px;font-weight:600">Price</td></tr>${productRows}<tr><td style="padding:8px 14px;color:#94a3b8;font-size:11px;border-top:1px solid #f1f5f9">Payment method</td><td style="padding:8px 14px;text-align:right;color:#475569;font-size:11px;border-top:1px solid #f1f5f9">Bangladesh Payment</td></tr><tr><td style="padding:12px 14px;font-weight:700;color:#0f172a;font-size:14px;border-top:2px solid #e2e8f0;background-color:#f8fafc">Total</td><td style="padding:12px 14px;text-align:right;font-weight:800;color:#7c3aed;font-size:18px;border-top:2px solid #e2e8f0;background-color:#f8fafc">&#2547;${totalAmount}</td></tr></table></td></tr></table>`;

      // ═══ Billing Card ═══
      const billingHtml = `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="billing-pad" style="padding:0 28px;margin:20px 0"><tr><td><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" style="background-color:#f8fafc;border-radius:8px;border:1px solid #e2e8f0"><tr><td style="padding:16px 20px"><p style="margin:0 0 10px;font-size:11px;color:#64748b;font-weight:600;text-transform:uppercase;letter-spacing:0.6px">Billing Address</p><p style="margin:0 0 4px;font-size:14px;color:#0f172a;font-weight:600">${firstOrder.customer_name}</p>${firstOrder.customer_phone ? `<p style="margin:0 0 3px;font-size:12px"><a href="tel:${firstOrder.customer_phone}" style="color:#7c3aed;text-decoration:none;font-weight:500">${firstOrder.customer_phone}</a></p>` : ''}${customerEmail ? `<p style="margin:0;font-size:12px;color:#475569">${customerEmail}</p>` : ''}</td></tr></table></td></tr></table>`;

      // ─── 1. Order Confirmation (to customer) ───
      if (customerEmail) {
        const confirmTpl = emailTemplates.find((t: any) => t.key === "order_confirmation");
        const shouldSend = !confirmTpl || confirmTpl.enabled !== false;

        if (shouldSend) {
          let subj = confirmTpl?.subject || `Order Confirmed #${orderNumber} - Payment Successful`;
          let html = "";

          if (confirmTpl?.body) {
            html = confirmTpl.body;
          } else {
            html = buildWooEmail(
              "linear-gradient(135deg,#7c3aed,#a78bfa)",
              "Thank you for your order!",
              `Order #${orderNumber} has been confirmed`,
              `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="content-pad" style="padding:24px 28px"><tr><td><p style="font-size:14px;color:#1e293b;margin:0 0 6px">Hi <strong style="color:#0f172a">${firstOrder.customer_name}</strong>,</p><p style="color:#475569;line-height:1.6;font-size:13px;margin:0">Your payment has been received successfully. Thank you for shopping with ${shopName}!</p></td></tr></table><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="content-pad" style="padding:0 28px 8px"><tr><td style="background-color:#f0fdf4;border-left:4px solid #22c55e;padding:10px 14px;border-radius:0 6px 6px 0"><p style="margin:0;color:#166534;font-size:12px;font-weight:500">Payment Verified - Your order is being processed</p></td></tr></table>${orderInfoHtml}${orderTableHtml}${billingHtml}<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="cta-pad" style="padding:18px 28px;text-align:center;border-top:1px solid #f1f5f9"><tr><td align="center"><a href="${shopUrl}/orders" style="display:inline-block;background-color:#7c3aed;color:#ffffff;padding:11px 28px;border-radius:6px;text-decoration:none;font-size:13px;font-weight:600">View My Orders</a></td></tr></table>`
            );
          }

          subj = replaceTags(subj);
          html = replaceTags(html);

          try {
            await fetch(`${supabaseUrl}/functions/v1/send-email`, {
              method: "POST",
              headers: { "Content-Type": "application/json", "Authorization": `Bearer ${supabaseKey}` },
              body: JSON.stringify({ to: customerEmail, subject: subj, html, text: `Hi ${firstOrder.customer_name}, Your order "${allProducts}" (Tk${totalAmount}) payment confirmed. Thank you! - ${shopName}` }),
            });
            console.log("Order confirmation email sent to", customerEmail);
          } catch (e) { console.error("Order confirmation email failed:", e); }
        }
      }

      // ─── 2. Delivery Email (to customer) ───
      if (customerEmail && deliveredKeys.length > 0) {
        const completedTpl = emailTemplates.find((t: any) => t.key === "completed");
        const shouldSend = !completedTpl || completedTpl.enabled !== false;

        if (shouldSend) {
          let subj = completedTpl?.subject || `Order #${orderNumber} Completed - Product Delivered!`;
          let html = "";

          if (completedTpl?.body) {
            html = completedTpl.body;
          } else {
            html = buildWooEmail(
              "linear-gradient(135deg,#059669,#34d399)",
              "Your order is complete!",
              "Product has been delivered successfully",
              `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="content-pad" style="padding:24px 28px"><tr><td><p style="font-size:14px;color:#1e293b;margin:0 0 6px">Hi <strong style="color:#0f172a">${firstOrder.customer_name}</strong>,</p><p style="color:#475569;line-height:1.6;font-size:13px;margin:0">Your order has been completed and product delivered. Your license key(s) are below.</p></td></tr></table>${orderInfoHtml}${orderTableHtml}<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="content-pad" style="padding:0 28px;margin:20px 0"><tr><td style="background-color:#faf5ff;border-radius:8px;padding:18px 20px;border:1px solid #e9d5ff"><p style="margin:0 0 12px;font-size:11px;color:#7c3aed;font-weight:600;text-transform:uppercase;letter-spacing:0.6px">Your License Key(s)</p>${deliveredKeys.map(dk => `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" style="margin-bottom:12px"><tr><td><p style="margin:0 0 4px;font-size:12px;color:#6b21a8;font-weight:600">${dk.product_name}</p><table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%"><tr><td style="background-color:#ffffff;border:1px solid #ddd6fe;border-radius:6px;padding:8px 12px"><code style="font-size:12px;color:#5b21b6;word-break:break-all;line-height:1.5;font-family:monospace">${dk.key_value}</code></td></tr></table></td></tr></table>`).join('')}</td></tr></table>${billingHtml}<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="cta-pad" style="padding:18px 28px;text-align:center;border-top:1px solid #f1f5f9"><tr><td align="center"><a href="${shopUrl}/orders" style="display:inline-block;background-color:#059669;color:#ffffff;padding:11px 28px;border-radius:6px;text-decoration:none;font-size:13px;font-weight:600">View My Orders</a></td></tr></table>`
            );
          }

          subj = replaceTags(subj);
          html = replaceTags(html);

          try {
            await fetch(`${supabaseUrl}/functions/v1/send-email`, {
              method: "POST",
              headers: { "Content-Type": "application/json", "Authorization": `Bearer ${supabaseKey}` },
              body: JSON.stringify({ to: customerEmail, subject: subj, html, text: `Hi ${firstOrder.customer_name}, Your product has been delivered. Check your profile. - ${shopName}` }),
            });
            console.log("Delivery email sent to", customerEmail);
          } catch (e) { console.error("Delivery email failed:", e); }
        }
      }

      // ─── 3. Admin Notification ───
      let adminEmail = "";
      try {
        const { data: adminRoles } = await supabase.from("user_roles").select("user_id").eq("role", "admin").limit(1);
        if (adminRoles && adminRoles.length > 0) {
          const { data: ap } = await supabase.from("profiles").select("email").eq("user_id", adminRoles[0].user_id).maybeSingle();
          if (ap?.email) adminEmail = ap.email;
        }
      } catch {}

      const newOrderTpl = emailTemplates.find((t: any) => t.key === "new_order");
      const shouldSendAdmin = !newOrderTpl || newOrderTpl.enabled !== false;
      const adminNotifyEmail = adminEmail || shopEmail;
      if (shouldSendAdmin && adminNotifyEmail) {
        let subj = newOrderTpl?.subject || `New Order #${orderNumber} - ${allProducts} (Tk${totalAmount})`;
        let html = "";

        if (newOrderTpl?.body) {
          html = newOrderTpl.body;
        } else {
          html = buildWooEmail(
            "linear-gradient(135deg,#7c3aed,#6366f1)",
            "New order received!",
            `${firstOrder.customer_name} placed a new order`,
            `<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="content-pad" style="padding:24px 28px"><tr><td style="background-color:#fef3c7;border-radius:8px;padding:14px 18px;border:1px solid #fcd34d"><p style="margin:0;color:#92400e;font-size:13px;font-weight:600">Congratulations on the sale!</p><p style="margin:4px 0 0;color:#a16207;font-size:12px">You have received a new order. Details below.</p></td></tr></table>${orderInfoHtml}${orderTableHtml}${billingHtml}<table role="presentation" cellpadding="0" cellspacing="0" border="0" width="100%" class="cta-pad" style="padding:18px 28px;text-align:center;border-top:1px solid #f1f5f9"><tr><td align="center"><a href="${shopUrl}/admin" style="display:inline-block;background-color:#7c3aed;color:#ffffff;padding:11px 28px;border-radius:6px;text-decoration:none;font-size:13px;font-weight:600">Manage Order</a></td></tr></table>`
          );
        }

        subj = replaceTags(subj);
        html = replaceTags(html);

        try {
          await fetch(`${supabaseUrl}/functions/v1/send-email`, {
            method: "POST",
            headers: { "Content-Type": "application/json", "Authorization": `Bearer ${supabaseKey}` },
            body: JSON.stringify({ to: adminNotifyEmail, subject: subj, html, text: `New Order: ${allProducts} (Tk${totalAmount}) - ${firstOrder.customer_name} (${firstOrder.customer_phone})` }),
          });
          console.log("Admin notification email sent to", adminNotifyEmail);
        } catch (e) { console.error("Admin email failed:", e); }
      }

      // Mark orders as completed
      if (deliveredKeys.length > 0 && order_ids && order_ids.length > 0) {
        for (const id of order_ids) await supabase.from("orders").update({ status: "completed" }).eq("id", id);
        console.log("Orders marked as completed:", order_ids);
      } else if (deliveredKeys.length > 0 && transaction_id) {
        await supabase.from("orders").update({ status: "completed" }).eq("transaction_id", transaction_id);
      }

      // SMS notification
      try {
        if (firstOrder.customer_phone) {
          const smsApiKey = Deno.env.get("SMS_API_KEY");
          if (smsApiKey) {
            const cleanNumber = firstOrder.customer_phone.replace(/[^0-9]/g, '');
            const smsMsg = `Dear ${firstOrder.customer_name}, Your order "${allProducts}" (Tk${totalAmount}) payment confirmed. Thank you! - ${shopName}`;
            await fetch(`https://hakvolution.com/CAVATER/api.php?key=${smsApiKey}&number=${cleanNumber}&msg=${encodeURIComponent(smsMsg)}`);
            console.log("SMS sent to", cleanNumber);
          }
        }
      } catch (e) { console.error("SMS failed:", e); }

      // WhatsApp notification
      try {
        if (firstOrder.customer_phone) {
          const { data: waSettings } = await supabase.from("site_settings").select("key, value").in("key", ["whatsapp_api_secret", "whatsapp_account_id", "whatsapp_templates"]);
          const waMap: Record<string, string> = {};
          waSettings?.forEach(r => { if (r.value) waMap[r.key] = r.value; });

          if (waMap.whatsapp_api_secret && waMap.whatsapp_account_id) {
            let waMsg = "";
            let shouldSendWa = true;

            if (waMap.whatsapp_templates) {
              try {
                const templates = JSON.parse(waMap.whatsapp_templates);
                const processingTpl = templates.find((t: any) => t.key === "processing");
                if (processingTpl) {
                  if (!processingTpl.enabled) { shouldSendWa = false; }
                  else if (processingTpl.message) {
                    waMsg = processingTpl.message;
                    let sn = shopName, se = shopEmail, su = "tzssbd.com", sp = shopPhone;
                    let ce = customerEmail;
                    const replacements: Record<string, string> = {
                      name: firstOrder.customer_name || "", items: allProducts, total: `&#2547;${totalAmount}`,
                      phone: firstOrder.customer_phone || "", email: ce, billing_email: ce,
                      billing_phone: firstOrder.customer_phone || "", order_id: orderNumber,
                      status: "processing", shop_name: sn, shop_email: se, shop_url: su, shop_phone: sp,
                      order_status: "processing", order_amount: `&#2547;${totalAmount}`, order_product: allProducts,
                      order_product_with_qty: allProducts, order_currency: "&#2547;",
                    };
                    for (const [key, val] of Object.entries(replacements)) {
                      waMsg = waMsg.replace(new RegExp(`\\{${key}\\}|\\[${key}\\]`, "g"), val);
                    }
                  }
                } else {
                  waMsg = `Dear ${firstOrder.customer_name}, Your order "${allProducts}" (Tk${totalAmount}) confirmed. Thank you! - ${shopName}`;
                }
              } catch {
                waMsg = `Dear ${firstOrder.customer_name}, Your order "${allProducts}" (Tk${totalAmount}) confirmed. Thank you! - ${shopName}`;
              }
            } else {
              waMsg = `Dear ${firstOrder.customer_name}, Your order "${allProducts}" (Tk${totalAmount}) confirmed. Thank you! - ${shopName}`;
            }

            if (shouldSendWa && waMsg) {
              let cleanNumber = firstOrder.customer_phone.replace(/[^0-9]/g, '');
              if (cleanNumber.startsWith('0')) cleanNumber = '88' + cleanNumber;
              else if (!cleanNumber.startsWith('88')) cleanNumber = '88' + cleanNumber;

              const formData = new FormData();
              formData.append('secret', waMap.whatsapp_api_secret);
              formData.append('account', waMap.whatsapp_account_id);
              formData.append('recipient', cleanNumber);
              formData.append('type', 'text');
              formData.append('message', waMsg);
              const waRes = await fetch('https://smsquicker.com/api/send/whatsapp', { method: 'POST', body: formData });
              console.log("WhatsApp sent to", cleanNumber, await waRes.text());
            }
          }
        }
      } catch (e) { console.error("WhatsApp failed:", e); }

      // Voice Call
      try {
        if (firstOrder.customer_phone) {
          const vcRes = await fetch(`${supabaseUrl}/functions/v1/send-voice-call`, {
            method: "POST",
            headers: { "Content-Type": "application/json", "Authorization": `Bearer ${supabaseKey}` },
            body: JSON.stringify({ phone_number: firstOrder.customer_phone, customer_name: firstOrder.customer_name }),
          });
          console.log("Voice call result:", JSON.stringify(await vcRes.json()));
        }
      } catch (e) { console.error("Voice call failed:", e); }

      // Telegram Notification
      try {
        const { data: tgSettings } = await supabase.from("site_settings").select("key, value").in("key", ["telegram_bot_token", "telegram_chat_id"]);
        const tgMap: Record<string, string> = {};
        tgSettings?.forEach((r: any) => { if (r.value) tgMap[r.key] = r.value; });
        const telegramBotToken = tgMap.telegram_bot_token || Deno.env.get("TELEGRAM_BOT_TOKEN");
        const telegramChatId = tgMap.telegram_chat_id || Deno.env.get("TELEGRAM_CHAT_ID");
        if (telegramBotToken && telegramChatId) {
          const keysInfo = deliveredKeys.length > 0
            ? `\n🔑 *Keys Delivered:* ${deliveredKeys.length}`
            : "\n⏳ *Keys:* Pending delivery";
          const tgMsg = `🛒 *New Order Received!*\n\n👤 *Customer:* ${firstOrder.customer_name}\n📞 *Phone:* ${firstOrder.customer_phone || "N/A"}\n📧 *Email:* ${customerEmail || "N/A"}\n\n📦 *Products:* ${allProducts}\n💰 *Amount:* ৳${totalAmount}\n🆔 *Order:* #${orderNumber}\n🔖 *Transaction:* ${firstOrder.transaction_id || "N/A"}${keysInfo}\n✅ *Payment:* Paid\n📊 *Status:* ${deliveredKeys.length > 0 ? "Completed" : "Processing"}`;

          const tgRes = await fetch(`https://api.telegram.org/bot${telegramBotToken}/sendMessage`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ chat_id: telegramChatId, text: tgMsg, parse_mode: "Markdown" }),
          });
          console.log("Telegram notification sent:", await tgRes.text());
        }
      } catch (e) { console.error("Telegram notification failed:", e); }
    }

    return new Response(JSON.stringify({ payment_status: isPaid ? "paid" : "unpaid" }), {
      headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  } catch (err) {
    return new Response(JSON.stringify({ error: "Internal error", details: String(err) }), {
      status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  }
});
